/* eslint-disable jest/no-disabled-tests */
import { nextTick } from 'vue';
import { GlButton } from '@gitlab/ui';
import AiGenie, { i18n } from 'ee/ai/components/ai_genie.vue';
import AiGenieChat from 'ee/ai/components/ai_genie_chat.vue';
import { shallowMountExtended } from 'helpers/vue_test_utils_helper';
import { setHTMLFixture, resetHTMLFixture } from 'helpers/fixtures';

describe('AiGenie', () => {
  let wrapper;
  const containerId = 'container';

  const createComponent = (propsData = {}) => {
    wrapper = shallowMountExtended(AiGenie, {
      propsData,
    });
  };
  const findButton = () => wrapper.findComponent(GlButton);
  const findGenieChat = () => wrapper.findComponent(AiGenieChat);
  const simulateSelection = () => {
    const selectionChangeEvent = new Event('selectionchange');
    document.dispatchEvent(selectionChangeEvent);
  };
  const getSelectionMock = () => {
    return {
      anchorNode: document.getElementById('first-paragraph'),
      isCollapsed: false,
      getRangeAt: jest.fn().mockImplementation(() => {
        return {
          getBoundingClientRect: jest.fn().mockImplementation(() => {
            return {
              top: 0,
              left: 0,
              right: 0,
              bottom: 0,
            };
          }),
        };
      }),
      rangeCount: 1,
    };
  };
  const waitForDebounce = async () => {
    await nextTick();
    jest.runOnlyPendingTimers();
  };

  beforeEach(() => {
    setHTMLFixture(
      `<div id="${containerId}" style="height:1000px; width: 800px"><p id="first-paragraph">Foo</p></div>`,
    );
    createComponent({
      containerId,
    });
  });

  afterEach(() => {
    resetHTMLFixture();
  });

  it('correctly renders the component by default', () => {
    expect(findButton().exists()).toBe(true);
    expect(findGenieChat().exists()).toBe(false);
  });

  it('correctly finds the container with the passed ID', () => {
    expect(wrapper.vm.container).toBeDefined();
    expect(wrapper.vm.container).toBe(document.getElementById(containerId));
  });

  describe('the toggle button', () => {
    it('is hidden by default, yet gets the correct props', () => {
      const btnWrapper = findButton();
      expect(btnWrapper.isVisible()).toBe(false);
      expect(btnWrapper.attributes('title')).toBe(i18n.GENIE_TOOLTIP);
    });
    it('is rendered when a text is selected', async () => {
      jest.spyOn(window, 'getSelection').mockImplementation(() => getSelectionMock());
      jest.spyOn(document.getElementById(containerId), 'contains').mockImplementation(() => true);
      simulateSelection();
      await waitForDebounce();
      expect(findButton().isVisible()).toBe(true);
    });
    it.skip('is positioned correctly at the top of the selection', () => {});
  });

  describe('selectionchange event listener', () => {
    let addEventListenerSpy;
    let removeEventListenerSpy;

    beforeEach(() => {
      addEventListenerSpy = jest.spyOn(document, 'addEventListener');
      removeEventListenerSpy = jest.spyOn(document, 'removeEventListener');
      createComponent({
        containerId,
      });
    });

    afterEach(() => {
      addEventListenerSpy.mockRestore();
      removeEventListenerSpy.mockRestore();
    });

    it('sets up the `selectionchange` event listener', () => {
      expect(addEventListenerSpy).toHaveBeenCalledWith(
        'selectionchange',
        wrapper.vm.debouncedSelectionChangeHandler,
      );
      expect(removeEventListenerSpy).not.toHaveBeenCalled();
    });

    it('removes the event listener when destroyed', () => {
      wrapper.destroy();
      expect(removeEventListenerSpy).toHaveBeenCalledWith(
        'selectionchange',
        wrapper.vm.debouncedSelectionChangeHandler,
      );
    });
  });

  describe('interaction', () => {
    beforeEach(() => {
      createComponent({
        containerId,
      });
    });

    it.skip('toggles genie when the button is clicked', () => {
      // Genie is visible and has `is-loading: true`
    });

    it.skip('once the response arrives, :content is set with the response message', () => {});
    it.skip('if the response fails, genie gets :error set with the error message', () => {});
    it.skip('when a snippet is selected, :selected-code gets the same content', () => {});
    it.skip('sets the snippet language', () => {});
  });
});
