# frozen_string_literal: true

module VulnerabilityFeedback
  class CreateService < ::BaseService
    include Gitlab::Utils::StrongMemoize

    def execute
      raise Gitlab::Access::AccessDeniedError unless can?(current_user, :create_vulnerability_feedback, vulnerability_feedback)

      if vulnerability_feedback.for_issue? && !vulnerability_feedback.vulnerability_data.blank?
        create_issue
      elsif vulnerability_feedback.for_merge_request? && !vulnerability_feedback.vulnerability_data.blank?
        create_merge_request
      else
        dismiss_existing_vulnerability
      end

      errors = vulnerability_feedback.errors.dup

      if vulnerability_feedback.persisted? && vulnerability_feedback.valid?
        success(vulnerability_feedback).merge(message: errors)
      else
        rollback_merge_request(vulnerability_feedback.merge_request) if vulnerability_feedback.merge_request

        error(vulnerability_feedback.errors)
      end

    rescue ArgumentError => e
      # VulnerabilityFeedback relies on #enum attributes which raise this exception
      error(e.message)
    end

    private

    attr_reader :params

    def vulnerability_feedback
      @vulnerability_feedback ||= @project.vulnerability_feedback.find_or_init_for(create_params)
    end

    def create_params
      @params[:migrated_to_state_transition] = true if params[:feedback_type] == 'dismissal'
      @params[:author] = @current_user
      @params.merge(comment_params).except(:dismiss_vulnerability)
    end

    def comment_params
      return {} unless @params[:comment].present?

      {
        comment_author: @current_user,
        comment_timestamp: Time.zone.now
      }
    end

    def success(vulnerability_feedback)
      super().merge(vulnerability_feedback: vulnerability_feedback)
    end

    def create_issue
      raise Gitlab::Access::AccessDeniedError unless can?(current_user, :read_security_resource, project)

      # Wrap Feedback and Issue creation in the same transaction
      ApplicationRecord.transaction do
        issue = @params[:issue]

        vulnerability = create_or_find_existing_vulnerability

        # Create a new issue if one does not exist
        unless issue
          result = Issues::CreateFromVulnerabilityService
            .new(@project, @current_user, vulnerability: vulnerability)
            .execute

          if result[:status] == :error
            vulnerability_feedback.errors.add(:issue, result[:message])
            raise ActiveRecord::Rollback
          end

          issue = result[:issue]
        end

        issue_link_result = create_vulnerability_issue_link(vulnerability, issue)

        if issue_link_result&.error?
          vulnerability_feedback.errors.add(:issue_link, issue_link_result.message)
          raise ActiveRecord::Rollback
        end

        vulnerability_feedback.issue = issue

        # Ensure created association is rolled back if feedback can't be saved
        raise ActiveRecord::Rollback unless vulnerability_feedback.save
      end
    end

    def create_merge_request
      vulnerability = create_or_find_existing_vulnerability

      result = MergeRequests::CreateFromVulnerabilityDataService
        .new(@project, @current_user, vulnerability_feedback.vulnerability_data)
        .execute

      if result[:status] == :success
        merge_request = result[:merge_request]

        merge_request_link_result = create_vulnerability_merge_request_link(vulnerability, merge_request)

        vulnerability_feedback.errors.add(:merge_request_link, merge_request_link_result.message)
        return if merge_request_link_result&.error?

        vulnerability_feedback.merge_request = merge_request

        vulnerability_feedback.save
      else
        vulnerability_feedback.errors.add(:merge_request, result[:message])
      end
    end

    # Gitaly RPCs cannot occur within a transaction so we must manually
    # rollback MR and branch creation
    def rollback_merge_request(merge_request)
      branch_name = merge_request.source_branch

      merge_request&.destroy &&
        ::Branches::DeleteService.new(project, current_user).execute(branch_name)
    end

    def create_vulnerability_issue_link(vulnerability, issue)
      VulnerabilityIssueLinks::CreateService
        .new(current_user, vulnerability, issue, link_type: Vulnerabilities::IssueLink.link_types[:created])
        .execute
    end

    def create_vulnerability_merge_request_link(vulnerability, merge_request)
      params = { vulnerability: vulnerability, merge_request: merge_request }
      VulnerabilityMergeRequestLinks::CreateService.new(project: @project,
                                                        current_user: @current_user,
                                                        params: params).execute
    end

    def dismiss_existing_vulnerability
      ApplicationRecord.transaction do
        if dismiss_vulnerability? && existing_vulnerability
          Vulnerabilities::DismissService.new(current_user,
                                              existing_vulnerability,
                                              params[:comment],
                                              dismiss_findings: false).execute
        end

        unless vulnerability_feedback.persisted?
          raise ActiveRecord::Rollback unless vulnerability_feedback.save
        end

        create_vulnerability('dismissed') unless existing_vulnerability
      end
    end

    def existing_vulnerability
      strong_memoize(:existing_vulnerability) { vulnerability_feedback.finding&.vulnerability }
    end

    def dismiss_vulnerability?
      params[:dismiss_vulnerability] != false && can?(current_user, :admin_vulnerability, project)
    end

    # This method is used to find or create vulnerability with respect to merge request
    # and issue feedback type when vulnerability_data needs to be present the other method
    # existing_vulnerability will go away with the deprecation of this service
    def create_or_find_existing_vulnerability
      find_existing_vulnerability || create_new_vulnerability
    end

    def find_existing_vulnerability
      # If we send a vulnerability_id, it should be valid. We can create a new Vulnerability if there's not one,
      # but if we expect there to be one and it's not, something went wrong.
      vulnerability_id = vulnerability_feedback.vulnerability_data[:vulnerability_id]

      return unless vulnerability_id

      project.vulnerabilities.find_by_id(vulnerability_id) || raise(Gitlab::Access::AccessDeniedError)
    end

    def create_new_vulnerability
      # If the vulnerability is being created here, that means that the vulnerability has not been
      # ingested to the main branch yet, so it remains detected or dismissed and not present_on_default_branch
      finding_state = if Vulnerabilities::Feedback.by_finding_uuid(params[:finding_uuid]).for_dismissal.present?
                        'dismissed'
                      else
                        'detected'
                      end

      vulnerability_response = create_vulnerability(finding_state)

      if vulnerability_response.error?
        vulnerability_feedback.errors.add(:vulnerability, vulnerability_response.message)

        return if vulnerability_feedback.for_merge_request?

        raise ActiveRecord::Rollback
      end

      vulnerability_response.payload[:vulnerability]
    end

    def create_vulnerability(state)
      Vulnerabilities::FindOrCreateFromSecurityFindingService.new(
        project: @project,
        current_user: @current_user,
        params: { security_finding_uuid: params[:finding_uuid],
                  comment: params[:comment],
                  dismissal_reason: params[:dismissal_reason] },
        state: state,
        present_on_default_branch: false).execute
    end
  end
end
