import { s__, __, sprintf } from '~/locale';
import Api from 'ee/api';

const MAX_RESPONSE_TOKENS = 100;
const MIN_RESPONSE_TOKENS = 50;
const TEMPERATURE = 0.3;
const DEFAULT_MODEL = 'gpt-3.5-turbo';
const MODEL_MAX_TOKENS = 4096; // max for 'gpt-3.5-turbo' model as per https://platform.openai.com/docs/models/gpt-3-5
const TOKENS_THRESHOLD = MODEL_MAX_TOKENS * 0.85; // We account for 15% fault in tokens calculation
export const TOO_LONG_ERROR_TYPE = 'too-long';
export const getExplainCodeInitPrompt = (filePath, text) => {
  return sprintf(
    s__(
      'AI|Explain the code from %{filePath} in human understandable language presented in Markdown format. In the response add neither original code snippet nor any title. `%{text}`',
    ),
    {
      filePath: filePath || __('random'),
      text,
    },
  );
};
export const computeTokens = (messages) => {
  const output = [];
  messages.flatMap((message) => [output.push(`${message.role}: ${message.content}`)]);

  // Below 4 is an approximate number of characters per token as per https://platform.openai.com/tokenizer
  return Math.ceil(output.join('\n').length / 4);
};
export const areMessagesWithinMaxLimits = (messages) => {
  return computeTokens(messages) + MAX_RESPONSE_TOKENS < TOKENS_THRESHOLD;
};
export const areMessagesWithinMinLimits = (messages) => {
  return computeTokens(messages) + MIN_RESPONSE_TOKENS < TOKENS_THRESHOLD;
};

export const askGenie = async (prompts) => {
  const messages = [
    {
      role: 'system',
      content: 'You are an assistant explaining to an engineer', // eslint-disable-line
    },
    ...prompts,
  ];

  let maxTokens = MAX_RESPONSE_TOKENS;
  if (!areMessagesWithinMaxLimits(messages)) {
    if (!areMessagesWithinMinLimits(messages)) {
      throw new Error(s__('AI|The text is too long. Please shorten it and try again.'), {
        cause: TOO_LONG_ERROR_TYPE,
      });
    } else {
      maxTokens = MIN_RESPONSE_TOKENS;
    }
  }

  const {
    data: { choices, usage },
  } = await Api.requestAIChat({
    model: DEFAULT_MODEL,
    max_tokens: maxTokens,
    temperature: TEMPERATURE,
    messages,
  }).catch((error) => {
    throw new Error(error.response.data.error.message);
  });

  return {
    text: choices[0].message.content,
    shouldTruncateNext: usage.total_tokens >= TOKENS_THRESHOLD,
  };
};
