# frozen_string_literal: true

module Gitlab
  module BackgroundMigration
    # The class to migrate the evidence data into their own records from the json attribute
    class MigrateEvidencesForVulnerabilityFindings < BatchedMigrationJob
      feature_category :vulnerability_management
      operation_name :migrate_evidences_for_vulnerability_findings

      # The class is mimicking Vulnerabilites::Finding
      class Finding < ApplicationRecord
        self.table_name = 'vulnerability_occurrences'

        validates :details, json_schema: { filename: 'vulnerability_finding_details', draft: 7 }, if: false
      end

      # The class is mimicking Vulnerabilites::Finding::Evidence
      class Evidence < ApplicationRecord
        self.table_name = 'vulnerability_finding_evidences'

        # This data has been already validated when parsed into vulnerability_occurrences.raw_metadata
        # Having this validation is a requerment from:
        # https://gitlab.com/gitlab-org/gitlab/-/blob/dc3262f850cbd0ac14171d3c389b1258b4749cda/spec/db/schema_spec.rb#L253-265
        validates :data, json_schema: { filename: "filename" }, if: false
      end

      def perform
        each_sub_batch do |sub_batch|
          migrate_evidences(sub_batch)
        end
      end

      private

      def migrate_evidences(sub_batch)
        attrs = sub_batch.filter_map do |finding|
          evidence = extract_evidence(finding.raw_metadata)

          next unless evidence

          build_evidence(finding, evidence)
        end.compact

        create_evidences(attrs) if attrs.present?
      end

      def build_evidence(finding, evidence)
        current_time = Time.current
        {
          vulnerability_occurrence_id: finding.id,
          data: evidence,
          created_at: current_time,
          updated_at: current_time
        }
      end

      def create_evidences(evidences)
        Evidence.upsert_all(evidences, returning: false, unique_by: %i[vulnerability_occurrence_id])
      end

      def extract_evidence(metadata)
        parsed_metadata = Gitlab::Json.parse(metadata)

        parsed_metadata['evidence']
      rescue JSON::ParserError
        nil
      end
    end
  end
end
