# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Issues::CreateFromVulnerabilityService, '#execute', feature_category: :vulnerability_management do
  let(:group)   { create(:group) }
  let(:project) { create(:project, :public, :repository, namespace: group) }
  let(:user)    { create(:user) }

  before do
    group.add_developer(user)
  end

  shared_examples 'a created issue' do
    let(:result) { described_class.new(project, user, params).execute }

    it 'creates the issue with the given params' do
      expect(result[:status]).to eq(:success)
      issue = result[:issue]
      expect(issue).to be_persisted
      expect(issue.project).to eq(project)
      expect(issue.author).to eq(user)
      expect(issue.title).to eq(expected_title)
      expect(issue.description).to eq(expected_description)
      expect(issue).to be_confidential
    end

    context 'when Issues::CreateService fails' do
      before do
        allow_next_instance_of(Issues::CreateService) do |create_service|
          allow(create_service).to receive(:execute).and_return(ServiceResponse.error(message: 'unexpected error'))
        end
      end

      it 'returns an error' do
        expect(result[:status]).to eq(:error)
        expect(result[:message]).to eq('unexpected error')
      end
    end
  end

  context 'when user does not have permission to create issue' do
    let(:result) { described_class.new(project, user, {}).execute }

    before do
      allow_next_instance_of(described_class) do |instance|
        allow(instance).to receive(:can?).with(user, :create_issue, project).and_return(false)
      end
    end

    it 'returns expected error' do
      expect(result[:status]).to eq(:error)
      expect(result[:message]).to eq("User is not permitted to create issue")
    end
  end

  context 'when issues are disabled on project' do
    let(:result) { described_class.new(project, user, {}).execute }
    let(:project) { build(:project, :public, namespace: group, issues_access_level: ProjectFeature::DISABLED) }

    it 'returns expected error' do
      expect(result[:status]).to eq(:error)
      expect(result[:message]).to eq("User is not permitted to create issue")
    end
  end

  context 'when successful' do
    let(:title) { "Vulnerability Title" }
    let(:vulnerability) { create(:vulnerability, :with_finding, title: title, project: project) }
    let(:vulnerability_presenter) { vulnerability.present }
    let(:params) do
      {
        vulnerability: vulnerability
      }
    end

    let(:expected_title) { "Investigate vulnerability: #{title}" }
    let(:expected_description) do
      <<~DESC.chomp
        Issue created from vulnerability #{ActionController::Base.helpers.link_to(vulnerability_presenter.id, vulnerability_url(vulnerability_presenter))}

        ### Description:

        Description of #{title}

        * Severity: high
        * Confidence: medium
        * Location: [#{vulnerability_presenter.location_text}](#{vulnerability_presenter.location_link})

        #{vulnerability_presenter.evidence_issue_description}

        ### Solution:

        #{vulnerability_presenter.solution}

        ### Identifiers:

        * [#{vulnerability_presenter.identifiers.first[:name]}](#{vulnerability_presenter.identifiers.first[:url]})

        ### Links:

        * [#{vulnerability_presenter.links.first[:name]}](#{vulnerability_presenter.links.first[:url]})

        ### Scanner:

        * Name: #{vulnerability_presenter.finding.scanner_name}
      DESC
    end

    it_behaves_like 'a created issue'

    context 'when the title of the vulnerability is longer than maximum issue title' do
      let(:max_title_length) { 255 }
      let(:title) { ('a' * max_title_length) }
      let(:expected_title) { "Investigate vulnerability: #{title}".truncate(max_title_length) }

      it_behaves_like 'a created issue'
    end
  end
end
