# frozen_string_literal: true

require 'spec_helper'

RSpec.describe 'Query.project(fullPath).vulnerabilitySeveritiesCount', feature_category: :vulnerability_management do
  let_it_be(:project) { create(:project) }
  let_it_be(:user) { create(:user) }
  let_it_be(:vulnerability) { create(:vulnerability, :high, :with_finding, project: project) }

  let_it_be(:query) do
    %(
      query {
        project(fullPath: "#{project.full_path}") {
          vulnerabilitySeveritiesCount {
            high
          }
        }
      }
    )
  end

  before do
    stub_licensed_features(security_dashboard: true)

    create_list(:vulnerability, 2, :high, :with_issue_links, :with_finding, resolved_on_default_branch: true, project: project)

    project.add_developer(user)
  end

  subject { GitlabSchema.execute(query, context: { current_user: user }).as_json }

  it "returns counts for each severity of the project's detected or confirmed vulnerabilities" do
    high_count = subject.dig('data', 'project', 'vulnerabilitySeveritiesCount', 'high')

    expect(high_count).to eq(3)
  end

  context 'with hasIssues filter' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            vulnerabilitySeveritiesCount(hasIssues: #{has_issues}) {
              high
            }
          }
        }
      )
    end

    context 'when counting vulnerabilities without issues' do
      let(:has_issues) { false }

      it 'counts vulnerabilities with issues' do
        expect(count_issues).to eq(1)
      end
    end

    context 'when counting vulnerabilities with issues' do
      let(:has_issues) { true }

      it 'counts vulnerabilities with issues' do
        expect(count_issues).to eq(2)
      end
    end
  end

  context 'with hasResolution filter' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            vulnerabilitySeveritiesCount(hasResolution: #{has_resolution}) {
              high
            }
          }
        }
      )
    end

    context 'when counting vulnerabilities without resolution' do
      let(:has_resolution) { false }

      it 'counts vulnerabilities with resolution' do
        expect(count_issues).to eq(1)
      end
    end

    context 'when counting vulnerabilities with resolution' do
      let(:has_resolution) { true }

      it 'counts vulnerabilities with resolution' do
        expect(count_issues).to eq(2)
      end
    end
  end

  def count_issues
    subject.dig('data', 'project', 'vulnerabilitySeveritiesCount', 'high')
  end

  context 'with scannerId filter' do
    let(:query) do
      %(
        query {
          project(fullPath: "#{project.full_path}") {
            vulnerabilitySeveritiesCount(scannerId: "#{GitlabSchema.id_from_object(vulnerability.finding.scanner)}") {
              high
            }
          }
        }
      )
    end

    it 'counts vulnerabilities with issues' do
      high_count = subject.dig('data', 'project', 'vulnerabilitySeveritiesCount', 'high')

      expect(high_count).to eq(1)
    end
  end
end
