import { nextTick } from 'vue';
import { GlFormGroup, GlDropdown, GlDropdownItem, GlFormRadio } from '@gitlab/ui';
import { mountExtended } from 'helpers/vue_test_utils_helper';
import SectionDetails from 'ee/vulnerabilities/components/new_vulnerability/section_details.vue';
import SeverityBadge from 'ee/vue_shared/security_reports/components/severity_badge.vue';
import { ERROR_SEVERITY, ERROR_STATUS } from 'ee/vulnerabilities/components/new_vulnerability/i18n';

describe('New vulnerability - Section Details', () => {
  let wrapper;

  const findFormGroup = (at) => wrapper.findAllComponents(GlFormGroup).at(at);

  const findDetectionMethodItem = (at) =>
    wrapper.findAllComponents(GlDropdown).at(0).findAllComponents(GlDropdownItem).at(at);

  const findSeverityMethodItem = (at) =>
    wrapper.findAllComponents(GlDropdown).at(1).findAllComponents(GlDropdownItem).at(at);

  const findStatusItem = (at) => wrapper.findAllComponents(GlFormRadio).at(at);

  const createWrapper = () => {
    return mountExtended(SectionDetails, {});
  };

  beforeEach(() => {
    wrapper = createWrapper();
  });

  it.each`
    value | text
    ${0}  | ${'GitLab Security Report'}
    ${1}  | ${'External Security Report'}
    ${2}  | ${'Bug Bounty'}
    ${3}  | ${'Code Review'}
    ${4}  | ${'Security Audit'}
  `('displays and handles detection method field: $text', ({ value, text }) => {
    const dropdownItem = findDetectionMethodItem(value);
    expect(dropdownItem.text()).toBe(text);
    dropdownItem.vm.$emit('click');
    expect(wrapper.emitted('change')[0][0]).toEqual({
      detectionMethod: `${value}`,
      severity: '',
      status: '',
    });
  });

  it.each`
    index | value
    ${0}  | ${'critical'}
    ${1}  | ${'high'}
    ${2}  | ${'medium'}
    ${3}  | ${'low'}
    ${4}  | ${'unknown'}
    ${5}  | ${'info'}
  `('displays and handles severity field: $value', ({ index, value }) => {
    const dropdownItem = findSeverityMethodItem(index);
    expect(dropdownItem.findComponent(SeverityBadge).props('severity')).toBe(value);
    dropdownItem.vm.$emit('click');
    expect(wrapper.emitted('change')[0][0]).toEqual({
      detectionMethod: -1,
      severity: value,
      status: '',
    });
  });

  it.each`
    index | value
    ${0}  | ${'detected'}
    ${1}  | ${'confirmed'}
    ${2}  | ${'resolved'}
  `('displays and handles status field', async ({ index, value }) => {
    await findStatusItem(index).find('input').trigger('change');
    expect(wrapper.emitted('change')[0][0]).toEqual({
      detectionMethod: -1,
      severity: '',
      status: value,
    });
  });

  it('does not display invalid state by default', () => {
    expect(findFormGroup(1).attributes('aria-invalid')).toBeUndefined();
    expect(findFormGroup(0).attributes('aria-invalid')).toBeUndefined();
  });

  it('handles form validation', async () => {
    wrapper.setProps({
      validationState: {
        severity: false,
        status: false,
      },
    });

    await nextTick();

    // severity input
    expect(findFormGroup(1).attributes('aria-invalid')).toBe('true');
    expect(findFormGroup(1).text()).toContain(ERROR_SEVERITY);

    // status input
    expect(findFormGroup(2).attributes('aria-invalid')).toBe('true');
    expect(findFormGroup(2).text()).toContain(ERROR_STATUS);
  });
});
