# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Gitlab::BackgroundMigration::DeleteApprovalRulesWithVulnerability do
  describe '#perform' do
    let(:batch_column) { :id }
    let(:sub_batch_size) { 1 }
    let(:pause_ms) { 0 }
    let(:connection) { ApplicationRecord.connection }
    let(:namespace) { table(:namespaces).create!(name: 'name', path: 'path') }
    let(:projects) { table(:projects) }
    let(:project) do
      projects
        .create!(name: "project", path: "project", namespace_id: namespace.id, project_namespace_id: namespace.id)
    end

    context 'with merge request approval rules' do
      let(:batch_table) { :approval_merge_request_rules }
      let(:approval_merge_request_rules) { table(:approval_merge_request_rules) }
      let(:merge_request) do
        table(:merge_requests).create!(target_project_id: project.id, target_branch: 'main', source_branch: 'feature')
      end

      let!(:merge_request_rule) do
        approval_merge_request_rules.create!(name: 'rule', merge_request_id: merge_request.id, report_type: 1)
      end

      let!(:merge_request_rule_other_report_type) do
        approval_merge_request_rules.create!(name: 'rule 2', merge_request_id: merge_request.id, report_type: 2)
      end

      let!(:merge_request_rule_last) do
        approval_merge_request_rules.create!(name: 'rule 3', merge_request_id: merge_request.id, report_type: 1)
      end

      subject do
        described_class.new(
          start_id: merge_request_rule.id,
          end_id: merge_request_rule_last.id,
          batch_table: batch_table,
          batch_column: batch_column,
          sub_batch_size: sub_batch_size,
          pause_ms: pause_ms,
          connection: connection
        ).perform
      end

      it 'removes merge request level approval rules with report_type equal to one' do
        expect { subject }.to change(approval_merge_request_rules, :count).from(3).to(1)

        expect(approval_merge_request_rules.all).to contain_exactly(merge_request_rule_other_report_type)
      end
    end

    context 'with project approval rules' do
      let(:batch_table) { :approval_project_rules }
      let(:approval_project_rules) { table(:approval_project_rules) }

      let!(:project_rule) do
        approval_project_rules.create!(name: 'rule', project_id: project.id, report_type: 1)
      end

      let!(:project_rule_other_report_type) do
        approval_project_rules.create!(name: 'rule 2', project_id: project.id, report_type: 2)
      end

      let!(:project_rule_last) do
        approval_project_rules.create!(name: 'rule 3', project_id: project.id, report_type: 1)
      end

      subject do
        described_class.new(
          start_id: project_rule.id,
          end_id: project_rule_last.id,
          batch_table: batch_table,
          batch_column: batch_column,
          sub_batch_size: sub_batch_size,
          pause_ms: pause_ms,
          connection: connection
        ).perform
      end

      it 'removes merge request level approval rules with report_type equal to one' do
        expect { subject }.to change(approval_project_rules, :count).from(3).to(1)

        expect(approval_project_rules.all).to contain_exactly(project_rule_other_report_type)
      end
    end
  end
end
