# frozen_string_literal: true

require 'spec_helper'

RSpec.describe Security::VulnerabilityReportDataEntity do
  let_it_be(:user) { create(:user) }
  let_it_be(:project) { create(:project) }
  let_it_be(:pipeline) { create(:ci_pipeline, project: project) }

  let(:entity) do
    described_class.new(pipeline, project: project, user: user)
  end

  subject { entity.as_json }

  context 'when false positives are available' do
    before do
      project.add_maintainer(user)
      stub_licensed_features(sast_fp_reduction: true)
    end

    it 'exposes correct attributes' do
      expect(subject).to include({
        pipeline_id: pipeline.id,
        pipeline_iid: pipeline.iid,
        source_branch: "master",
        empty_state_svg_path: match_asset_path("illustrations/user-not-logged-in.svg"),
        project_id: project.id,
        vulnerabilities_endpoint: "/api/v4/projects/#{project.id}/vulnerability_findings?pipeline_id=#{pipeline.id}",
        pipeline_jobs_path: "/api/v4/projects/#{project.id}/pipelines/#{pipeline.id}/jobs",
        vulnerability_exports_endpoint: "/api/v4/security/projects/#{project.id}/vulnerability_exports",
        empty_state_unauthorized_svg_path: match_asset_path("illustrations/user-not-logged-in.svg"),
        empty_state_forbidden_svg_path: match_asset_path("illustrations/lock_promotion.svg"),
        project_full_path: project.full_path,
        commit_path_template: "/#{project.full_path}/-/commit/$COMMIT_SHA",
        can_admin_vulnerability: "false",
        can_view_false_positive: "true"
      })
    end
  end

  context "when false positives are not available" do
    it 'exposes correct can_view_false_positive attribute' do
      expect(subject[:can_view_false_positive]).to eq("false")
    end
  end

  context "when user is not logged in" do
    let(:user) { nil }

    it 'exposes can_admin_vulnerability attribute with a fallback' do
      expect(subject[:can_admin_vulnerability]).to eq("false")
    end
  end
end
