import Vue from 'vue';
import { DASHBOARD_TYPES } from 'ee/security_dashboard/store/constants';
import { parseBoolean } from '~/lib/utils/common_utils';
import UnavailableState from './components/shared/empty_states/unavailable_state.vue';
import apolloProvider from './graphql/provider';
import createRouter from './router';
import ProjectVulnerabilityReport from './components/project/project_vulnerability_report.vue';
import GroupVulnerabilityReport from './components/group/group_vulnerability_report.vue';
import InstanceVulnerabilityReport from './components/instance/instance_vulnerability_report.vue';
import projectVulnerabilitiesQuery from './graphql/queries/project_vulnerabilities.query.graphql';
import groupVulnerabilitiesQuery from './graphql/queries/group_vulnerabilities.query.graphql';
import instanceVulnerabilitiesQuery from './graphql/queries/instance_vulnerabilities.query.graphql';

const DASHBOARD_CONFIGS = {
  [DASHBOARD_TYPES.PROJECT]: {
    component: ProjectVulnerabilityReport,
    query: projectVulnerabilitiesQuery,
  },
  [DASHBOARD_TYPES.GROUP]: {
    component: GroupVulnerabilityReport,
    query: groupVulnerabilitiesQuery,
  },
  [DASHBOARD_TYPES.INSTANCE]: {
    component: InstanceVulnerabilityReport,
    query: instanceVulnerabilitiesQuery,
  },
};

export default (el, dashboardType) => {
  if (!el) {
    return null;
  }

  const {
    isUnavailable,
    emptyStateSvgPath,
    noVulnerabilitiesSvgPath,
    securityDashboardEmptySvgPath,
    newProjectPipelinePath,
    hasVulnerabilities,
    hasProjects,
    scanners,
    vulnerabilitiesExportEndpoint,
    projectFullPath,
    autoFixMrsPath,
    groupFullPath,
    instanceDashboardSettingsPath,
    pipelineCreatedAt,
    pipelineId,
    pipelinePath,
    pipelineSecurityBuildsFailedCount,
    pipelineSecurityBuildsFailedPath,
    pipelineHasErrors,
    pipelineHasWarnings,
    hasJiraVulnerabilitiesIntegrationEnabled,
    operationalConfigurationPath,
    securityConfigurationPath,
    canAdminVulnerability,
    canViewFalsePositive,
    newVulnerabilityPath,
  } = el.dataset;

  if (isUnavailable) {
    return new Vue({
      el,
      render(createElement) {
        return createElement(UnavailableState, {
          props: { svgPath: emptyStateSvgPath },
        });
      },
    });
  }

  const provide = {
    dashboardType,
    noVulnerabilitiesSvgPath,
    emptyStateSvgPath,
    securityDashboardEmptySvgPath,
    newProjectPipelinePath,
    instanceDashboardSettingsPath,
    operationalConfigurationPath,
    securityConfigurationPath,
    vulnerabilitiesExportEndpoint,
    groupFullPath,
    projectFullPath,
    fullPath: projectFullPath || groupFullPath,
    autoFixMrsPath,
    canAdminVulnerability: parseBoolean(canAdminVulnerability),
    hasVulnerabilities: parseBoolean(hasVulnerabilities),
    hasProjects: parseBoolean(hasProjects),
    scanners: scanners ? JSON.parse(scanners) : [],
    hasJiraVulnerabilitiesIntegrationEnabled: parseBoolean(
      hasJiraVulnerabilitiesIntegrationEnabled,
    ),
    canViewFalsePositive: parseBoolean(canViewFalsePositive),
    newVulnerabilityPath,
    vulnerabilitiesQuery: DASHBOARD_CONFIGS[dashboardType].query,
  };

  if (dashboardType === DASHBOARD_TYPES.PROJECT) {
    provide.pipeline = {
      createdAt: pipelineCreatedAt,
      id: pipelineId,
      path: pipelinePath,
      securityBuildsFailedCount: Number(pipelineSecurityBuildsFailedCount),
      securityBuildsFailedPath: pipelineSecurityBuildsFailedPath,
      hasParsingErrors: parseBoolean(pipelineHasErrors),
      hasParsingWarnings: parseBoolean(pipelineHasWarnings),
    };
  }

  const router = createRouter();

  return new Vue({
    el,
    name: 'VulnerabilityReportRoot',
    router,
    apolloProvider,
    provide,
    render(createElement) {
      return createElement(DASHBOARD_CONFIGS[dashboardType].component);
    },
  });
};
