CREATE SCHEMA gitlab_partitions_dynamic;

COMMENT ON SCHEMA gitlab_partitions_dynamic IS 'Schema to hold partitions managed dynamically from the application, e.g. for time space partitioning.';

CREATE SCHEMA gitlab_partitions_static;

COMMENT ON SCHEMA gitlab_partitions_static IS 'Schema to hold static partitions, e.g. for hash partitioning';

CREATE EXTENSION IF NOT EXISTS btree_gist;

CREATE EXTENSION IF NOT EXISTS pg_trgm;

CREATE FUNCTION delete_associated_project_namespace() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
DELETE FROM namespaces
WHERE namespaces.id = OLD.project_namespace_id AND
namespaces.type = 'Project';
RETURN NULL;

END
$$;

CREATE FUNCTION insert_into_loose_foreign_keys_deleted_records() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
  INSERT INTO loose_foreign_keys_deleted_records
  (fully_qualified_table_name, primary_key_value)
  SELECT TG_TABLE_SCHEMA || '.' || TG_TABLE_NAME, old_table.id FROM old_table;

  RETURN NULL;
END
$$;

CREATE FUNCTION insert_namespaces_sync_event() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO namespaces_sync_events (namespace_id)
VALUES(COALESCE(NEW.id, OLD.id));
RETURN NULL;

END
$$;

CREATE FUNCTION insert_projects_sync_event() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
INSERT INTO projects_sync_events (project_id)
VALUES(COALESCE(NEW.id, OLD.id));
RETURN NULL;

END
$$;

CREATE FUNCTION integrations_set_type_new() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE integrations SET type_new = regexp_replace(NEW.type, '\A(.+)Service\Z', 'Integrations::\1')
WHERE integrations.id = NEW.id;
RETURN NULL;

END
$$;

CREATE FUNCTION next_traversal_ids_sibling(traversal_ids integer[]) RETURNS integer[]
    LANGUAGE plpgsql IMMUTABLE STRICT
    AS $$
BEGIN
  return traversal_ids[1:array_length(traversal_ids, 1)-1] ||
  ARRAY[traversal_ids[array_length(traversal_ids, 1)]+1];
END;
$$;

CREATE FUNCTION set_has_external_issue_tracker() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE projects SET has_external_issue_tracker = (
  EXISTS
  (
    SELECT 1
    FROM integrations
    WHERE project_id = COALESCE(NEW.project_id, OLD.project_id)
      AND active = TRUE
      AND category = 'issue_tracker'
  )
)
WHERE projects.id = COALESCE(NEW.project_id, OLD.project_id);
RETURN NULL;

END
$$;

CREATE FUNCTION set_has_external_wiki() RETURNS trigger
    LANGUAGE plpgsql
    AS $$
BEGIN
UPDATE projects SET has_external_wiki = COALESCE(NEW.active, FALSE)
WHERE projects.id = COALESCE(NEW.project_id, OLD.project_id);
RETURN NULL;

END
$$;

CREATE TABLE audit_events (
    id bigint NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
)
PARTITION BY RANGE (created_at);

CREATE SEQUENCE audit_events_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_id_seq OWNED BY audit_events.id;

CREATE TABLE gitlab_partitions_dynamic.audit_events_000000 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_000000 FOR VALUES FROM (MINVALUE) TO ('2022-12-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202212 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202212 FOR VALUES FROM ('2022-12-01 00:00:00') TO ('2023-01-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202301 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202301 FOR VALUES FROM ('2023-01-01 00:00:00') TO ('2023-02-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202302 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202302 FOR VALUES FROM ('2023-02-01 00:00:00') TO ('2023-03-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202303 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202303 FOR VALUES FROM ('2023-03-01 00:00:00') TO ('2023-04-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202304 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202304 FOR VALUES FROM ('2023-04-01 00:00:00') TO ('2023-05-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202305 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202305 FOR VALUES FROM ('2023-05-01 00:00:00') TO ('2023-06-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202306 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202306 FOR VALUES FROM ('2023-06-01 00:00:00') TO ('2023-07-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202307 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202307 FOR VALUES FROM ('2023-07-01 00:00:00') TO ('2023-08-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202308 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202308 FOR VALUES FROM ('2023-08-01 00:00:00') TO ('2023-09-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.audit_events_202309 (
    id bigint DEFAULT nextval('audit_events_id_seq'::regclass) NOT NULL,
    author_id integer NOT NULL,
    entity_id integer NOT NULL,
    entity_type character varying NOT NULL,
    details text,
    ip_address inet,
    author_name text,
    target_details text,
    entity_path text,
    created_at timestamp without time zone NOT NULL,
    target_type text,
    target_id bigint,
    CONSTRAINT check_492aaa021d CHECK ((char_length(entity_path) <= 5500)),
    CONSTRAINT check_83ff8406e2 CHECK ((char_length(author_name) <= 255)),
    CONSTRAINT check_97a8c868e7 CHECK ((char_length(target_type) <= 255)),
    CONSTRAINT check_d493ec90b5 CHECK ((char_length(target_details) <= 5500))
);
ALTER TABLE ONLY audit_events ATTACH PARTITION gitlab_partitions_dynamic.audit_events_202309 FOR VALUES FROM ('2023-09-01 00:00:00') TO ('2023-10-01 00:00:00');

CREATE TABLE batched_background_migration_job_transition_logs (
    id bigint NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
)
PARTITION BY RANGE (created_at);

CREATE SEQUENCE batched_background_migration_job_transition_logs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migration_job_transition_logs_id_seq OWNED BY batched_background_migration_job_transition_logs.id;

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_000000 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_000000 FOR VALUES FROM (MINVALUE) TO ('2023-02-28 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202303 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202303 FOR VALUES FROM ('2023-02-28 18:00:00-06') TO ('2023-03-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202304 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202304 FOR VALUES FROM ('2023-03-31 19:00:00-05') TO ('2023-04-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202305 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202305 FOR VALUES FROM ('2023-04-30 19:00:00-05') TO ('2023-05-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202306 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202306 FOR VALUES FROM ('2023-05-31 19:00:00-05') TO ('2023-06-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202307 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202307 FOR VALUES FROM ('2023-06-30 19:00:00-05') TO ('2023-07-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202308 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202308 FOR VALUES FROM ('2023-07-31 19:00:00-05') TO ('2023-08-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202309 (
    id bigint DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass) NOT NULL,
    batched_background_migration_job_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    previous_status smallint NOT NULL,
    next_status smallint NOT NULL,
    exception_class text,
    exception_message text,
    CONSTRAINT check_50e580811a CHECK ((char_length(exception_message) <= 1000)),
    CONSTRAINT check_76e202c37a CHECK ((char_length(exception_class) <= 100))
);
ALTER TABLE ONLY batched_background_migration_job_transition_logs ATTACH PARTITION gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202309 FOR VALUES FROM ('2023-08-31 19:00:00-05') TO ('2023-09-30 19:00:00-05');

CREATE TABLE incident_management_pending_alert_escalations (
    id bigint NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
)
PARTITION BY RANGE (process_at);

CREATE SEQUENCE incident_management_pending_alert_escalations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_pending_alert_escalations_id_seq OWNED BY incident_management_pending_alert_escalations.id;

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202211 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202212 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202301 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202301 FOR VALUES FROM ('2022-12-31 18:00:00-06') TO ('2023-01-31 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202302 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202302 FOR VALUES FROM ('2023-01-31 18:00:00-06') TO ('2023-02-28 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202303 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202303 FOR VALUES FROM ('2023-02-28 18:00:00-06') TO ('2023-03-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202304 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202304 FOR VALUES FROM ('2023-03-31 19:00:00-05') TO ('2023-04-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202305 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202305 FOR VALUES FROM ('2023-04-30 19:00:00-05') TO ('2023-05-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202306 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202306 FOR VALUES FROM ('2023-05-31 19:00:00-05') TO ('2023-06-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202307 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202307 FOR VALUES FROM ('2023-06-30 19:00:00-05') TO ('2023-07-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202308 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202308 FOR VALUES FROM ('2023-07-31 19:00:00-05') TO ('2023-08-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202309 (
    id bigint DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    alert_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_alert_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202309 FOR VALUES FROM ('2023-08-31 19:00:00-05') TO ('2023-09-30 19:00:00-05');

CREATE TABLE incident_management_pending_issue_escalations (
    id bigint NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
)
PARTITION BY RANGE (process_at);

CREATE SEQUENCE incident_management_pending_issue_escalations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_pending_issue_escalations_id_seq OWNED BY incident_management_pending_issue_escalations.id;

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202211 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202212 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202301 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202301 FOR VALUES FROM ('2022-12-31 18:00:00-06') TO ('2023-01-31 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202302 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202302 FOR VALUES FROM ('2023-01-31 18:00:00-06') TO ('2023-02-28 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202303 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202303 FOR VALUES FROM ('2023-02-28 18:00:00-06') TO ('2023-03-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202304 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202304 FOR VALUES FROM ('2023-03-31 19:00:00-05') TO ('2023-04-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202305 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202305 FOR VALUES FROM ('2023-04-30 19:00:00-05') TO ('2023-05-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202306 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202306 FOR VALUES FROM ('2023-05-31 19:00:00-05') TO ('2023-06-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202307 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202307 FOR VALUES FROM ('2023-06-30 19:00:00-05') TO ('2023-07-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202308 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202308 FOR VALUES FROM ('2023-07-31 19:00:00-05') TO ('2023-08-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202309 (
    id bigint DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass) NOT NULL,
    rule_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    process_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);
ALTER TABLE ONLY incident_management_pending_issue_escalations ATTACH PARTITION gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202309 FOR VALUES FROM ('2023-08-31 19:00:00-05') TO ('2023-09-30 19:00:00-05');

CREATE TABLE verification_codes (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
)
PARTITION BY RANGE (created_at);

COMMENT ON TABLE verification_codes IS 'JiHu-specific table';

CREATE TABLE gitlab_partitions_dynamic.verification_codes_000000 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_000000 FOR VALUES FROM (MINVALUE) TO ('2023-01-31 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202302 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202302 FOR VALUES FROM ('2023-01-31 18:00:00-06') TO ('2023-02-28 18:00:00-06');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202303 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202303 FOR VALUES FROM ('2023-02-28 18:00:00-06') TO ('2023-03-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202304 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202304 FOR VALUES FROM ('2023-03-31 19:00:00-05') TO ('2023-04-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202305 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202305 FOR VALUES FROM ('2023-04-30 19:00:00-05') TO ('2023-05-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202306 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202306 FOR VALUES FROM ('2023-05-31 19:00:00-05') TO ('2023-06-30 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202307 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202307 FOR VALUES FROM ('2023-06-30 19:00:00-05') TO ('2023-07-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202308 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202308 FOR VALUES FROM ('2023-07-31 19:00:00-05') TO ('2023-08-31 19:00:00-05');

CREATE TABLE gitlab_partitions_dynamic.verification_codes_202309 (
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    visitor_id_code text NOT NULL,
    code text NOT NULL,
    phone text NOT NULL,
    CONSTRAINT check_9b84e6aaff CHECK ((char_length(code) <= 8)),
    CONSTRAINT check_ccc542256b CHECK ((char_length(visitor_id_code) <= 64)),
    CONSTRAINT check_f5684c195b CHECK ((char_length(phone) <= 32))
);
ALTER TABLE ONLY verification_codes ATTACH PARTITION gitlab_partitions_dynamic.verification_codes_202309 FOR VALUES FROM ('2023-08-31 19:00:00-05') TO ('2023-09-30 19:00:00-05');

CREATE TABLE web_hook_logs (
    id bigint NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
)
PARTITION BY RANGE (created_at);

CREATE SEQUENCE web_hook_logs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE web_hook_logs_id_seq OWNED BY web_hook_logs.id;

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_000000 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202212 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202212 FOR VALUES FROM ('2022-12-01 00:00:00') TO ('2023-01-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202301 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202301 FOR VALUES FROM ('2023-01-01 00:00:00') TO ('2023-02-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202302 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202302 FOR VALUES FROM ('2023-02-01 00:00:00') TO ('2023-03-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202303 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202303 FOR VALUES FROM ('2023-03-01 00:00:00') TO ('2023-04-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202304 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202304 FOR VALUES FROM ('2023-04-01 00:00:00') TO ('2023-05-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202305 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202305 FOR VALUES FROM ('2023-05-01 00:00:00') TO ('2023-06-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202306 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202306 FOR VALUES FROM ('2023-06-01 00:00:00') TO ('2023-07-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202307 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202307 FOR VALUES FROM ('2023-07-01 00:00:00') TO ('2023-08-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202308 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202308 FOR VALUES FROM ('2023-08-01 00:00:00') TO ('2023-09-01 00:00:00');

CREATE TABLE gitlab_partitions_dynamic.web_hook_logs_202309 (
    id bigint DEFAULT nextval('web_hook_logs_id_seq'::regclass) NOT NULL,
    web_hook_id integer NOT NULL,
    trigger character varying,
    url character varying,
    request_headers text,
    request_data text,
    response_headers text,
    response_body text,
    response_status character varying,
    execution_duration double precision,
    internal_error_message character varying,
    updated_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL
);
ALTER TABLE ONLY web_hook_logs ATTACH PARTITION gitlab_partitions_dynamic.web_hook_logs_202309 FOR VALUES FROM ('2023-09-01 00:00:00') TO ('2023-10-01 00:00:00');

CREATE TABLE analytics_cycle_analytics_issue_stage_events (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
)
PARTITION BY HASH (stage_event_hash_id);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_00 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_00 FOR VALUES WITH (modulus 32, remainder 0);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_01 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_01 FOR VALUES WITH (modulus 32, remainder 1);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_02 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_02 FOR VALUES WITH (modulus 32, remainder 2);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_03 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_03 FOR VALUES WITH (modulus 32, remainder 3);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_04 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_04 FOR VALUES WITH (modulus 32, remainder 4);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_05 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_05 FOR VALUES WITH (modulus 32, remainder 5);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_06 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_06 FOR VALUES WITH (modulus 32, remainder 6);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_07 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_07 FOR VALUES WITH (modulus 32, remainder 7);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_08 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_08 FOR VALUES WITH (modulus 32, remainder 8);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_09 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_09 FOR VALUES WITH (modulus 32, remainder 9);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_10 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_10 FOR VALUES WITH (modulus 32, remainder 10);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_11 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_11 FOR VALUES WITH (modulus 32, remainder 11);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_12 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_12 FOR VALUES WITH (modulus 32, remainder 12);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_13 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_13 FOR VALUES WITH (modulus 32, remainder 13);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_14 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_14 FOR VALUES WITH (modulus 32, remainder 14);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_15 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_15 FOR VALUES WITH (modulus 32, remainder 15);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_16 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_16 FOR VALUES WITH (modulus 32, remainder 16);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_17 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_17 FOR VALUES WITH (modulus 32, remainder 17);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_18 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_18 FOR VALUES WITH (modulus 32, remainder 18);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_19 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_19 FOR VALUES WITH (modulus 32, remainder 19);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_20 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_20 FOR VALUES WITH (modulus 32, remainder 20);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_21 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_21 FOR VALUES WITH (modulus 32, remainder 21);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_22 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_22 FOR VALUES WITH (modulus 32, remainder 22);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_23 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_23 FOR VALUES WITH (modulus 32, remainder 23);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_24 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_24 FOR VALUES WITH (modulus 32, remainder 24);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_25 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_25 FOR VALUES WITH (modulus 32, remainder 25);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_26 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_26 FOR VALUES WITH (modulus 32, remainder 26);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_27 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_27 FOR VALUES WITH (modulus 32, remainder 27);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_28 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_28 FOR VALUES WITH (modulus 32, remainder 28);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_29 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_29 FOR VALUES WITH (modulus 32, remainder 29);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_30 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_30 FOR VALUES WITH (modulus 32, remainder 30);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_31 (
    stage_event_hash_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_31 FOR VALUES WITH (modulus 32, remainder 31);

CREATE TABLE analytics_cycle_analytics_merge_request_stage_events (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
)
PARTITION BY HASH (stage_event_hash_id);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_00 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_00 FOR VALUES WITH (modulus 32, remainder 0);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_01 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_01 FOR VALUES WITH (modulus 32, remainder 1);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_02 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_02 FOR VALUES WITH (modulus 32, remainder 2);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_03 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_03 FOR VALUES WITH (modulus 32, remainder 3);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_04 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_04 FOR VALUES WITH (modulus 32, remainder 4);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_05 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_05 FOR VALUES WITH (modulus 32, remainder 5);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_06 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_06 FOR VALUES WITH (modulus 32, remainder 6);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_07 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_07 FOR VALUES WITH (modulus 32, remainder 7);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_08 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_08 FOR VALUES WITH (modulus 32, remainder 8);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_09 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_09 FOR VALUES WITH (modulus 32, remainder 9);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_10 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_10 FOR VALUES WITH (modulus 32, remainder 10);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_11 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_11 FOR VALUES WITH (modulus 32, remainder 11);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_12 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_12 FOR VALUES WITH (modulus 32, remainder 12);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_13 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_13 FOR VALUES WITH (modulus 32, remainder 13);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_14 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_14 FOR VALUES WITH (modulus 32, remainder 14);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_15 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_15 FOR VALUES WITH (modulus 32, remainder 15);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_16 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_16 FOR VALUES WITH (modulus 32, remainder 16);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_17 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_17 FOR VALUES WITH (modulus 32, remainder 17);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_18 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_18 FOR VALUES WITH (modulus 32, remainder 18);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_19 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_19 FOR VALUES WITH (modulus 32, remainder 19);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_20 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_20 FOR VALUES WITH (modulus 32, remainder 20);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_21 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_21 FOR VALUES WITH (modulus 32, remainder 21);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_22 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_22 FOR VALUES WITH (modulus 32, remainder 22);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_23 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_23 FOR VALUES WITH (modulus 32, remainder 23);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_24 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_24 FOR VALUES WITH (modulus 32, remainder 24);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_25 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_25 FOR VALUES WITH (modulus 32, remainder 25);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_26 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_26 FOR VALUES WITH (modulus 32, remainder 26);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_27 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_27 FOR VALUES WITH (modulus 32, remainder 27);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_28 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_28 FOR VALUES WITH (modulus 32, remainder 28);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_29 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_29 FOR VALUES WITH (modulus 32, remainder 29);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_30 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_30 FOR VALUES WITH (modulus 32, remainder 30);

CREATE TABLE gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_31 (
    stage_event_hash_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    group_id bigint NOT NULL,
    project_id bigint NOT NULL,
    milestone_id bigint,
    author_id bigint,
    start_event_timestamp timestamp with time zone NOT NULL,
    end_event_timestamp timestamp with time zone,
    state_id smallint DEFAULT 1 NOT NULL
);
ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events ATTACH PARTITION gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_31 FOR VALUES WITH (modulus 32, remainder 31);

CREATE TABLE issue_search_data (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
)
PARTITION BY HASH (project_id);

CREATE TABLE gitlab_partitions_static.issue_search_data_00 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_00 FOR VALUES WITH (modulus 64, remainder 0);

CREATE TABLE gitlab_partitions_static.issue_search_data_01 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_01 FOR VALUES WITH (modulus 64, remainder 1);

CREATE TABLE gitlab_partitions_static.issue_search_data_02 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_02 FOR VALUES WITH (modulus 64, remainder 2);

CREATE TABLE gitlab_partitions_static.issue_search_data_03 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_03 FOR VALUES WITH (modulus 64, remainder 3);

CREATE TABLE gitlab_partitions_static.issue_search_data_04 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_04 FOR VALUES WITH (modulus 64, remainder 4);

CREATE TABLE gitlab_partitions_static.issue_search_data_05 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_05 FOR VALUES WITH (modulus 64, remainder 5);

CREATE TABLE gitlab_partitions_static.issue_search_data_06 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_06 FOR VALUES WITH (modulus 64, remainder 6);

CREATE TABLE gitlab_partitions_static.issue_search_data_07 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_07 FOR VALUES WITH (modulus 64, remainder 7);

CREATE TABLE gitlab_partitions_static.issue_search_data_08 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_08 FOR VALUES WITH (modulus 64, remainder 8);

CREATE TABLE gitlab_partitions_static.issue_search_data_09 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_09 FOR VALUES WITH (modulus 64, remainder 9);

CREATE TABLE gitlab_partitions_static.issue_search_data_10 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_10 FOR VALUES WITH (modulus 64, remainder 10);

CREATE TABLE gitlab_partitions_static.issue_search_data_11 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_11 FOR VALUES WITH (modulus 64, remainder 11);

CREATE TABLE gitlab_partitions_static.issue_search_data_12 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_12 FOR VALUES WITH (modulus 64, remainder 12);

CREATE TABLE gitlab_partitions_static.issue_search_data_13 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_13 FOR VALUES WITH (modulus 64, remainder 13);

CREATE TABLE gitlab_partitions_static.issue_search_data_14 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_14 FOR VALUES WITH (modulus 64, remainder 14);

CREATE TABLE gitlab_partitions_static.issue_search_data_15 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_15 FOR VALUES WITH (modulus 64, remainder 15);

CREATE TABLE gitlab_partitions_static.issue_search_data_16 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_16 FOR VALUES WITH (modulus 64, remainder 16);

CREATE TABLE gitlab_partitions_static.issue_search_data_17 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_17 FOR VALUES WITH (modulus 64, remainder 17);

CREATE TABLE gitlab_partitions_static.issue_search_data_18 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_18 FOR VALUES WITH (modulus 64, remainder 18);

CREATE TABLE gitlab_partitions_static.issue_search_data_19 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_19 FOR VALUES WITH (modulus 64, remainder 19);

CREATE TABLE gitlab_partitions_static.issue_search_data_20 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_20 FOR VALUES WITH (modulus 64, remainder 20);

CREATE TABLE gitlab_partitions_static.issue_search_data_21 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_21 FOR VALUES WITH (modulus 64, remainder 21);

CREATE TABLE gitlab_partitions_static.issue_search_data_22 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_22 FOR VALUES WITH (modulus 64, remainder 22);

CREATE TABLE gitlab_partitions_static.issue_search_data_23 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_23 FOR VALUES WITH (modulus 64, remainder 23);

CREATE TABLE gitlab_partitions_static.issue_search_data_24 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_24 FOR VALUES WITH (modulus 64, remainder 24);

CREATE TABLE gitlab_partitions_static.issue_search_data_25 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_25 FOR VALUES WITH (modulus 64, remainder 25);

CREATE TABLE gitlab_partitions_static.issue_search_data_26 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_26 FOR VALUES WITH (modulus 64, remainder 26);

CREATE TABLE gitlab_partitions_static.issue_search_data_27 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_27 FOR VALUES WITH (modulus 64, remainder 27);

CREATE TABLE gitlab_partitions_static.issue_search_data_28 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_28 FOR VALUES WITH (modulus 64, remainder 28);

CREATE TABLE gitlab_partitions_static.issue_search_data_29 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_29 FOR VALUES WITH (modulus 64, remainder 29);

CREATE TABLE gitlab_partitions_static.issue_search_data_30 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_30 FOR VALUES WITH (modulus 64, remainder 30);

CREATE TABLE gitlab_partitions_static.issue_search_data_31 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_31 FOR VALUES WITH (modulus 64, remainder 31);

CREATE TABLE gitlab_partitions_static.issue_search_data_32 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_32 FOR VALUES WITH (modulus 64, remainder 32);

CREATE TABLE gitlab_partitions_static.issue_search_data_33 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_33 FOR VALUES WITH (modulus 64, remainder 33);

CREATE TABLE gitlab_partitions_static.issue_search_data_34 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_34 FOR VALUES WITH (modulus 64, remainder 34);

CREATE TABLE gitlab_partitions_static.issue_search_data_35 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_35 FOR VALUES WITH (modulus 64, remainder 35);

CREATE TABLE gitlab_partitions_static.issue_search_data_36 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_36 FOR VALUES WITH (modulus 64, remainder 36);

CREATE TABLE gitlab_partitions_static.issue_search_data_37 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_37 FOR VALUES WITH (modulus 64, remainder 37);

CREATE TABLE gitlab_partitions_static.issue_search_data_38 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_38 FOR VALUES WITH (modulus 64, remainder 38);

CREATE TABLE gitlab_partitions_static.issue_search_data_39 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_39 FOR VALUES WITH (modulus 64, remainder 39);

CREATE TABLE gitlab_partitions_static.issue_search_data_40 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_40 FOR VALUES WITH (modulus 64, remainder 40);

CREATE TABLE gitlab_partitions_static.issue_search_data_41 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_41 FOR VALUES WITH (modulus 64, remainder 41);

CREATE TABLE gitlab_partitions_static.issue_search_data_42 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_42 FOR VALUES WITH (modulus 64, remainder 42);

CREATE TABLE gitlab_partitions_static.issue_search_data_43 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_43 FOR VALUES WITH (modulus 64, remainder 43);

CREATE TABLE gitlab_partitions_static.issue_search_data_44 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_44 FOR VALUES WITH (modulus 64, remainder 44);

CREATE TABLE gitlab_partitions_static.issue_search_data_45 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_45 FOR VALUES WITH (modulus 64, remainder 45);

CREATE TABLE gitlab_partitions_static.issue_search_data_46 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_46 FOR VALUES WITH (modulus 64, remainder 46);

CREATE TABLE gitlab_partitions_static.issue_search_data_47 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_47 FOR VALUES WITH (modulus 64, remainder 47);

CREATE TABLE gitlab_partitions_static.issue_search_data_48 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_48 FOR VALUES WITH (modulus 64, remainder 48);

CREATE TABLE gitlab_partitions_static.issue_search_data_49 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_49 FOR VALUES WITH (modulus 64, remainder 49);

CREATE TABLE gitlab_partitions_static.issue_search_data_50 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_50 FOR VALUES WITH (modulus 64, remainder 50);

CREATE TABLE gitlab_partitions_static.issue_search_data_51 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_51 FOR VALUES WITH (modulus 64, remainder 51);

CREATE TABLE gitlab_partitions_static.issue_search_data_52 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_52 FOR VALUES WITH (modulus 64, remainder 52);

CREATE TABLE gitlab_partitions_static.issue_search_data_53 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_53 FOR VALUES WITH (modulus 64, remainder 53);

CREATE TABLE gitlab_partitions_static.issue_search_data_54 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_54 FOR VALUES WITH (modulus 64, remainder 54);

CREATE TABLE gitlab_partitions_static.issue_search_data_55 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_55 FOR VALUES WITH (modulus 64, remainder 55);

CREATE TABLE gitlab_partitions_static.issue_search_data_56 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_56 FOR VALUES WITH (modulus 64, remainder 56);

CREATE TABLE gitlab_partitions_static.issue_search_data_57 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_57 FOR VALUES WITH (modulus 64, remainder 57);

CREATE TABLE gitlab_partitions_static.issue_search_data_58 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_58 FOR VALUES WITH (modulus 64, remainder 58);

CREATE TABLE gitlab_partitions_static.issue_search_data_59 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_59 FOR VALUES WITH (modulus 64, remainder 59);

CREATE TABLE gitlab_partitions_static.issue_search_data_60 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_60 FOR VALUES WITH (modulus 64, remainder 60);

CREATE TABLE gitlab_partitions_static.issue_search_data_61 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_61 FOR VALUES WITH (modulus 64, remainder 61);

CREATE TABLE gitlab_partitions_static.issue_search_data_62 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_62 FOR VALUES WITH (modulus 64, remainder 62);

CREATE TABLE gitlab_partitions_static.issue_search_data_63 (
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    search_vector tsvector
);
ALTER TABLE ONLY issue_search_data ATTACH PARTITION gitlab_partitions_static.issue_search_data_63 FOR VALUES WITH (modulus 64, remainder 63);

CREATE TABLE loose_foreign_keys_deleted_records (
    id bigint NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    primary_key_value bigint NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    fully_qualified_table_name text NOT NULL,
    consume_after timestamp with time zone DEFAULT now(),
    CONSTRAINT check_1a541f3235 CHECK ((char_length(fully_qualified_table_name) <= 150))
)
PARTITION BY LIST (partition);

CREATE SEQUENCE loose_foreign_keys_deleted_records_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE loose_foreign_keys_deleted_records_id_seq OWNED BY loose_foreign_keys_deleted_records.id;

CREATE TABLE gitlab_partitions_static.loose_foreign_keys_deleted_records_1 (
    id bigint DEFAULT nextval('loose_foreign_keys_deleted_records_id_seq'::regclass) NOT NULL,
    partition bigint DEFAULT 1 NOT NULL,
    primary_key_value bigint NOT NULL,
    status smallint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    fully_qualified_table_name text NOT NULL,
    consume_after timestamp with time zone DEFAULT now(),
    CONSTRAINT check_1a541f3235 CHECK ((char_length(fully_qualified_table_name) <= 150))
);
ALTER TABLE ONLY loose_foreign_keys_deleted_records ATTACH PARTITION gitlab_partitions_static.loose_foreign_keys_deleted_records_1 FOR VALUES IN ('1');

CREATE TABLE product_analytics_events_experimental (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
)
PARTITION BY HASH (project_id);

CREATE SEQUENCE product_analytics_events_experimental_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE product_analytics_events_experimental_id_seq OWNED BY product_analytics_events_experimental.id;

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_00 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_00 FOR VALUES WITH (modulus 64, remainder 0);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_01 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_01 FOR VALUES WITH (modulus 64, remainder 1);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_02 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_02 FOR VALUES WITH (modulus 64, remainder 2);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_03 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_03 FOR VALUES WITH (modulus 64, remainder 3);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_04 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_04 FOR VALUES WITH (modulus 64, remainder 4);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_05 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_05 FOR VALUES WITH (modulus 64, remainder 5);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_06 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_06 FOR VALUES WITH (modulus 64, remainder 6);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_07 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_07 FOR VALUES WITH (modulus 64, remainder 7);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_08 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_08 FOR VALUES WITH (modulus 64, remainder 8);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_09 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_09 FOR VALUES WITH (modulus 64, remainder 9);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_10 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_10 FOR VALUES WITH (modulus 64, remainder 10);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_11 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_11 FOR VALUES WITH (modulus 64, remainder 11);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_12 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_12 FOR VALUES WITH (modulus 64, remainder 12);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_13 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_13 FOR VALUES WITH (modulus 64, remainder 13);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_14 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_14 FOR VALUES WITH (modulus 64, remainder 14);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_15 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_15 FOR VALUES WITH (modulus 64, remainder 15);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_16 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_16 FOR VALUES WITH (modulus 64, remainder 16);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_17 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_17 FOR VALUES WITH (modulus 64, remainder 17);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_18 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_18 FOR VALUES WITH (modulus 64, remainder 18);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_19 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_19 FOR VALUES WITH (modulus 64, remainder 19);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_20 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_20 FOR VALUES WITH (modulus 64, remainder 20);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_21 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_21 FOR VALUES WITH (modulus 64, remainder 21);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_22 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_22 FOR VALUES WITH (modulus 64, remainder 22);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_23 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_23 FOR VALUES WITH (modulus 64, remainder 23);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_24 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_24 FOR VALUES WITH (modulus 64, remainder 24);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_25 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_25 FOR VALUES WITH (modulus 64, remainder 25);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_26 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_26 FOR VALUES WITH (modulus 64, remainder 26);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_27 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_27 FOR VALUES WITH (modulus 64, remainder 27);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_28 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_28 FOR VALUES WITH (modulus 64, remainder 28);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_29 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_29 FOR VALUES WITH (modulus 64, remainder 29);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_30 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_30 FOR VALUES WITH (modulus 64, remainder 30);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_31 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_31 FOR VALUES WITH (modulus 64, remainder 31);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_32 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_32 FOR VALUES WITH (modulus 64, remainder 32);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_33 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_33 FOR VALUES WITH (modulus 64, remainder 33);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_34 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_34 FOR VALUES WITH (modulus 64, remainder 34);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_35 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_35 FOR VALUES WITH (modulus 64, remainder 35);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_36 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_36 FOR VALUES WITH (modulus 64, remainder 36);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_37 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_37 FOR VALUES WITH (modulus 64, remainder 37);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_38 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_38 FOR VALUES WITH (modulus 64, remainder 38);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_39 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_39 FOR VALUES WITH (modulus 64, remainder 39);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_40 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_40 FOR VALUES WITH (modulus 64, remainder 40);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_41 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_41 FOR VALUES WITH (modulus 64, remainder 41);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_42 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_42 FOR VALUES WITH (modulus 64, remainder 42);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_43 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_43 FOR VALUES WITH (modulus 64, remainder 43);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_44 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_44 FOR VALUES WITH (modulus 64, remainder 44);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_45 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_45 FOR VALUES WITH (modulus 64, remainder 45);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_46 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_46 FOR VALUES WITH (modulus 64, remainder 46);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_47 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_47 FOR VALUES WITH (modulus 64, remainder 47);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_48 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_48 FOR VALUES WITH (modulus 64, remainder 48);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_49 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_49 FOR VALUES WITH (modulus 64, remainder 49);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_50 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_50 FOR VALUES WITH (modulus 64, remainder 50);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_51 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_51 FOR VALUES WITH (modulus 64, remainder 51);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_52 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_52 FOR VALUES WITH (modulus 64, remainder 52);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_53 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_53 FOR VALUES WITH (modulus 64, remainder 53);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_54 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_54 FOR VALUES WITH (modulus 64, remainder 54);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_55 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_55 FOR VALUES WITH (modulus 64, remainder 55);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_56 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_56 FOR VALUES WITH (modulus 64, remainder 56);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_57 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_57 FOR VALUES WITH (modulus 64, remainder 57);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_58 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_58 FOR VALUES WITH (modulus 64, remainder 58);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_59 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_59 FOR VALUES WITH (modulus 64, remainder 59);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_60 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_60 FOR VALUES WITH (modulus 64, remainder 60);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_61 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_61 FOR VALUES WITH (modulus 64, remainder 61);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_62 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_62 FOR VALUES WITH (modulus 64, remainder 62);

CREATE TABLE gitlab_partitions_static.product_analytics_events_experimental_63 (
    id bigint DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass) NOT NULL,
    project_id integer NOT NULL,
    platform character varying(255),
    etl_tstamp timestamp with time zone,
    collector_tstamp timestamp with time zone NOT NULL,
    dvce_created_tstamp timestamp with time zone,
    event character varying(128),
    event_id character(36) NOT NULL,
    txn_id integer,
    name_tracker character varying(128),
    v_tracker character varying(100),
    v_collector character varying(100) NOT NULL,
    v_etl character varying(100) NOT NULL,
    user_id character varying(255),
    user_ipaddress character varying(45),
    user_fingerprint character varying(50),
    domain_userid character varying(36),
    domain_sessionidx smallint,
    network_userid character varying(38),
    geo_country character(2),
    geo_region character(3),
    geo_city character varying(75),
    geo_zipcode character varying(15),
    geo_latitude double precision,
    geo_longitude double precision,
    geo_region_name character varying(100),
    ip_isp character varying(100),
    ip_organization character varying(100),
    ip_domain character varying(100),
    ip_netspeed character varying(100),
    page_url text,
    page_title character varying(2000),
    page_referrer text,
    page_urlscheme character varying(16),
    page_urlhost character varying(255),
    page_urlport integer,
    page_urlpath character varying(3000),
    page_urlquery character varying(6000),
    page_urlfragment character varying(3000),
    refr_urlscheme character varying(16),
    refr_urlhost character varying(255),
    refr_urlport integer,
    refr_urlpath character varying(6000),
    refr_urlquery character varying(6000),
    refr_urlfragment character varying(3000),
    refr_medium character varying(25),
    refr_source character varying(50),
    refr_term character varying(255),
    mkt_medium character varying(255),
    mkt_source character varying(255),
    mkt_term character varying(255),
    mkt_content character varying(500),
    mkt_campaign character varying(255),
    se_category character varying(1000),
    se_action character varying(1000),
    se_label character varying(1000),
    se_property character varying(1000),
    se_value double precision,
    tr_orderid character varying(255),
    tr_affiliation character varying(255),
    tr_total numeric(18,2),
    tr_tax numeric(18,2),
    tr_shipping numeric(18,2),
    tr_city character varying(255),
    tr_state character varying(255),
    tr_country character varying(255),
    ti_orderid character varying(255),
    ti_sku character varying(255),
    ti_name character varying(255),
    ti_category character varying(255),
    ti_price numeric(18,2),
    ti_quantity integer,
    pp_xoffset_min integer,
    pp_xoffset_max integer,
    pp_yoffset_min integer,
    pp_yoffset_max integer,
    useragent character varying(1000),
    br_name character varying(50),
    br_family character varying(50),
    br_version character varying(50),
    br_type character varying(50),
    br_renderengine character varying(50),
    br_lang character varying(255),
    br_features_pdf boolean,
    br_features_flash boolean,
    br_features_java boolean,
    br_features_director boolean,
    br_features_quicktime boolean,
    br_features_realplayer boolean,
    br_features_windowsmedia boolean,
    br_features_gears boolean,
    br_features_silverlight boolean,
    br_cookies boolean,
    br_colordepth character varying(12),
    br_viewwidth integer,
    br_viewheight integer,
    os_name character varying(50),
    os_family character varying(50),
    os_manufacturer character varying(50),
    os_timezone character varying(50),
    dvce_type character varying(50),
    dvce_ismobile boolean,
    dvce_screenwidth integer,
    dvce_screenheight integer,
    doc_charset character varying(128),
    doc_width integer,
    doc_height integer,
    tr_currency character(3),
    tr_total_base numeric(18,2),
    tr_tax_base numeric(18,2),
    tr_shipping_base numeric(18,2),
    ti_currency character(3),
    ti_price_base numeric(18,2),
    base_currency character(3),
    geo_timezone character varying(64),
    mkt_clickid character varying(128),
    mkt_network character varying(64),
    etl_tags character varying(500),
    dvce_sent_tstamp timestamp with time zone,
    refr_domain_userid character varying(36),
    refr_dvce_tstamp timestamp with time zone,
    domain_sessionid character(36),
    derived_tstamp timestamp with time zone,
    event_vendor character varying(1000),
    event_name character varying(1000),
    event_format character varying(128),
    event_version character varying(128),
    event_fingerprint character varying(128),
    true_tstamp timestamp with time zone
);
ALTER TABLE ONLY product_analytics_events_experimental ATTACH PARTITION gitlab_partitions_static.product_analytics_events_experimental_63 FOR VALUES WITH (modulus 64, remainder 63);

CREATE TABLE abuse_reports (
    id integer NOT NULL,
    reporter_id integer,
    user_id integer,
    message text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    message_html text,
    cached_markdown_version integer
);

CREATE SEQUENCE abuse_reports_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE abuse_reports_id_seq OWNED BY abuse_reports.id;

CREATE TABLE agent_activity_events (
    id bigint NOT NULL,
    agent_id bigint NOT NULL,
    user_id bigint,
    project_id bigint,
    merge_request_id bigint,
    agent_token_id bigint,
    recorded_at timestamp with time zone NOT NULL,
    kind smallint NOT NULL,
    level smallint NOT NULL,
    sha bytea,
    detail text,
    CONSTRAINT check_068205e735 CHECK ((char_length(detail) <= 255))
);

CREATE SEQUENCE agent_activity_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_activity_events_id_seq OWNED BY agent_activity_events.id;

CREATE TABLE agent_group_authorizations (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_group_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_group_authorizations_id_seq OWNED BY agent_group_authorizations.id;

CREATE TABLE agent_project_authorizations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    agent_id bigint NOT NULL,
    config jsonb NOT NULL
);

CREATE SEQUENCE agent_project_authorizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE agent_project_authorizations_id_seq OWNED BY agent_project_authorizations.id;

CREATE TABLE alert_management_alert_assignees (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    alert_id bigint NOT NULL
);

CREATE SEQUENCE alert_management_alert_assignees_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alert_assignees_id_seq OWNED BY alert_management_alert_assignees.id;

CREATE TABLE alert_management_alert_user_mentions (
    id bigint NOT NULL,
    alert_management_alert_id bigint NOT NULL,
    note_id bigint,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE alert_management_alert_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alert_user_mentions_id_seq OWNED BY alert_management_alert_user_mentions.id;

CREATE TABLE alert_management_alerts (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone NOT NULL,
    ended_at timestamp with time zone,
    events integer DEFAULT 1 NOT NULL,
    iid integer NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    fingerprint bytea,
    issue_id bigint,
    project_id bigint NOT NULL,
    title text NOT NULL,
    description text,
    service text,
    monitoring_tool text,
    hosts text[] DEFAULT '{}'::text[] NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    prometheus_alert_id integer,
    environment_id integer,
    domain smallint DEFAULT 0,
    CONSTRAINT check_2df3e2fdc1 CHECK ((char_length(monitoring_tool) <= 100)),
    CONSTRAINT check_5e9e57cadb CHECK ((char_length(description) <= 1000)),
    CONSTRAINT check_bac14dddde CHECK ((char_length(service) <= 100)),
    CONSTRAINT check_d1d1c2d14c CHECK ((char_length(title) <= 200))
);

CREATE SEQUENCE alert_management_alerts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_alerts_id_seq OWNED BY alert_management_alerts.id;

CREATE TABLE alert_management_http_integrations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    active boolean DEFAULT false NOT NULL,
    encrypted_token text NOT NULL,
    encrypted_token_iv text NOT NULL,
    endpoint_identifier text NOT NULL,
    name text NOT NULL,
    payload_example jsonb DEFAULT '{}'::jsonb NOT NULL,
    payload_attribute_mapping jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_286943b636 CHECK ((char_length(encrypted_token_iv) <= 255)),
    CONSTRAINT check_392143ccf4 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e270820180 CHECK ((char_length(endpoint_identifier) <= 255)),
    CONSTRAINT check_f68577c4af CHECK ((char_length(encrypted_token) <= 255))
);

CREATE SEQUENCE alert_management_http_integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE alert_management_http_integrations_id_seq OWNED BY alert_management_http_integrations.id;

CREATE TABLE allowed_email_domains (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id integer NOT NULL,
    domain character varying(255) NOT NULL
);

CREATE SEQUENCE allowed_email_domains_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE allowed_email_domains_id_seq OWNED BY allowed_email_domains.id;

CREATE TABLE analytics_cycle_analytics_group_stages (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_position integer,
    start_event_identifier integer NOT NULL,
    end_event_identifier integer NOT NULL,
    group_id bigint NOT NULL,
    start_event_label_id bigint,
    end_event_label_id bigint,
    hidden boolean DEFAULT false NOT NULL,
    custom boolean DEFAULT true NOT NULL,
    name character varying(255) NOT NULL,
    group_value_stream_id bigint NOT NULL,
    stage_event_hash_id bigint,
    CONSTRAINT check_e6bd4271b5 CHECK ((stage_event_hash_id IS NOT NULL))
);

CREATE SEQUENCE analytics_cycle_analytics_group_stages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_group_stages_id_seq OWNED BY analytics_cycle_analytics_group_stages.id;

CREATE TABLE analytics_cycle_analytics_group_value_streams (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_bc1ed5f1f7 CHECK ((char_length(name) <= 100))
);

CREATE SEQUENCE analytics_cycle_analytics_group_value_streams_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_group_value_streams_id_seq OWNED BY analytics_cycle_analytics_group_value_streams.id;

CREATE TABLE analytics_cycle_analytics_project_stages (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    relative_position integer,
    start_event_identifier integer NOT NULL,
    end_event_identifier integer NOT NULL,
    project_id bigint NOT NULL,
    start_event_label_id bigint,
    end_event_label_id bigint,
    hidden boolean DEFAULT false NOT NULL,
    custom boolean DEFAULT true NOT NULL,
    name character varying(255) NOT NULL,
    project_value_stream_id bigint NOT NULL,
    stage_event_hash_id bigint,
    CONSTRAINT check_8f6019de1e CHECK ((stage_event_hash_id IS NOT NULL))
);

CREATE SEQUENCE analytics_cycle_analytics_project_stages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_project_stages_id_seq OWNED BY analytics_cycle_analytics_project_stages.id;

CREATE TABLE analytics_cycle_analytics_project_value_streams (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_9b1970a898 CHECK ((char_length(name) <= 100))
);

CREATE SEQUENCE analytics_cycle_analytics_project_value_streams_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_project_value_streams_id_seq OWNED BY analytics_cycle_analytics_project_value_streams.id;

CREATE TABLE analytics_cycle_analytics_stage_event_hashes (
    id bigint NOT NULL,
    hash_sha256 bytea
);

CREATE SEQUENCE analytics_cycle_analytics_stage_event_hashes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_cycle_analytics_stage_event_hashes_id_seq OWNED BY analytics_cycle_analytics_stage_event_hashes.id;

CREATE TABLE analytics_devops_adoption_segments (
    id bigint NOT NULL,
    last_recorded_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id integer,
    display_namespace_id integer
);

CREATE SEQUENCE analytics_devops_adoption_segments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_devops_adoption_segments_id_seq OWNED BY analytics_devops_adoption_segments.id;

CREATE TABLE analytics_devops_adoption_snapshots (
    id bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    issue_opened boolean NOT NULL,
    merge_request_opened boolean NOT NULL,
    merge_request_approved boolean NOT NULL,
    runner_configured boolean NOT NULL,
    pipeline_succeeded boolean NOT NULL,
    deploy_succeeded boolean NOT NULL,
    security_scan_succeeded boolean NOT NULL,
    end_time timestamp with time zone NOT NULL,
    total_projects_count integer,
    code_owners_used_count integer,
    namespace_id integer,
    sast_enabled_count integer,
    dast_enabled_count integer,
    dependency_scanning_enabled_count integer,
    coverage_fuzzing_enabled_count integer,
    vulnerability_management_used_count integer,
    CONSTRAINT check_3f472de131 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE analytics_devops_adoption_snapshots_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_devops_adoption_snapshots_id_seq OWNED BY analytics_devops_adoption_snapshots.id;

CREATE TABLE analytics_language_trend_repository_languages (
    file_count integer DEFAULT 0 NOT NULL,
    programming_language_id bigint NOT NULL,
    project_id bigint NOT NULL,
    loc integer DEFAULT 0 NOT NULL,
    bytes integer DEFAULT 0 NOT NULL,
    percentage smallint DEFAULT 0 NOT NULL,
    snapshot_date date NOT NULL
);

CREATE TABLE analytics_usage_trends_measurements (
    id bigint NOT NULL,
    count bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    identifier smallint NOT NULL
);

CREATE SEQUENCE analytics_usage_trends_measurements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE analytics_usage_trends_measurements_id_seq OWNED BY analytics_usage_trends_measurements.id;

CREATE TABLE appearances (
    id integer NOT NULL,
    title character varying NOT NULL,
    description text NOT NULL,
    header_logo character varying,
    logo character varying,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    description_html text,
    cached_markdown_version integer,
    favicon character varying,
    new_project_guidelines text,
    new_project_guidelines_html text,
    header_message text,
    header_message_html text,
    footer_message text,
    footer_message_html text,
    message_background_color text,
    message_font_color text,
    email_header_and_footer_enabled boolean DEFAULT false NOT NULL,
    updated_by integer,
    profile_image_guidelines text,
    profile_image_guidelines_html text,
    CONSTRAINT appearances_profile_image_guidelines CHECK ((char_length(profile_image_guidelines) <= 4096))
);

CREATE SEQUENCE appearances_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE appearances_id_seq OWNED BY appearances.id;

CREATE TABLE application_setting_terms (
    id integer NOT NULL,
    cached_markdown_version integer,
    terms text NOT NULL,
    terms_html text
);

CREATE SEQUENCE application_setting_terms_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE application_setting_terms_id_seq OWNED BY application_setting_terms.id;

CREATE TABLE application_settings (
    id integer NOT NULL,
    default_projects_limit integer,
    signup_enabled boolean,
    gravatar_enabled boolean,
    sign_in_text text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    home_page_url character varying,
    default_branch_protection integer DEFAULT 2,
    restricted_visibility_levels text,
    version_check_enabled boolean DEFAULT true,
    max_attachment_size integer DEFAULT 10 NOT NULL,
    default_project_visibility integer DEFAULT 0 NOT NULL,
    default_snippet_visibility integer DEFAULT 0 NOT NULL,
    user_oauth_applications boolean DEFAULT true,
    after_sign_out_path character varying,
    session_expire_delay integer DEFAULT 10080 NOT NULL,
    import_sources text,
    help_page_text text,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    max_artifacts_size integer DEFAULT 100 NOT NULL,
    runners_registration_token character varying,
    max_pages_size integer DEFAULT 100 NOT NULL,
    require_two_factor_authentication boolean DEFAULT false,
    two_factor_grace_period integer DEFAULT 48,
    metrics_enabled boolean DEFAULT false,
    metrics_host character varying DEFAULT 'localhost'::character varying,
    metrics_pool_size integer DEFAULT 16,
    metrics_timeout integer DEFAULT 10,
    metrics_method_call_threshold integer DEFAULT 10,
    recaptcha_enabled boolean DEFAULT false,
    metrics_port integer DEFAULT 8089,
    akismet_enabled boolean DEFAULT false,
    metrics_sample_interval integer DEFAULT 15,
    email_author_in_body boolean DEFAULT false,
    default_group_visibility integer,
    repository_checks_enabled boolean DEFAULT false,
    shared_runners_text text,
    metrics_packet_size integer DEFAULT 1,
    disabled_oauth_sign_in_sources text,
    health_check_access_token character varying,
    send_user_confirmation_email boolean DEFAULT false,
    container_registry_token_expire_delay integer DEFAULT 5,
    after_sign_up_text text,
    user_default_external boolean DEFAULT false NOT NULL,
    repository_storages character varying DEFAULT 'default'::character varying,
    enabled_git_access_protocol character varying,
    usage_ping_enabled boolean DEFAULT true NOT NULL,
    sign_in_text_html text,
    help_page_text_html text,
    shared_runners_text_html text,
    after_sign_up_text_html text,
    rsa_key_restriction integer DEFAULT 0 NOT NULL,
    dsa_key_restriction integer DEFAULT '-1'::integer NOT NULL,
    ecdsa_key_restriction integer DEFAULT 0 NOT NULL,
    ed25519_key_restriction integer DEFAULT 0 NOT NULL,
    housekeeping_enabled boolean DEFAULT true NOT NULL,
    housekeeping_bitmaps_enabled boolean DEFAULT true NOT NULL,
    housekeeping_incremental_repack_period integer DEFAULT 10 NOT NULL,
    housekeeping_full_repack_period integer DEFAULT 50 NOT NULL,
    housekeeping_gc_period integer DEFAULT 200 NOT NULL,
    html_emails_enabled boolean DEFAULT true,
    plantuml_url character varying,
    plantuml_enabled boolean,
    terminal_max_session_time integer DEFAULT 0 NOT NULL,
    unique_ips_limit_per_user integer,
    unique_ips_limit_time_window integer,
    unique_ips_limit_enabled boolean DEFAULT false NOT NULL,
    default_artifacts_expire_in character varying DEFAULT '0'::character varying NOT NULL,
    uuid character varying,
    polling_interval_multiplier numeric DEFAULT 1.0 NOT NULL,
    cached_markdown_version integer,
    prometheus_metrics_enabled boolean DEFAULT true NOT NULL,
    authorized_keys_enabled boolean DEFAULT true NOT NULL,
    help_page_hide_commercial_content boolean DEFAULT false,
    help_page_support_url character varying,
    performance_bar_allowed_group_id integer,
    hashed_storage_enabled boolean DEFAULT true NOT NULL,
    project_export_enabled boolean DEFAULT true NOT NULL,
    auto_devops_enabled boolean DEFAULT true NOT NULL,
    throttle_unauthenticated_enabled boolean DEFAULT false NOT NULL,
    throttle_unauthenticated_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_api_requests_per_period integer DEFAULT 7200 NOT NULL,
    throttle_authenticated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_web_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_web_requests_per_period integer DEFAULT 7200 NOT NULL,
    throttle_authenticated_web_period_in_seconds integer DEFAULT 3600 NOT NULL,
    gitaly_timeout_default integer DEFAULT 55 NOT NULL,
    gitaly_timeout_medium integer DEFAULT 30 NOT NULL,
    gitaly_timeout_fast integer DEFAULT 10 NOT NULL,
    password_authentication_enabled_for_web boolean,
    password_authentication_enabled_for_git boolean DEFAULT true NOT NULL,
    external_authorization_service_enabled boolean DEFAULT false NOT NULL,
    external_authorization_service_url character varying,
    external_authorization_service_default_label character varying,
    default_project_creation integer DEFAULT 2 NOT NULL,
    auto_devops_domain character varying,
    pages_domain_verification_enabled boolean DEFAULT true NOT NULL,
    user_default_internal_regex character varying,
    external_authorization_service_timeout double precision DEFAULT 0.5,
    external_auth_client_cert text,
    encrypted_external_auth_client_key text,
    encrypted_external_auth_client_key_iv character varying,
    encrypted_external_auth_client_key_pass character varying,
    encrypted_external_auth_client_key_pass_iv character varying,
    enforce_terms boolean DEFAULT false,
    mirror_available boolean DEFAULT true NOT NULL,
    hide_third_party_offers boolean DEFAULT false NOT NULL,
    receive_max_input_size integer,
    web_ide_clientside_preview_enabled boolean DEFAULT false NOT NULL,
    user_show_add_ssh_key_message boolean DEFAULT true NOT NULL,
    outbound_local_requests_whitelist character varying(255)[] DEFAULT '{}'::character varying[] NOT NULL,
    usage_stats_set_by_user_id integer,
    diff_max_patch_bytes integer DEFAULT 204800 NOT NULL,
    archive_builds_in_seconds integer,
    commit_email_hostname character varying,
    first_day_of_week integer DEFAULT 0 NOT NULL,
    protected_ci_variables boolean DEFAULT true NOT NULL,
    runners_registration_token_encrypted character varying,
    local_markdown_version integer DEFAULT 0 NOT NULL,
    asset_proxy_enabled boolean DEFAULT false NOT NULL,
    asset_proxy_url character varying,
    encrypted_asset_proxy_secret_key text,
    encrypted_asset_proxy_secret_key_iv character varying,
    lets_encrypt_notification_email character varying,
    lets_encrypt_terms_of_service_accepted boolean DEFAULT false NOT NULL,
    help_text text,
    elasticsearch_indexing boolean DEFAULT false NOT NULL,
    elasticsearch_search boolean DEFAULT false NOT NULL,
    shared_runners_minutes integer DEFAULT 0 NOT NULL,
    repository_size_limit bigint DEFAULT 0,
    elasticsearch_url character varying DEFAULT 'http://localhost:9200'::character varying,
    elasticsearch_aws boolean DEFAULT false NOT NULL,
    elasticsearch_aws_region character varying DEFAULT 'us-east-1'::character varying,
    elasticsearch_aws_access_key character varying,
    geo_status_timeout integer DEFAULT 10,
    check_namespace_plan boolean DEFAULT false NOT NULL,
    mirror_max_delay integer DEFAULT 300 NOT NULL,
    mirror_max_capacity integer DEFAULT 100 NOT NULL,
    mirror_capacity_threshold integer DEFAULT 50 NOT NULL,
    slack_app_enabled boolean DEFAULT false,
    slack_app_id character varying,
    allow_group_owners_to_manage_ldap boolean DEFAULT true NOT NULL,
    email_additional_text character varying,
    file_template_project_id integer,
    pseudonymizer_enabled boolean DEFAULT false NOT NULL,
    snowplow_enabled boolean DEFAULT false NOT NULL,
    snowplow_cookie_domain character varying,
    custom_project_templates_group_id integer,
    elasticsearch_limit_indexing boolean DEFAULT false NOT NULL,
    geo_node_allowed_ips character varying DEFAULT '0.0.0.0/0, ::/0'::character varying,
    elasticsearch_shards integer DEFAULT 5 NOT NULL,
    elasticsearch_replicas integer DEFAULT 1 NOT NULL,
    encrypted_lets_encrypt_private_key text,
    encrypted_lets_encrypt_private_key_iv text,
    required_instance_ci_template character varying,
    dns_rebinding_protection_enabled boolean DEFAULT true NOT NULL,
    lock_memberships_to_ldap boolean DEFAULT false NOT NULL,
    default_project_deletion_protection boolean DEFAULT false NOT NULL,
    time_tracking_limit_to_hours boolean DEFAULT false NOT NULL,
    grafana_enabled boolean DEFAULT false NOT NULL,
    grafana_url character varying DEFAULT '/-/grafana'::character varying NOT NULL,
    raw_blob_request_limit integer DEFAULT 300 NOT NULL,
    login_recaptcha_protection_enabled boolean DEFAULT false NOT NULL,
    static_objects_external_storage_url character varying(255),
    static_objects_external_storage_auth_token character varying(255),
    instance_administration_project_id bigint,
    allow_local_requests_from_web_hooks_and_services boolean DEFAULT false,
    allow_local_requests_from_system_hooks boolean DEFAULT true NOT NULL,
    throttle_protected_paths_enabled boolean DEFAULT false NOT NULL,
    throttle_protected_paths_requests_per_period integer DEFAULT 10 NOT NULL,
    throttle_protected_paths_period_in_seconds integer DEFAULT 60 NOT NULL,
    protected_paths character varying(255)[] DEFAULT '{/users/password,/users/sign_in,/api/v3/session.json,/api/v3/session,/api/v4/session.json,/api/v4/session,/users,/users/confirmation,/unsubscribes/,/import/github/personal_access_token,/admin/session,/oauth/authorize,/oauth/token}'::character varying[],
    snowplow_collector_hostname character varying,
    sourcegraph_enabled boolean DEFAULT false NOT NULL,
    sourcegraph_url character varying(255),
    max_personal_access_token_lifetime integer,
    throttle_incident_management_notification_enabled boolean DEFAULT false NOT NULL,
    throttle_incident_management_notification_period_in_seconds integer DEFAULT 3600,
    throttle_incident_management_notification_per_period integer DEFAULT 3600,
    push_event_hooks_limit integer DEFAULT 3 NOT NULL,
    productivity_analytics_start_date timestamp with time zone,
    push_event_activities_limit integer DEFAULT 3 NOT NULL,
    custom_http_clone_url_root character varying(511),
    deletion_adjourned_period integer DEFAULT 7 NOT NULL,
    snowplow_app_id character varying,
    eks_integration_enabled boolean DEFAULT false NOT NULL,
    eks_account_id character varying(128),
    eks_access_key_id character varying(128),
    encrypted_eks_secret_access_key_iv character varying(255),
    encrypted_eks_secret_access_key text,
    license_trial_ends_on date,
    sourcegraph_public_only boolean DEFAULT true NOT NULL,
    default_ci_config_path character varying(255),
    snippet_size_limit bigint DEFAULT 52428800 NOT NULL,
    encrypted_akismet_api_key text,
    encrypted_akismet_api_key_iv character varying(255),
    encrypted_elasticsearch_aws_secret_access_key text,
    encrypted_elasticsearch_aws_secret_access_key_iv character varying(255),
    encrypted_recaptcha_private_key text,
    encrypted_recaptcha_private_key_iv character varying(255),
    encrypted_recaptcha_site_key text,
    encrypted_recaptcha_site_key_iv character varying(255),
    encrypted_slack_app_secret text,
    encrypted_slack_app_secret_iv character varying(255),
    encrypted_slack_app_verification_token text,
    encrypted_slack_app_verification_token_iv character varying(255),
    minimum_password_length integer DEFAULT 8 NOT NULL,
    updating_name_disabled_for_users boolean DEFAULT false NOT NULL,
    force_pages_access_control boolean DEFAULT false NOT NULL,
    instance_administrators_group_id integer,
    disable_overriding_approvers_per_merge_request boolean DEFAULT false NOT NULL,
    prevent_merge_requests_author_approval boolean DEFAULT false NOT NULL,
    prevent_merge_requests_committers_approval boolean DEFAULT false NOT NULL,
    elasticsearch_indexed_field_length_limit integer DEFAULT 0 NOT NULL,
    elasticsearch_max_bulk_size_mb smallint DEFAULT 10 NOT NULL,
    elasticsearch_max_bulk_concurrency smallint DEFAULT 10 NOT NULL,
    email_restrictions_enabled boolean DEFAULT false NOT NULL,
    email_restrictions text,
    npm_package_requests_forwarding boolean DEFAULT true NOT NULL,
    push_rule_id bigint,
    issues_create_limit integer DEFAULT 0 NOT NULL,
    container_expiration_policies_enable_historic_entries boolean DEFAULT false NOT NULL,
    group_owners_can_manage_default_branch_protection boolean DEFAULT true NOT NULL,
    container_registry_vendor text DEFAULT ''::text NOT NULL,
    container_registry_version text DEFAULT ''::text NOT NULL,
    container_registry_features text[] DEFAULT '{}'::text[] NOT NULL,
    spam_check_endpoint_url text,
    spam_check_endpoint_enabled boolean DEFAULT false NOT NULL,
    elasticsearch_pause_indexing boolean DEFAULT false NOT NULL,
    repository_storages_weighted jsonb DEFAULT '{}'::jsonb NOT NULL,
    max_import_size integer DEFAULT 0 NOT NULL,
    enforce_pat_expiration boolean DEFAULT true NOT NULL,
    compliance_frameworks smallint[] DEFAULT '{}'::smallint[] NOT NULL,
    notify_on_unknown_sign_in boolean DEFAULT true NOT NULL,
    default_branch_name text,
    maintenance_mode boolean DEFAULT false NOT NULL,
    maintenance_mode_message text,
    project_import_limit integer DEFAULT 6 NOT NULL,
    project_export_limit integer DEFAULT 6 NOT NULL,
    project_download_export_limit integer DEFAULT 1 NOT NULL,
    group_import_limit integer DEFAULT 6 NOT NULL,
    group_export_limit integer DEFAULT 6 NOT NULL,
    group_download_export_limit integer DEFAULT 1 NOT NULL,
    container_registry_delete_tags_service_timeout integer DEFAULT 250 NOT NULL,
    wiki_page_max_content_bytes bigint DEFAULT 52428800 NOT NULL,
    elasticsearch_indexed_file_size_limit_kb integer DEFAULT 1024 NOT NULL,
    enforce_namespace_storage_limit boolean DEFAULT false NOT NULL,
    gitpod_enabled boolean DEFAULT false NOT NULL,
    gitpod_url text DEFAULT 'https://gitpod.io/'::text,
    elasticsearch_client_request_timeout integer DEFAULT 0 NOT NULL,
    abuse_notification_email character varying,
    kroki_url text,
    kroki_enabled boolean DEFAULT false NOT NULL,
    help_page_documentation_base_url text,
    container_registry_expiration_policies_worker_capacity integer DEFAULT 0 NOT NULL,
    require_admin_approval_after_user_signup boolean DEFAULT true NOT NULL,
    automatic_purchased_storage_allocation boolean DEFAULT false NOT NULL,
    encrypted_ci_jwt_signing_key text,
    encrypted_ci_jwt_signing_key_iv text,
    elasticsearch_analyzers_smartcn_enabled boolean DEFAULT false NOT NULL,
    elasticsearch_analyzers_smartcn_search boolean DEFAULT false NOT NULL,
    elasticsearch_analyzers_kuromoji_enabled boolean DEFAULT false NOT NULL,
    elasticsearch_analyzers_kuromoji_search boolean DEFAULT false NOT NULL,
    new_user_signups_cap integer,
    secret_detection_token_revocation_enabled boolean DEFAULT false NOT NULL,
    secret_detection_token_revocation_url text,
    encrypted_secret_detection_token_revocation_token text,
    encrypted_secret_detection_token_revocation_token_iv text,
    domain_denylist_enabled boolean DEFAULT false,
    domain_denylist text,
    domain_allowlist text,
    secret_detection_revocation_token_types_url text,
    encrypted_cloud_license_auth_token text,
    encrypted_cloud_license_auth_token_iv text,
    personal_access_token_prefix text DEFAULT 'glpat-'::text,
    kroki_formats jsonb DEFAULT '{}'::jsonb NOT NULL,
    disable_feed_token boolean DEFAULT false NOT NULL,
    container_registry_cleanup_tags_service_max_list_size integer DEFAULT 200 NOT NULL,
    invisible_captcha_enabled boolean DEFAULT false NOT NULL,
    rate_limiting_response_text text,
    keep_latest_artifact boolean DEFAULT true NOT NULL,
    enforce_ssh_key_expiration boolean DEFAULT true NOT NULL,
    git_two_factor_session_expiry integer DEFAULT 15 NOT NULL,
    notes_create_limit integer DEFAULT 300 NOT NULL,
    notes_create_limit_allowlist text[] DEFAULT '{}'::text[] NOT NULL,
    in_product_marketing_emails_enabled boolean DEFAULT true NOT NULL,
    asset_proxy_whitelist text,
    delayed_project_removal boolean DEFAULT false NOT NULL,
    lock_delayed_project_removal boolean DEFAULT false NOT NULL,
    admin_mode boolean DEFAULT false NOT NULL,
    throttle_unauthenticated_packages_api_requests_per_period integer DEFAULT 800 NOT NULL,
    throttle_unauthenticated_packages_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_authenticated_packages_api_requests_per_period integer DEFAULT 1000 NOT NULL,
    throttle_authenticated_packages_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_unauthenticated_packages_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_packages_api_enabled boolean DEFAULT false NOT NULL,
    valid_runner_registrars character varying[] DEFAULT '{project,group}'::character varying[],
    whats_new_variant smallint DEFAULT 0,
    external_pipeline_validation_service_timeout integer,
    encrypted_external_pipeline_validation_service_token text,
    encrypted_external_pipeline_validation_service_token_iv text,
    external_pipeline_validation_service_url text,
    deactivate_dormant_users boolean DEFAULT false NOT NULL,
    encrypted_spam_check_api_key bytea,
    encrypted_spam_check_api_key_iv bytea,
    elasticsearch_username text,
    encrypted_elasticsearch_password bytea,
    encrypted_elasticsearch_password_iv bytea,
    floc_enabled boolean DEFAULT false NOT NULL,
    diff_max_lines integer DEFAULT 50000 NOT NULL,
    diff_max_files integer DEFAULT 1000 NOT NULL,
    encrypted_mailgun_signing_key bytea,
    encrypted_mailgun_signing_key_iv bytea,
    mailgun_events_enabled boolean DEFAULT false NOT NULL,
    usage_ping_features_enabled boolean DEFAULT false NOT NULL,
    encrypted_customers_dot_jwt_signing_key bytea,
    encrypted_customers_dot_jwt_signing_key_iv bytea,
    pypi_package_requests_forwarding boolean DEFAULT true NOT NULL,
    max_yaml_size_bytes bigint DEFAULT 1048576 NOT NULL,
    max_yaml_depth integer DEFAULT 100 NOT NULL,
    throttle_unauthenticated_files_api_requests_per_period integer DEFAULT 125 NOT NULL,
    throttle_unauthenticated_files_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_authenticated_files_api_requests_per_period integer DEFAULT 500 NOT NULL,
    throttle_authenticated_files_api_period_in_seconds integer DEFAULT 15 NOT NULL,
    throttle_unauthenticated_files_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_files_api_enabled boolean DEFAULT false NOT NULL,
    user_deactivation_emails_enabled boolean DEFAULT true NOT NULL,
    throttle_unauthenticated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_unauthenticated_api_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_git_lfs_requests_per_period integer DEFAULT 1000 NOT NULL,
    throttle_authenticated_git_lfs_period_in_seconds integer DEFAULT 60 NOT NULL,
    throttle_authenticated_git_lfs_enabled boolean DEFAULT false NOT NULL,
    jobs_per_stage_page_size integer DEFAULT 200 NOT NULL,
    sidekiq_job_limiter_mode smallint DEFAULT 1 NOT NULL,
    sidekiq_job_limiter_compression_threshold_bytes integer DEFAULT 100000 NOT NULL,
    sidekiq_job_limiter_limit_bytes integer DEFAULT 0 NOT NULL,
    dependency_proxy_ttl_group_policy_worker_capacity smallint DEFAULT 2 NOT NULL,
    throttle_unauthenticated_deprecated_api_requests_per_period integer DEFAULT 1800 NOT NULL,
    throttle_unauthenticated_deprecated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_unauthenticated_deprecated_api_enabled boolean DEFAULT false NOT NULL,
    throttle_authenticated_deprecated_api_requests_per_period integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_deprecated_api_period_in_seconds integer DEFAULT 3600 NOT NULL,
    throttle_authenticated_deprecated_api_enabled boolean DEFAULT false NOT NULL,
    suggest_pipeline_enabled boolean DEFAULT true NOT NULL,
    content_validation_endpoint_url text,
    encrypted_content_validation_api_key bytea,
    encrypted_content_validation_api_key_iv bytea,
    content_validation_endpoint_enabled boolean DEFAULT false NOT NULL,
    sentry_enabled boolean DEFAULT false NOT NULL,
    sentry_dsn text,
    sentry_clientside_dsn text,
    sentry_environment text,
    max_ssh_key_lifetime integer,
    static_objects_external_storage_auth_token_encrypted text,
    future_subscriptions jsonb DEFAULT '[]'::jsonb NOT NULL,
    CONSTRAINT app_settings_container_reg_cleanup_tags_max_list_size_positive CHECK ((container_registry_cleanup_tags_service_max_list_size >= 0)),
    CONSTRAINT app_settings_dep_proxy_ttl_policies_worker_capacity_positive CHECK ((dependency_proxy_ttl_group_policy_worker_capacity >= 0)),
    CONSTRAINT app_settings_ext_pipeline_validation_service_url_text_limit CHECK ((char_length(external_pipeline_validation_service_url) <= 255)),
    CONSTRAINT app_settings_registry_exp_policies_worker_capacity_positive CHECK ((container_registry_expiration_policies_worker_capacity >= 0)),
    CONSTRAINT app_settings_yaml_max_depth_positive CHECK ((max_yaml_depth > 0)),
    CONSTRAINT app_settings_yaml_max_size_positive CHECK ((max_yaml_size_bytes > 0)),
    CONSTRAINT check_17d9558205 CHECK ((char_length(kroki_url) <= 1024)),
    CONSTRAINT check_2dba05b802 CHECK ((char_length(gitpod_url) <= 255)),
    CONSTRAINT check_32710817e9 CHECK ((char_length(static_objects_external_storage_auth_token_encrypted) <= 255)),
    CONSTRAINT check_3def0f1829 CHECK ((char_length(sentry_clientside_dsn) <= 255)),
    CONSTRAINT check_4f8b811780 CHECK ((char_length(sentry_dsn) <= 255)),
    CONSTRAINT check_51700b31b5 CHECK ((char_length(default_branch_name) <= 255)),
    CONSTRAINT check_57123c9593 CHECK ((char_length(help_page_documentation_base_url) <= 255)),
    CONSTRAINT check_5a84c3ffdc CHECK ((char_length(content_validation_endpoint_url) <= 255)),
    CONSTRAINT check_5bcba483c4 CHECK ((char_length(sentry_environment) <= 255)),
    CONSTRAINT check_718b4458ae CHECK ((char_length(personal_access_token_prefix) <= 20)),
    CONSTRAINT check_7227fad848 CHECK ((char_length(rate_limiting_response_text) <= 255)),
    CONSTRAINT check_85a39b68ff CHECK ((char_length(encrypted_ci_jwt_signing_key_iv) <= 255)),
    CONSTRAINT check_9a719834eb CHECK ((char_length(secret_detection_token_revocation_url) <= 255)),
    CONSTRAINT check_9c6c447a13 CHECK ((char_length(maintenance_mode_message) <= 255)),
    CONSTRAINT check_a5704163cc CHECK ((char_length(secret_detection_revocation_token_types_url) <= 255)),
    CONSTRAINT check_b4f67a6296 CHECK ((allow_local_requests_from_web_hooks_and_services IS NOT NULL)),
    CONSTRAINT check_d03919528d CHECK ((char_length(container_registry_vendor) <= 255)),
    CONSTRAINT check_d820146492 CHECK ((char_length(spam_check_endpoint_url) <= 255)),
    CONSTRAINT check_e5024c8801 CHECK ((char_length(elasticsearch_username) <= 255)),
    CONSTRAINT check_e5aba18f02 CHECK ((char_length(container_registry_version) <= 255)),
    CONSTRAINT check_ef6176834f CHECK ((char_length(encrypted_cloud_license_auth_token_iv) <= 255))
);

COMMENT ON COLUMN application_settings.content_validation_endpoint_url IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_content_validation_api_key IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.encrypted_content_validation_api_key_iv IS 'JiHu-specific column';

COMMENT ON COLUMN application_settings.content_validation_endpoint_enabled IS 'JiHu-specific column';

CREATE SEQUENCE application_settings_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE application_settings_id_seq OWNED BY application_settings.id;

CREATE TABLE approval_merge_request_rule_sources (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL
);

CREATE SEQUENCE approval_merge_request_rule_sources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rule_sources_id_seq OWNED BY approval_merge_request_rule_sources.id;

CREATE TABLE approval_merge_request_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    merge_request_id integer NOT NULL,
    approvals_required smallint DEFAULT 0 NOT NULL,
    name character varying NOT NULL,
    rule_type smallint DEFAULT 1 NOT NULL,
    report_type smallint,
    section text,
    modified_from_project_rule boolean DEFAULT false NOT NULL,
    orchestration_policy_idx smallint,
    CONSTRAINT check_6fca5928b2 CHECK ((char_length(section) <= 255))
);

CREATE TABLE approval_merge_request_rules_approved_approvers (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    user_id integer NOT NULL
);

CREATE SEQUENCE approval_merge_request_rules_approved_approvers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_approved_approvers_id_seq OWNED BY approval_merge_request_rules_approved_approvers.id;

CREATE TABLE approval_merge_request_rules_groups (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    group_id integer NOT NULL
);

CREATE SEQUENCE approval_merge_request_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_groups_id_seq OWNED BY approval_merge_request_rules_groups.id;

CREATE SEQUENCE approval_merge_request_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_id_seq OWNED BY approval_merge_request_rules.id;

CREATE TABLE approval_merge_request_rules_users (
    id bigint NOT NULL,
    approval_merge_request_rule_id bigint NOT NULL,
    user_id integer NOT NULL
);

CREATE SEQUENCE approval_merge_request_rules_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_merge_request_rules_users_id_seq OWNED BY approval_merge_request_rules_users.id;

CREATE TABLE approval_project_rules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    approvals_required smallint DEFAULT 0 NOT NULL,
    name character varying NOT NULL,
    rule_type smallint DEFAULT 0 NOT NULL,
    scanners text[],
    vulnerabilities_allowed smallint DEFAULT 0 NOT NULL,
    severity_levels text[] DEFAULT '{}'::text[] NOT NULL,
    report_type smallint,
    vulnerability_states text[] DEFAULT '{newly_detected}'::text[] NOT NULL,
    orchestration_policy_idx smallint
);

CREATE TABLE approval_project_rules_groups (
    id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL,
    group_id integer NOT NULL
);

CREATE SEQUENCE approval_project_rules_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_groups_id_seq OWNED BY approval_project_rules_groups.id;

CREATE SEQUENCE approval_project_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_id_seq OWNED BY approval_project_rules.id;

CREATE TABLE approval_project_rules_protected_branches (
    approval_project_rule_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL
);

CREATE TABLE approval_project_rules_users (
    id bigint NOT NULL,
    approval_project_rule_id bigint NOT NULL,
    user_id integer NOT NULL
);

CREATE SEQUENCE approval_project_rules_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approval_project_rules_users_id_seq OWNED BY approval_project_rules_users.id;

CREATE TABLE approvals (
    id integer NOT NULL,
    merge_request_id integer NOT NULL,
    user_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE approvals_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approvals_id_seq OWNED BY approvals.id;

CREATE TABLE approver_groups (
    id integer NOT NULL,
    target_id integer NOT NULL,
    target_type character varying NOT NULL,
    group_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE approver_groups_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approver_groups_id_seq OWNED BY approver_groups.id;

CREATE TABLE approvers (
    id integer NOT NULL,
    target_id integer NOT NULL,
    target_type character varying,
    user_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE approvers_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE approvers_id_seq OWNED BY approvers.id;

CREATE TABLE atlassian_identities (
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expires_at timestamp with time zone,
    extern_uid text NOT NULL,
    encrypted_token bytea,
    encrypted_token_iv bytea,
    encrypted_refresh_token bytea,
    encrypted_refresh_token_iv bytea,
    CONSTRAINT atlassian_identities_refresh_token_iv_length_constraint CHECK ((octet_length(encrypted_refresh_token_iv) <= 12)),
    CONSTRAINT atlassian_identities_refresh_token_length_constraint CHECK ((octet_length(encrypted_refresh_token) <= 512)),
    CONSTRAINT atlassian_identities_token_iv_length_constraint CHECK ((octet_length(encrypted_token_iv) <= 12)),
    CONSTRAINT atlassian_identities_token_length_constraint CHECK ((octet_length(encrypted_token) <= 2048)),
    CONSTRAINT check_32f5779763 CHECK ((char_length(extern_uid) <= 255))
);

CREATE SEQUENCE atlassian_identities_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE atlassian_identities_user_id_seq OWNED BY atlassian_identities.user_id;

CREATE TABLE audit_events_external_audit_event_destinations (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    destination_url text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_2feafb9daf CHECK ((char_length(destination_url) <= 255))
);

CREATE SEQUENCE audit_events_external_audit_event_destinations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE audit_events_external_audit_event_destinations_id_seq OWNED BY audit_events_external_audit_event_destinations.id;

CREATE TABLE authentication_events (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    user_id bigint,
    result smallint NOT NULL,
    ip_address inet,
    provider text NOT NULL,
    user_name text NOT NULL,
    CONSTRAINT check_45a6cc4e80 CHECK ((char_length(user_name) <= 255)),
    CONSTRAINT check_c64f424630 CHECK ((char_length(provider) <= 64))
);

CREATE SEQUENCE authentication_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE authentication_events_id_seq OWNED BY authentication_events.id;

CREATE TABLE award_emoji (
    id integer NOT NULL,
    name character varying,
    user_id integer,
    awardable_type character varying,
    awardable_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE award_emoji_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE award_emoji_id_seq OWNED BY award_emoji.id;

CREATE TABLE aws_roles (
    user_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    role_arn character varying(2048),
    role_external_id character varying(64) NOT NULL,
    region text,
    CONSTRAINT check_57adedab55 CHECK ((char_length(region) <= 255))
);

CREATE TABLE background_migration_jobs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    class_name text NOT NULL,
    arguments jsonb NOT NULL,
    CONSTRAINT check_b0de0a5852 CHECK ((char_length(class_name) <= 200))
);

CREATE SEQUENCE background_migration_jobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE background_migration_jobs_id_seq OWNED BY background_migration_jobs.id;

CREATE TABLE badges (
    id integer NOT NULL,
    link_url character varying NOT NULL,
    image_url character varying NOT NULL,
    project_id integer,
    group_id integer,
    type character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying(255)
);

CREATE SEQUENCE badges_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE badges_id_seq OWNED BY badges.id;

CREATE TABLE banned_users (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL
);

CREATE TABLE batched_background_migration_jobs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    batched_background_migration_id bigint NOT NULL,
    min_value bigint NOT NULL,
    max_value bigint NOT NULL,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    attempts smallint DEFAULT 0 NOT NULL,
    metrics jsonb DEFAULT '{}'::jsonb NOT NULL,
    pause_ms integer DEFAULT 100 NOT NULL
);

CREATE SEQUENCE batched_background_migration_jobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migration_jobs_id_seq OWNED BY batched_background_migration_jobs.id;

CREATE TABLE batched_background_migrations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    min_value bigint DEFAULT 1 NOT NULL,
    max_value bigint NOT NULL,
    batch_size integer NOT NULL,
    sub_batch_size integer NOT NULL,
    "interval" smallint NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    job_class_name text NOT NULL,
    batch_class_name text DEFAULT 'PrimaryKeyBatchingStrategy'::text NOT NULL,
    table_name text NOT NULL,
    column_name text NOT NULL,
    job_arguments jsonb DEFAULT '"[]"'::jsonb NOT NULL,
    total_tuple_count bigint,
    pause_ms integer DEFAULT 100 NOT NULL,
    CONSTRAINT check_5bb0382d6f CHECK ((char_length(column_name) <= 63)),
    CONSTRAINT check_6b6a06254a CHECK ((char_length(table_name) <= 63)),
    CONSTRAINT check_batch_size_in_range CHECK ((batch_size >= sub_batch_size)),
    CONSTRAINT check_e6c75b1e29 CHECK ((char_length(job_class_name) <= 100)),
    CONSTRAINT check_fe10674721 CHECK ((char_length(batch_class_name) <= 100)),
    CONSTRAINT check_max_value_in_range CHECK ((max_value >= min_value)),
    CONSTRAINT check_positive_min_value CHECK ((min_value > 0)),
    CONSTRAINT check_positive_sub_batch_size CHECK ((sub_batch_size > 0))
);

CREATE SEQUENCE batched_background_migrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE batched_background_migrations_id_seq OWNED BY batched_background_migrations.id;

CREATE TABLE board_assignees (
    id integer NOT NULL,
    board_id integer NOT NULL,
    assignee_id integer NOT NULL
);

CREATE SEQUENCE board_assignees_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_assignees_id_seq OWNED BY board_assignees.id;

CREATE TABLE board_group_recent_visits (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id integer,
    board_id integer,
    group_id integer
);

CREATE SEQUENCE board_group_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_group_recent_visits_id_seq OWNED BY board_group_recent_visits.id;

CREATE TABLE board_labels (
    id integer NOT NULL,
    board_id integer NOT NULL,
    label_id integer NOT NULL
);

CREATE SEQUENCE board_labels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_labels_id_seq OWNED BY board_labels.id;

CREATE TABLE board_project_recent_visits (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id integer,
    project_id integer,
    board_id integer
);

CREATE SEQUENCE board_project_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_project_recent_visits_id_seq OWNED BY board_project_recent_visits.id;

CREATE TABLE board_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    board_id bigint NOT NULL,
    hide_labels boolean,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE board_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE board_user_preferences_id_seq OWNED BY board_user_preferences.id;

CREATE TABLE boards (
    id integer NOT NULL,
    project_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    group_id integer,
    milestone_id integer,
    weight integer,
    name character varying DEFAULT 'Development'::character varying NOT NULL,
    hide_backlog_list boolean DEFAULT false NOT NULL,
    hide_closed_list boolean DEFAULT false NOT NULL,
    iteration_id bigint,
    iteration_cadence_id bigint
);

CREATE TABLE boards_epic_board_labels (
    id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    label_id bigint NOT NULL
);

CREATE SEQUENCE boards_epic_board_labels_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_labels_id_seq OWNED BY boards_epic_board_labels.id;

CREATE TABLE boards_epic_board_positions (
    id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    epic_id bigint NOT NULL,
    relative_position integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE boards_epic_board_positions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_positions_id_seq OWNED BY boards_epic_board_positions.id;

CREATE TABLE boards_epic_board_recent_visits (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_board_id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE boards_epic_board_recent_visits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_board_recent_visits_id_seq OWNED BY boards_epic_board_recent_visits.id;

CREATE TABLE boards_epic_boards (
    id bigint NOT NULL,
    hide_backlog_list boolean DEFAULT false NOT NULL,
    hide_closed_list boolean DEFAULT false NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text DEFAULT 'Development'::text NOT NULL,
    CONSTRAINT check_bcbbffe601 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE boards_epic_boards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_boards_id_seq OWNED BY boards_epic_boards.id;

CREATE TABLE boards_epic_list_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_list_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    collapsed boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE boards_epic_list_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_list_user_preferences_id_seq OWNED BY boards_epic_list_user_preferences.id;

CREATE TABLE boards_epic_lists (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    epic_board_id bigint NOT NULL,
    label_id bigint,
    "position" integer,
    list_type smallint DEFAULT 1 NOT NULL,
    CONSTRAINT boards_epic_lists_position_constraint CHECK (((list_type <> 1) OR (("position" IS NOT NULL) AND ("position" >= 0))))
);

CREATE SEQUENCE boards_epic_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_lists_id_seq OWNED BY boards_epic_lists.id;

CREATE TABLE boards_epic_user_preferences (
    id bigint NOT NULL,
    board_id bigint NOT NULL,
    user_id bigint NOT NULL,
    epic_id bigint NOT NULL,
    collapsed boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE boards_epic_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_epic_user_preferences_id_seq OWNED BY boards_epic_user_preferences.id;

CREATE SEQUENCE boards_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE boards_id_seq OWNED BY boards.id;

CREATE TABLE broadcast_messages (
    id integer NOT NULL,
    message text NOT NULL,
    starts_at timestamp without time zone NOT NULL,
    ends_at timestamp without time zone NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    color character varying,
    font character varying,
    message_html text NOT NULL,
    cached_markdown_version integer,
    dismissable boolean,
    target_path character varying(255),
    broadcast_type smallint DEFAULT 1 NOT NULL
);

CREATE SEQUENCE broadcast_messages_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE broadcast_messages_id_seq OWNED BY broadcast_messages.id;

CREATE TABLE bulk_import_configurations (
    id bigint NOT NULL,
    bulk_import_id integer NOT NULL,
    encrypted_url text,
    encrypted_url_iv text,
    encrypted_access_token text,
    encrypted_access_token_iv text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE bulk_import_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_configurations_id_seq OWNED BY bulk_import_configurations.id;

CREATE TABLE bulk_import_entities (
    id bigint NOT NULL,
    bulk_import_id bigint NOT NULL,
    parent_id bigint,
    namespace_id bigint,
    project_id bigint,
    source_type smallint NOT NULL,
    source_full_path text NOT NULL,
    destination_name text NOT NULL,
    destination_namespace text NOT NULL,
    status smallint NOT NULL,
    jid text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_13f279f7da CHECK ((char_length(source_full_path) <= 255)),
    CONSTRAINT check_715d725ea2 CHECK ((char_length(destination_name) <= 255)),
    CONSTRAINT check_796a4d9cc6 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_b834fff4d9 CHECK ((char_length(destination_namespace) <= 255))
);

CREATE SEQUENCE bulk_import_entities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_entities_id_seq OWNED BY bulk_import_entities.id;

CREATE TABLE bulk_import_export_uploads (
    id bigint NOT NULL,
    export_id bigint NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    export_file text,
    CONSTRAINT check_5add76239d CHECK ((char_length(export_file) <= 255))
);

CREATE SEQUENCE bulk_import_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_export_uploads_id_seq OWNED BY bulk_import_export_uploads.id;

CREATE TABLE bulk_import_exports (
    id bigint NOT NULL,
    group_id bigint,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    relation text NOT NULL,
    jid text,
    error text,
    CONSTRAINT check_24cb010672 CHECK ((char_length(relation) <= 255)),
    CONSTRAINT check_8f0f357334 CHECK ((char_length(error) <= 255)),
    CONSTRAINT check_9ee6d14d33 CHECK ((char_length(jid) <= 255))
);

CREATE SEQUENCE bulk_import_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_exports_id_seq OWNED BY bulk_import_exports.id;

CREATE TABLE bulk_import_failures (
    id bigint NOT NULL,
    bulk_import_entity_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    pipeline_class text NOT NULL,
    exception_class text NOT NULL,
    exception_message text NOT NULL,
    correlation_id_value text,
    pipeline_step text,
    CONSTRAINT check_053d65c7a4 CHECK ((char_length(pipeline_class) <= 255)),
    CONSTRAINT check_6eca8f972e CHECK ((char_length(exception_message) <= 255)),
    CONSTRAINT check_721a422375 CHECK ((char_length(pipeline_step) <= 255)),
    CONSTRAINT check_c7dba8398e CHECK ((char_length(exception_class) <= 255)),
    CONSTRAINT check_e787285882 CHECK ((char_length(correlation_id_value) <= 255))
);

CREATE SEQUENCE bulk_import_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_failures_id_seq OWNED BY bulk_import_failures.id;

CREATE TABLE bulk_import_trackers (
    id bigint NOT NULL,
    bulk_import_entity_id bigint NOT NULL,
    relation text NOT NULL,
    next_page text,
    has_next_page boolean DEFAULT false NOT NULL,
    jid text,
    stage smallint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_2d45cae629 CHECK ((char_length(relation) <= 255)),
    CONSTRAINT check_40aeaa600b CHECK ((char_length(next_page) <= 255)),
    CONSTRAINT check_603f91cb06 CHECK ((char_length(jid) <= 255)),
    CONSTRAINT check_next_page_requirement CHECK (((has_next_page IS FALSE) OR (next_page IS NOT NULL)))
);

CREATE SEQUENCE bulk_import_trackers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_import_trackers_id_seq OWNED BY bulk_import_trackers.id;

CREATE TABLE bulk_imports (
    id bigint NOT NULL,
    user_id integer NOT NULL,
    source_type smallint NOT NULL,
    status smallint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    source_version text,
    CONSTRAINT check_ea4e58775a CHECK ((char_length(source_version) <= 63))
);

CREATE SEQUENCE bulk_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE bulk_imports_id_seq OWNED BY bulk_imports.id;

CREATE TABLE chat_names (
    id integer NOT NULL,
    user_id integer NOT NULL,
    service_id integer NOT NULL,
    team_id character varying NOT NULL,
    team_domain character varying,
    chat_id character varying NOT NULL,
    chat_name character varying,
    last_used_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE chat_names_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE chat_names_id_seq OWNED BY chat_names.id;

CREATE TABLE chat_teams (
    id integer NOT NULL,
    namespace_id integer NOT NULL,
    team_id character varying,
    name character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE chat_teams_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE chat_teams_id_seq OWNED BY chat_teams.id;

CREATE TABLE ci_build_needs (
    id integer NOT NULL,
    name text NOT NULL,
    artifacts boolean DEFAULT true NOT NULL,
    optional boolean DEFAULT false NOT NULL,
    build_id bigint NOT NULL
);

CREATE SEQUENCE ci_build_needs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_needs_id_seq OWNED BY ci_build_needs.id;

CREATE TABLE ci_build_pending_states (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    build_id bigint NOT NULL,
    state smallint,
    failure_reason smallint,
    trace_checksum bytea,
    trace_bytesize bigint
);

CREATE SEQUENCE ci_build_pending_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_pending_states_id_seq OWNED BY ci_build_pending_states.id;

CREATE TABLE ci_build_report_results (
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    data jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE ci_build_report_results_build_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_report_results_build_id_seq OWNED BY ci_build_report_results.build_id;

CREATE TABLE ci_build_trace_chunks (
    id bigint NOT NULL,
    chunk_index integer NOT NULL,
    data_store integer NOT NULL,
    raw_data bytea,
    checksum bytea,
    lock_version integer DEFAULT 0 NOT NULL,
    build_id bigint NOT NULL
);

CREATE SEQUENCE ci_build_trace_chunks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_build_trace_chunks_id_seq OWNED BY ci_build_trace_chunks.id;

CREATE TABLE ci_build_trace_metadata (
    build_id bigint NOT NULL,
    trace_artifact_id bigint,
    archival_attempts smallint DEFAULT 0 NOT NULL,
    checksum bytea,
    remote_checksum bytea,
    last_archival_attempt_at timestamp with time zone,
    archived_at timestamp with time zone
);

CREATE TABLE ci_builds (
    status character varying,
    finished_at timestamp without time zone,
    trace text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    started_at timestamp without time zone,
    runner_id integer,
    coverage double precision,
    commit_id integer,
    name character varying,
    options text,
    allow_failure boolean DEFAULT false NOT NULL,
    stage character varying,
    trigger_request_id integer,
    stage_idx integer,
    tag boolean,
    ref character varying,
    user_id integer,
    type character varying,
    target_url character varying,
    description character varying,
    project_id integer,
    erased_by_id integer,
    erased_at timestamp without time zone,
    artifacts_expire_at timestamp without time zone,
    environment character varying,
    "when" character varying,
    yaml_variables text,
    queued_at timestamp without time zone,
    token character varying,
    lock_version integer DEFAULT 0,
    coverage_regex character varying,
    auto_canceled_by_id integer,
    retried boolean,
    protected boolean,
    failure_reason integer,
    scheduled_at timestamp with time zone,
    token_encrypted character varying,
    upstream_pipeline_id integer,
    processed boolean,
    resource_group_id bigint,
    waiting_for_resource_at timestamp with time zone,
    scheduling_type smallint,
    id bigint NOT NULL,
    stage_id bigint
);

CREATE SEQUENCE ci_builds_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_id_seq OWNED BY ci_builds.id;

CREATE TABLE ci_builds_metadata (
    project_id integer NOT NULL,
    timeout integer,
    timeout_source integer DEFAULT 1 NOT NULL,
    config_options jsonb,
    config_variables jsonb,
    interruptible boolean,
    has_exposed_artifacts boolean,
    environment_auto_stop_in character varying(255),
    expanded_environment_name character varying(255),
    secrets jsonb DEFAULT '{}'::jsonb NOT NULL,
    build_id bigint NOT NULL,
    id bigint NOT NULL,
    runtime_runner_features jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE ci_builds_metadata_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_metadata_id_seq OWNED BY ci_builds_metadata.id;

CREATE TABLE ci_builds_runner_session (
    id bigint NOT NULL,
    url character varying NOT NULL,
    certificate character varying,
    "authorization" character varying,
    build_id bigint NOT NULL
);

CREATE SEQUENCE ci_builds_runner_session_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_builds_runner_session_id_seq OWNED BY ci_builds_runner_session.id;

CREATE TABLE ci_daily_build_group_report_results (
    id bigint NOT NULL,
    date date NOT NULL,
    project_id bigint NOT NULL,
    last_pipeline_id bigint NOT NULL,
    ref_path text NOT NULL,
    group_name text NOT NULL,
    data jsonb NOT NULL,
    default_branch boolean DEFAULT false NOT NULL,
    group_id bigint
);

CREATE SEQUENCE ci_daily_build_group_report_results_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_daily_build_group_report_results_id_seq OWNED BY ci_daily_build_group_report_results.id;

CREATE TABLE ci_deleted_objects (
    id bigint NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    pick_up_at timestamp with time zone DEFAULT now() NOT NULL,
    store_dir text NOT NULL,
    file text NOT NULL,
    CONSTRAINT check_5e151d6912 CHECK ((char_length(store_dir) <= 1024))
);

CREATE SEQUENCE ci_deleted_objects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_deleted_objects_id_seq OWNED BY ci_deleted_objects.id;

CREATE TABLE ci_freeze_periods (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    freeze_start text NOT NULL,
    freeze_end text NOT NULL,
    cron_timezone text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_4a7939e04e CHECK ((char_length(freeze_end) <= 998)),
    CONSTRAINT check_a92607bd2b CHECK ((char_length(freeze_start) <= 998)),
    CONSTRAINT check_b14055adc3 CHECK ((char_length(cron_timezone) <= 255))
);

CREATE SEQUENCE ci_freeze_periods_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_freeze_periods_id_seq OWNED BY ci_freeze_periods.id;

CREATE TABLE ci_group_variables (
    id integer NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    group_id integer NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    masked boolean DEFAULT false NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    environment_scope text DEFAULT '*'::text NOT NULL,
    CONSTRAINT check_dfe009485a CHECK ((char_length(environment_scope) <= 255))
);

CREATE SEQUENCE ci_group_variables_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_group_variables_id_seq OWNED BY ci_group_variables.id;

CREATE TABLE ci_instance_variables (
    id bigint NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    masked boolean DEFAULT false,
    protected boolean DEFAULT false,
    key text NOT NULL,
    encrypted_value text,
    encrypted_value_iv text,
    CONSTRAINT check_07a45a5bcb CHECK ((char_length(encrypted_value_iv) <= 255)),
    CONSTRAINT check_5aede12208 CHECK ((char_length(key) <= 255)),
    CONSTRAINT check_956afd70f1 CHECK ((char_length(encrypted_value) <= 13579))
);

CREATE SEQUENCE ci_instance_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_instance_variables_id_seq OWNED BY ci_instance_variables.id;

CREATE TABLE ci_job_artifact_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    job_artifact_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_df832b66ea CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE ci_job_artifact_states_job_artifact_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_artifact_states_job_artifact_id_seq OWNED BY ci_job_artifact_states.job_artifact_id;

CREATE TABLE ci_job_artifacts (
    project_id integer NOT NULL,
    file_type integer NOT NULL,
    size bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expire_at timestamp with time zone,
    file character varying,
    file_store integer DEFAULT 1,
    file_sha256 bytea,
    file_format smallint,
    file_location smallint,
    id bigint NOT NULL,
    job_id bigint NOT NULL,
    locked smallint DEFAULT 2,
    original_filename text,
    CONSTRAINT check_27f0f6dbab CHECK ((file_store IS NOT NULL)),
    CONSTRAINT check_85573000db CHECK ((char_length(original_filename) <= 512))
);

CREATE SEQUENCE ci_job_artifacts_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_artifacts_id_seq OWNED BY ci_job_artifacts.id;

CREATE TABLE ci_job_token_project_scope_links (
    id bigint NOT NULL,
    source_project_id bigint NOT NULL,
    target_project_id bigint NOT NULL,
    added_by_id bigint,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE ci_job_token_project_scope_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_token_project_scope_links_id_seq OWNED BY ci_job_token_project_scope_links.id;

CREATE TABLE ci_job_variables (
    id bigint NOT NULL,
    key character varying NOT NULL,
    encrypted_value text,
    encrypted_value_iv character varying,
    job_id bigint NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    source smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE ci_job_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_job_variables_id_seq OWNED BY ci_job_variables.id;

CREATE TABLE ci_minutes_additional_packs (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    expires_at date,
    number_of_minutes integer NOT NULL,
    purchase_xid text,
    CONSTRAINT check_d7ef254af0 CHECK ((char_length(purchase_xid) <= 50))
);

CREATE SEQUENCE ci_minutes_additional_packs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_minutes_additional_packs_id_seq OWNED BY ci_minutes_additional_packs.id;

CREATE TABLE ci_namespace_mirrors (
    id bigint NOT NULL,
    namespace_id integer NOT NULL,
    traversal_ids integer[] DEFAULT '{}'::integer[] NOT NULL
);

CREATE SEQUENCE ci_namespace_mirrors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_namespace_mirrors_id_seq OWNED BY ci_namespace_mirrors.id;

CREATE TABLE ci_namespace_monthly_usages (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    date date NOT NULL,
    additional_amount_available integer DEFAULT 0 NOT NULL,
    amount_used numeric(18,2) DEFAULT 0.0 NOT NULL,
    notification_level smallint DEFAULT 100 NOT NULL,
    shared_runners_duration integer DEFAULT 0 NOT NULL,
    created_at timestamp with time zone,
    CONSTRAINT ci_namespace_monthly_usages_year_month_constraint CHECK ((date = date_trunc('month'::text, (date)::timestamp with time zone)))
);

CREATE SEQUENCE ci_namespace_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_namespace_monthly_usages_id_seq OWNED BY ci_namespace_monthly_usages.id;

CREATE TABLE ci_pending_builds (
    id bigint NOT NULL,
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    instance_runners_enabled boolean DEFAULT false NOT NULL,
    namespace_id bigint,
    minutes_exceeded boolean DEFAULT false NOT NULL,
    tag_ids integer[] DEFAULT '{}'::integer[],
    namespace_traversal_ids integer[] DEFAULT '{}'::integer[]
);

CREATE SEQUENCE ci_pending_builds_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pending_builds_id_seq OWNED BY ci_pending_builds.id;

CREATE TABLE ci_pipeline_artifacts (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    pipeline_id bigint NOT NULL,
    project_id bigint NOT NULL,
    size integer NOT NULL,
    file_store smallint DEFAULT 1 NOT NULL,
    file_type smallint NOT NULL,
    file_format smallint NOT NULL,
    file text,
    expire_at timestamp with time zone,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_191b5850ec CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_abeeb71caf CHECK ((file IS NOT NULL)),
    CONSTRAINT ci_pipeline_artifacts_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE ci_pipeline_artifacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_artifacts_id_seq OWNED BY ci_pipeline_artifacts.id;

CREATE TABLE ci_pipeline_chat_data (
    id bigint NOT NULL,
    pipeline_id integer NOT NULL,
    chat_name_id integer NOT NULL,
    response_url text NOT NULL
);

CREATE SEQUENCE ci_pipeline_chat_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_chat_data_id_seq OWNED BY ci_pipeline_chat_data.id;

CREATE TABLE ci_pipeline_messages (
    id bigint NOT NULL,
    severity smallint DEFAULT 0 NOT NULL,
    pipeline_id integer NOT NULL,
    content text NOT NULL,
    CONSTRAINT check_58ca2981b2 CHECK ((char_length(content) <= 10000))
);

CREATE SEQUENCE ci_pipeline_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_messages_id_seq OWNED BY ci_pipeline_messages.id;

CREATE TABLE ci_pipeline_schedule_variables (
    id integer NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    pipeline_schedule_id integer NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone,
    variable_type smallint DEFAULT 1 NOT NULL
);

CREATE SEQUENCE ci_pipeline_schedule_variables_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_schedule_variables_id_seq OWNED BY ci_pipeline_schedule_variables.id;

CREATE TABLE ci_pipeline_schedules (
    id integer NOT NULL,
    description character varying,
    ref character varying,
    cron character varying,
    cron_timezone character varying,
    next_run_at timestamp without time zone,
    project_id integer,
    owner_id integer,
    active boolean DEFAULT true,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE ci_pipeline_schedules_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_schedules_id_seq OWNED BY ci_pipeline_schedules.id;

CREATE TABLE ci_pipeline_variables (
    id integer NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    pipeline_id integer NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL
);

CREATE SEQUENCE ci_pipeline_variables_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipeline_variables_id_seq OWNED BY ci_pipeline_variables.id;

CREATE TABLE ci_pipelines (
    id integer NOT NULL,
    ref character varying,
    sha character varying,
    before_sha character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    tag boolean DEFAULT false,
    yaml_errors text,
    committed_at timestamp without time zone,
    project_id integer,
    status character varying,
    started_at timestamp without time zone,
    finished_at timestamp without time zone,
    duration integer,
    user_id integer,
    lock_version integer DEFAULT 0,
    auto_canceled_by_id integer,
    pipeline_schedule_id integer,
    source integer,
    protected boolean,
    config_source integer,
    failure_reason integer,
    iid integer,
    merge_request_id integer,
    source_sha bytea,
    target_sha bytea,
    external_pull_request_id bigint,
    ci_ref_id bigint,
    locked smallint DEFAULT 1 NOT NULL
);

CREATE TABLE ci_pipelines_config (
    pipeline_id bigint NOT NULL,
    content text NOT NULL
);

CREATE SEQUENCE ci_pipelines_config_pipeline_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipelines_config_pipeline_id_seq OWNED BY ci_pipelines_config.pipeline_id;

CREATE SEQUENCE ci_pipelines_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_pipelines_id_seq OWNED BY ci_pipelines.id;

CREATE TABLE ci_platform_metrics (
    id bigint NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    platform_target text NOT NULL,
    count integer NOT NULL,
    CONSTRAINT check_f922abc32b CHECK ((char_length(platform_target) <= 255)),
    CONSTRAINT ci_platform_metrics_check_count_positive CHECK ((count > 0))
);

CREATE SEQUENCE ci_platform_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_platform_metrics_id_seq OWNED BY ci_platform_metrics.id;

CREATE TABLE ci_project_mirrors (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    namespace_id integer NOT NULL
);

CREATE SEQUENCE ci_project_mirrors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_project_mirrors_id_seq OWNED BY ci_project_mirrors.id;

CREATE TABLE ci_project_monthly_usages (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    date date NOT NULL,
    amount_used numeric(18,2) DEFAULT 0.0 NOT NULL,
    shared_runners_duration integer DEFAULT 0 NOT NULL,
    created_at timestamp with time zone,
    CONSTRAINT ci_project_monthly_usages_year_month_constraint CHECK ((date = date_trunc('month'::text, (date)::timestamp with time zone)))
);

CREATE SEQUENCE ci_project_monthly_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_project_monthly_usages_id_seq OWNED BY ci_project_monthly_usages.id;

CREATE TABLE ci_refs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    lock_version integer DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    ref_path text NOT NULL
);

CREATE SEQUENCE ci_refs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_refs_id_seq OWNED BY ci_refs.id;

CREATE TABLE ci_resource_groups (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    key character varying(255) NOT NULL,
    process_mode smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE ci_resource_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_resource_groups_id_seq OWNED BY ci_resource_groups.id;

CREATE TABLE ci_resources (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    resource_group_id bigint NOT NULL,
    build_id bigint
);

CREATE SEQUENCE ci_resources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_resources_id_seq OWNED BY ci_resources.id;

CREATE TABLE ci_runner_namespaces (
    id integer NOT NULL,
    runner_id integer,
    namespace_id integer
);

CREATE SEQUENCE ci_runner_namespaces_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_namespaces_id_seq OWNED BY ci_runner_namespaces.id;

CREATE TABLE ci_runner_projects (
    id integer NOT NULL,
    runner_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id integer
);

CREATE SEQUENCE ci_runner_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runner_projects_id_seq OWNED BY ci_runner_projects.id;

CREATE TABLE ci_runners (
    id integer NOT NULL,
    token character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    description character varying,
    contacted_at timestamp without time zone,
    active boolean DEFAULT true NOT NULL,
    name character varying,
    version character varying,
    revision character varying,
    platform character varying,
    architecture character varying,
    run_untagged boolean DEFAULT true NOT NULL,
    locked boolean DEFAULT false NOT NULL,
    access_level integer DEFAULT 0 NOT NULL,
    maximum_timeout integer,
    ip_address character varying,
    runner_type smallint NOT NULL,
    token_encrypted character varying,
    public_projects_minutes_cost_factor double precision DEFAULT 0.0 NOT NULL,
    private_projects_minutes_cost_factor double precision DEFAULT 1.0 NOT NULL,
    config jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE ci_runners_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_runners_id_seq OWNED BY ci_runners.id;

CREATE TABLE ci_running_builds (
    id bigint NOT NULL,
    build_id bigint NOT NULL,
    project_id bigint NOT NULL,
    runner_id bigint NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    runner_type smallint NOT NULL
);

CREATE SEQUENCE ci_running_builds_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_running_builds_id_seq OWNED BY ci_running_builds.id;

CREATE TABLE ci_sources_pipelines (
    id integer NOT NULL,
    project_id integer,
    pipeline_id integer,
    source_project_id integer,
    source_pipeline_id integer,
    source_job_id bigint
);

CREATE SEQUENCE ci_sources_pipelines_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_sources_pipelines_id_seq OWNED BY ci_sources_pipelines.id;

CREATE TABLE ci_sources_projects (
    id bigint NOT NULL,
    pipeline_id bigint NOT NULL,
    source_project_id bigint NOT NULL
);

CREATE SEQUENCE ci_sources_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_sources_projects_id_seq OWNED BY ci_sources_projects.id;

CREATE TABLE ci_stages (
    project_id integer,
    pipeline_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying,
    status integer,
    lock_version integer DEFAULT 0,
    "position" integer,
    id bigint NOT NULL
);

CREATE SEQUENCE ci_stages_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_stages_id_seq OWNED BY ci_stages.id;

CREATE TABLE ci_subscriptions_projects (
    id bigint NOT NULL,
    downstream_project_id bigint NOT NULL,
    upstream_project_id bigint NOT NULL
);

CREATE SEQUENCE ci_subscriptions_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_subscriptions_projects_id_seq OWNED BY ci_subscriptions_projects.id;

CREATE TABLE ci_trigger_requests (
    id integer NOT NULL,
    trigger_id integer NOT NULL,
    variables text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    commit_id integer
);

CREATE SEQUENCE ci_trigger_requests_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_trigger_requests_id_seq OWNED BY ci_trigger_requests.id;

CREATE TABLE ci_triggers (
    id integer NOT NULL,
    token character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id integer,
    owner_id integer NOT NULL,
    description character varying,
    ref character varying
);

CREATE SEQUENCE ci_triggers_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_triggers_id_seq OWNED BY ci_triggers.id;

CREATE TABLE ci_unit_test_failures (
    id bigint NOT NULL,
    failed_at timestamp with time zone NOT NULL,
    unit_test_id bigint NOT NULL,
    build_id bigint NOT NULL
);

CREATE SEQUENCE ci_unit_test_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_unit_test_failures_id_seq OWNED BY ci_unit_test_failures.id;

CREATE TABLE ci_unit_tests (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    key_hash text NOT NULL,
    name text NOT NULL,
    suite_name text NOT NULL,
    CONSTRAINT check_248fae1a3b CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b288215ffe CHECK ((char_length(key_hash) <= 64)),
    CONSTRAINT check_c2d57b3c49 CHECK ((char_length(suite_name) <= 255))
);

CREATE SEQUENCE ci_unit_tests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_unit_tests_id_seq OWNED BY ci_unit_tests.id;

CREATE TABLE ci_variables (
    id integer NOT NULL,
    key character varying NOT NULL,
    value text,
    encrypted_value text,
    encrypted_value_salt character varying,
    encrypted_value_iv character varying,
    project_id integer NOT NULL,
    protected boolean DEFAULT false NOT NULL,
    environment_scope character varying DEFAULT '*'::character varying NOT NULL,
    masked boolean DEFAULT false NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL
);

CREATE SEQUENCE ci_variables_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ci_variables_id_seq OWNED BY ci_variables.id;

CREATE TABLE cluster_agent_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    agent_id bigint NOT NULL,
    token_encrypted text NOT NULL,
    created_by_user_id bigint,
    description text,
    name text,
    last_used_at timestamp with time zone,
    CONSTRAINT check_0fb634d04d CHECK ((name IS NOT NULL)),
    CONSTRAINT check_2b79dbb315 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_4e4ec5070a CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_c60daed227 CHECK ((char_length(token_encrypted) <= 255))
);

CREATE SEQUENCE cluster_agent_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agent_tokens_id_seq OWNED BY cluster_agent_tokens.id;

CREATE TABLE cluster_agents (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    created_by_user_id bigint,
    CONSTRAINT check_3498369510 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE cluster_agents_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_agents_id_seq OWNED BY cluster_agents.id;

CREATE TABLE cluster_groups (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    group_id integer NOT NULL
);

CREATE SEQUENCE cluster_groups_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_groups_id_seq OWNED BY cluster_groups.id;

CREATE TABLE cluster_platforms_kubernetes (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    api_url text,
    ca_cert text,
    namespace character varying,
    username character varying,
    encrypted_password text,
    encrypted_password_iv character varying,
    encrypted_token text,
    encrypted_token_iv character varying,
    authorization_type smallint
);

CREATE SEQUENCE cluster_platforms_kubernetes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_platforms_kubernetes_id_seq OWNED BY cluster_platforms_kubernetes.id;

CREATE TABLE cluster_projects (
    id integer NOT NULL,
    project_id integer NOT NULL,
    cluster_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE cluster_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_projects_id_seq OWNED BY cluster_projects.id;

CREATE TABLE cluster_providers_aws (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    num_nodes integer NOT NULL,
    status integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    key_name character varying(255) NOT NULL,
    role_arn character varying(2048) NOT NULL,
    region character varying(255) NOT NULL,
    vpc_id character varying(255) NOT NULL,
    subnet_ids character varying(255)[] DEFAULT '{}'::character varying[] NOT NULL,
    security_group_id character varying(255) NOT NULL,
    instance_type character varying(255) NOT NULL,
    access_key_id character varying(255),
    encrypted_secret_access_key_iv character varying(255),
    encrypted_secret_access_key text,
    session_token text,
    status_reason text,
    kubernetes_version text DEFAULT '1.14'::text NOT NULL,
    CONSTRAINT check_f1f42cd85e CHECK ((char_length(kubernetes_version) <= 30))
);

CREATE SEQUENCE cluster_providers_aws_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_providers_aws_id_seq OWNED BY cluster_providers_aws.id;

CREATE TABLE cluster_providers_gcp (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    status integer,
    num_nodes integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status_reason text,
    gcp_project_id character varying NOT NULL,
    zone character varying NOT NULL,
    machine_type character varying,
    operation_id character varying,
    endpoint character varying,
    encrypted_access_token text,
    encrypted_access_token_iv character varying,
    legacy_abac boolean DEFAULT false NOT NULL,
    cloud_run boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE cluster_providers_gcp_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE cluster_providers_gcp_id_seq OWNED BY cluster_providers_gcp.id;

CREATE TABLE clusters (
    id integer NOT NULL,
    user_id integer,
    provider_type integer,
    platform_type integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT true,
    name character varying NOT NULL,
    environment_scope character varying DEFAULT '*'::character varying NOT NULL,
    cluster_type smallint DEFAULT 3 NOT NULL,
    domain character varying,
    managed boolean DEFAULT true NOT NULL,
    namespace_per_environment boolean DEFAULT true NOT NULL,
    cleanup_status smallint DEFAULT 1 NOT NULL,
    cleanup_status_reason text,
    management_project_id integer,
    helm_major_version integer DEFAULT 3 NOT NULL
);

CREATE TABLE clusters_applications_cert_managers (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    status integer NOT NULL,
    version character varying NOT NULL,
    email character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status_reason text
);

CREATE SEQUENCE clusters_applications_cert_managers_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_cert_managers_id_seq OWNED BY clusters_applications_cert_managers.id;

CREATE TABLE clusters_applications_cilium (
    id bigint NOT NULL,
    cluster_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status integer NOT NULL,
    status_reason text
);

CREATE SEQUENCE clusters_applications_cilium_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_cilium_id_seq OWNED BY clusters_applications_cilium.id;

CREATE TABLE clusters_applications_crossplane (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cluster_id bigint NOT NULL,
    status integer NOT NULL,
    version character varying(255) NOT NULL,
    stack character varying(255) NOT NULL,
    status_reason text
);

CREATE SEQUENCE clusters_applications_crossplane_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_crossplane_id_seq OWNED BY clusters_applications_crossplane.id;

CREATE TABLE clusters_applications_elastic_stacks (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cluster_id bigint NOT NULL,
    status integer NOT NULL,
    version character varying(255) NOT NULL,
    status_reason text
);

CREATE SEQUENCE clusters_applications_elastic_stacks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_elastic_stacks_id_seq OWNED BY clusters_applications_elastic_stacks.id;

CREATE TABLE clusters_applications_helm (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status integer NOT NULL,
    version character varying NOT NULL,
    status_reason text,
    encrypted_ca_key text,
    encrypted_ca_key_iv text,
    ca_cert text
);

CREATE SEQUENCE clusters_applications_helm_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_helm_id_seq OWNED BY clusters_applications_helm.id;

CREATE TABLE clusters_applications_ingress (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status integer NOT NULL,
    ingress_type integer NOT NULL,
    version character varying NOT NULL,
    cluster_ip character varying,
    status_reason text,
    external_ip character varying,
    external_hostname character varying
);

CREATE SEQUENCE clusters_applications_ingress_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_ingress_id_seq OWNED BY clusters_applications_ingress.id;

CREATE TABLE clusters_applications_jupyter (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    oauth_application_id integer,
    status integer NOT NULL,
    version character varying NOT NULL,
    hostname character varying,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status_reason text
);

CREATE SEQUENCE clusters_applications_jupyter_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_jupyter_id_seq OWNED BY clusters_applications_jupyter.id;

CREATE TABLE clusters_applications_knative (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status integer NOT NULL,
    version character varying NOT NULL,
    hostname character varying,
    status_reason text,
    external_ip character varying,
    external_hostname character varying
);

CREATE SEQUENCE clusters_applications_knative_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_knative_id_seq OWNED BY clusters_applications_knative.id;

CREATE TABLE clusters_applications_prometheus (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    status integer NOT NULL,
    version character varying NOT NULL,
    status_reason text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_alert_manager_token character varying,
    encrypted_alert_manager_token_iv character varying,
    last_update_started_at timestamp with time zone,
    healthy boolean
);

CREATE SEQUENCE clusters_applications_prometheus_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_prometheus_id_seq OWNED BY clusters_applications_prometheus.id;

CREATE TABLE clusters_applications_runners (
    id integer NOT NULL,
    cluster_id integer NOT NULL,
    runner_id integer,
    status integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    version character varying NOT NULL,
    status_reason text,
    privileged boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE clusters_applications_runners_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_applications_runners_id_seq OWNED BY clusters_applications_runners.id;

CREATE SEQUENCE clusters_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_id_seq OWNED BY clusters.id;

CREATE TABLE clusters_integration_elasticstack (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cluster_id bigint NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    chart_version text,
    CONSTRAINT check_f8d671ce04 CHECK ((char_length(chart_version) <= 10))
);

CREATE TABLE clusters_integration_prometheus (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    cluster_id bigint NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    encrypted_alert_manager_token text,
    encrypted_alert_manager_token_iv text,
    health_status smallint DEFAULT 0 NOT NULL
);

CREATE TABLE clusters_kubernetes_namespaces (
    id bigint NOT NULL,
    cluster_id integer NOT NULL,
    project_id integer,
    cluster_project_id integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_service_account_token_iv character varying,
    namespace character varying NOT NULL,
    service_account_name character varying,
    encrypted_service_account_token text,
    environment_id bigint
);

CREATE SEQUENCE clusters_kubernetes_namespaces_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE clusters_kubernetes_namespaces_id_seq OWNED BY clusters_kubernetes_namespaces.id;

CREATE TABLE commit_user_mentions (
    id bigint NOT NULL,
    note_id integer NOT NULL,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[],
    commit_id character varying,
    CONSTRAINT check_724d773807 CHECK ((commit_id IS NOT NULL))
);

CREATE SEQUENCE commit_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE commit_user_mentions_id_seq OWNED BY commit_user_mentions.id;

CREATE TABLE compliance_management_frameworks (
    id bigint NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    color text NOT NULL,
    namespace_id integer NOT NULL,
    regulated boolean DEFAULT true NOT NULL,
    pipeline_configuration_full_path text,
    CONSTRAINT check_08cd34b2c2 CHECK ((char_length(color) <= 10)),
    CONSTRAINT check_1617e0b87e CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_ab00bc2193 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e7a9972435 CHECK ((char_length(pipeline_configuration_full_path) <= 255))
);

CREATE SEQUENCE compliance_management_frameworks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE compliance_management_frameworks_id_seq OWNED BY compliance_management_frameworks.id;

CREATE TABLE container_expiration_policies (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_run_at timestamp with time zone,
    project_id bigint NOT NULL,
    name_regex character varying(255) DEFAULT '.*'::character varying,
    cadence character varying(12) DEFAULT '1d'::character varying NOT NULL,
    older_than character varying(12) DEFAULT '90d'::character varying,
    keep_n integer DEFAULT 10,
    enabled boolean DEFAULT false NOT NULL,
    name_regex_keep text,
    CONSTRAINT container_expiration_policies_name_regex_keep CHECK ((char_length(name_regex_keep) <= 255))
);

CREATE TABLE container_repositories (
    id integer NOT NULL,
    project_id integer NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    status smallint,
    expiration_policy_started_at timestamp with time zone,
    expiration_policy_cleanup_status smallint DEFAULT 0 NOT NULL,
    expiration_policy_completed_at timestamp with time zone
);

CREATE SEQUENCE container_repositories_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE container_repositories_id_seq OWNED BY container_repositories.id;

CREATE TABLE content_blocked_states (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    commit_sha bytea NOT NULL,
    blob_sha bytea NOT NULL,
    path text NOT NULL,
    container_identifier text NOT NULL,
    CONSTRAINT check_023093d38f CHECK ((char_length(container_identifier) <= 255)),
    CONSTRAINT check_1870100678 CHECK ((char_length(path) <= 2048))
);

COMMENT ON TABLE content_blocked_states IS 'JiHu-specific table';

CREATE SEQUENCE content_blocked_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE content_blocked_states_id_seq OWNED BY content_blocked_states.id;

CREATE TABLE conversational_development_index_metrics (
    id integer NOT NULL,
    leader_issues double precision NOT NULL,
    instance_issues double precision NOT NULL,
    leader_notes double precision NOT NULL,
    instance_notes double precision NOT NULL,
    leader_milestones double precision NOT NULL,
    instance_milestones double precision NOT NULL,
    leader_boards double precision NOT NULL,
    instance_boards double precision NOT NULL,
    leader_merge_requests double precision NOT NULL,
    instance_merge_requests double precision NOT NULL,
    leader_ci_pipelines double precision NOT NULL,
    instance_ci_pipelines double precision NOT NULL,
    leader_environments double precision NOT NULL,
    instance_environments double precision NOT NULL,
    leader_deployments double precision NOT NULL,
    instance_deployments double precision NOT NULL,
    leader_projects_prometheus_active double precision NOT NULL,
    instance_projects_prometheus_active double precision NOT NULL,
    leader_service_desk_issues double precision NOT NULL,
    instance_service_desk_issues double precision NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    percentage_boards double precision DEFAULT 0.0 NOT NULL,
    percentage_ci_pipelines double precision DEFAULT 0.0 NOT NULL,
    percentage_deployments double precision DEFAULT 0.0 NOT NULL,
    percentage_environments double precision DEFAULT 0.0 NOT NULL,
    percentage_issues double precision DEFAULT 0.0 NOT NULL,
    percentage_merge_requests double precision DEFAULT 0.0 NOT NULL,
    percentage_milestones double precision DEFAULT 0.0 NOT NULL,
    percentage_notes double precision DEFAULT 0.0 NOT NULL,
    percentage_projects_prometheus_active double precision DEFAULT 0.0 NOT NULL,
    percentage_service_desk_issues double precision DEFAULT 0.0 NOT NULL
);

CREATE SEQUENCE conversational_development_index_metrics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE conversational_development_index_metrics_id_seq OWNED BY conversational_development_index_metrics.id;

CREATE TABLE coverage_fuzzing_corpuses (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    package_id bigint NOT NULL,
    file_updated_at timestamp with time zone DEFAULT now() NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE coverage_fuzzing_corpuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE coverage_fuzzing_corpuses_id_seq OWNED BY coverage_fuzzing_corpuses.id;

CREATE TABLE csv_issue_imports (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE csv_issue_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE csv_issue_imports_id_seq OWNED BY csv_issue_imports.id;

CREATE TABLE custom_emoji (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    file text NOT NULL,
    external boolean DEFAULT true NOT NULL,
    creator_id bigint NOT NULL,
    CONSTRAINT check_8c586dd507 CHECK ((char_length(name) <= 36)),
    CONSTRAINT check_dd5d60f1fb CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE custom_emoji_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE custom_emoji_id_seq OWNED BY custom_emoji.id;

CREATE TABLE customer_relations_contacts (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    organization_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    phone text,
    first_name text NOT NULL,
    last_name text NOT NULL,
    email text,
    description text,
    CONSTRAINT check_1195f4c929 CHECK ((char_length(first_name) <= 255)),
    CONSTRAINT check_40c70da037 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_cd2d67c484 CHECK ((char_length(last_name) <= 255)),
    CONSTRAINT check_f4b7f78c89 CHECK ((char_length(phone) <= 32)),
    CONSTRAINT check_fc0adabf60 CHECK ((char_length(email) <= 255))
);

CREATE SEQUENCE customer_relations_contacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE customer_relations_contacts_id_seq OWNED BY customer_relations_contacts.id;

CREATE TABLE customer_relations_organizations (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    default_rate numeric(18,2),
    name text NOT NULL,
    description text,
    CONSTRAINT check_2ba9ef1c4c CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e476b6058e CHECK ((char_length(description) <= 1024))
);

CREATE SEQUENCE customer_relations_organizations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE customer_relations_organizations_id_seq OWNED BY customer_relations_organizations.id;

CREATE TABLE dast_profile_schedules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_profile_id bigint NOT NULL,
    user_id bigint,
    next_run_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT true NOT NULL,
    cron text NOT NULL,
    cadence jsonb DEFAULT '{}'::jsonb NOT NULL,
    timezone text NOT NULL,
    starts_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_86531ea73f CHECK ((char_length(cron) <= 255)),
    CONSTRAINT check_be4d1c3af1 CHECK ((char_length(timezone) <= 255))
);

COMMENT ON TABLE dast_profile_schedules IS '{"owner":"group::dynamic analysis","description":"Scheduling for scans using DAST Profiles"}';

CREATE SEQUENCE dast_profile_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_profile_schedules_id_seq OWNED BY dast_profile_schedules.id;

CREATE TABLE dast_profiles (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_site_profile_id bigint NOT NULL,
    dast_scanner_profile_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    branch_name text,
    CONSTRAINT check_5fcf73bf61 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_6c9d775949 CHECK ((char_length(branch_name) <= 255)),
    CONSTRAINT check_c34e505c24 CHECK ((char_length(description) <= 255))
);

COMMENT ON TABLE dast_profiles IS '{"owner":"group::dynamic analysis","description":"Profile used to run a DAST on-demand scan"}';

CREATE SEQUENCE dast_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_profiles_id_seq OWNED BY dast_profiles.id;

CREATE TABLE dast_profiles_pipelines (
    dast_profile_id bigint NOT NULL,
    ci_pipeline_id bigint NOT NULL
);

COMMENT ON TABLE dast_profiles_pipelines IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Profiles and CI Pipelines"}';

CREATE TABLE dast_scanner_profiles (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    spider_timeout smallint,
    target_timeout smallint,
    name text NOT NULL,
    scan_type smallint DEFAULT 1 NOT NULL,
    use_ajax_spider boolean DEFAULT false NOT NULL,
    show_debug_messages boolean DEFAULT false NOT NULL,
    CONSTRAINT check_568568fabf CHECK ((char_length(name) <= 255))
);

CREATE TABLE dast_scanner_profiles_builds (
    dast_scanner_profile_id bigint NOT NULL,
    ci_build_id bigint NOT NULL
);

COMMENT ON TABLE dast_scanner_profiles_builds IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Scanner Profiles and CI Builds"}';

CREATE SEQUENCE dast_scanner_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_scanner_profiles_id_seq OWNED BY dast_scanner_profiles.id;

CREATE TABLE dast_site_profile_secret_variables (
    id bigint NOT NULL,
    dast_site_profile_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    variable_type smallint DEFAULT 1 NOT NULL,
    key text NOT NULL,
    encrypted_value bytea NOT NULL,
    encrypted_value_iv bytea NOT NULL,
    CONSTRAINT check_236213f179 CHECK ((length(encrypted_value) <= 13352)),
    CONSTRAINT check_8cbef204b2 CHECK ((char_length(key) <= 255)),
    CONSTRAINT check_b49080abbf CHECK ((length(encrypted_value_iv) <= 17))
);

COMMENT ON TABLE dast_site_profile_secret_variables IS '{"owner":"group::dynamic analysis","description":"Secret variables used in DAST on-demand scans"}';

CREATE SEQUENCE dast_site_profile_secret_variables_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_profile_secret_variables_id_seq OWNED BY dast_site_profile_secret_variables.id;

CREATE TABLE dast_site_profiles (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    dast_site_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    excluded_urls text[] DEFAULT '{}'::text[] NOT NULL,
    auth_enabled boolean DEFAULT false NOT NULL,
    auth_url text,
    auth_username_field text,
    auth_password_field text,
    auth_username text,
    target_type smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_5203110fee CHECK ((char_length(auth_username_field) <= 255)),
    CONSTRAINT check_6cfab17b48 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_c329dffdba CHECK ((char_length(auth_password_field) <= 255)),
    CONSTRAINT check_d446f7047b CHECK ((char_length(auth_url) <= 1024)),
    CONSTRAINT check_f22f18002a CHECK ((char_length(auth_username) <= 255))
);

CREATE TABLE dast_site_profiles_builds (
    dast_site_profile_id bigint NOT NULL,
    ci_build_id bigint NOT NULL
);

COMMENT ON TABLE dast_site_profiles_builds IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Site Profiles and CI Builds"}';

CREATE SEQUENCE dast_site_profiles_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_profiles_id_seq OWNED BY dast_site_profiles.id;

CREATE TABLE dast_site_profiles_pipelines (
    dast_site_profile_id bigint NOT NULL,
    ci_pipeline_id bigint NOT NULL
);

COMMENT ON TABLE dast_site_profiles_pipelines IS '{"owner":"group::dynamic analysis","description":"Join table between DAST Site Profiles and CI Pipelines"}';

CREATE TABLE dast_site_tokens (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    expired_at timestamp with time zone,
    token text NOT NULL,
    url text NOT NULL,
    CONSTRAINT check_02a6bf20a7 CHECK ((char_length(token) <= 255)),
    CONSTRAINT check_69ab8622a6 CHECK ((char_length(url) <= 255))
);

CREATE SEQUENCE dast_site_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_tokens_id_seq OWNED BY dast_site_tokens.id;

CREATE TABLE dast_site_validations (
    id bigint NOT NULL,
    dast_site_token_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    validation_started_at timestamp with time zone,
    validation_passed_at timestamp with time zone,
    validation_failed_at timestamp with time zone,
    validation_last_retried_at timestamp with time zone,
    validation_strategy smallint NOT NULL,
    url_base text NOT NULL,
    url_path text NOT NULL,
    state text DEFAULT 'pending'::text NOT NULL,
    CONSTRAINT check_13b34efe4b CHECK ((char_length(url_path) <= 255)),
    CONSTRAINT check_283be72e9b CHECK ((char_length(state) <= 255)),
    CONSTRAINT check_cd3b538210 CHECK ((char_length(url_base) <= 255))
);

CREATE SEQUENCE dast_site_validations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_site_validations_id_seq OWNED BY dast_site_validations.id;

CREATE TABLE dast_sites (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    url text NOT NULL,
    dast_site_validation_id bigint,
    CONSTRAINT check_46df8b449c CHECK ((char_length(url) <= 255))
);

CREATE SEQUENCE dast_sites_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dast_sites_id_seq OWNED BY dast_sites.id;

CREATE TABLE dependency_proxy_blobs (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    file text NOT NULL,
    file_name character varying NOT NULL,
    file_store integer,
    group_id integer NOT NULL,
    size bigint,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    read_at timestamp with time zone DEFAULT now() NOT NULL
);

CREATE SEQUENCE dependency_proxy_blobs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_blobs_id_seq OWNED BY dependency_proxy_blobs.id;

CREATE TABLE dependency_proxy_group_settings (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    group_id integer NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE dependency_proxy_group_settings_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_group_settings_id_seq OWNED BY dependency_proxy_group_settings.id;

CREATE TABLE dependency_proxy_image_ttl_group_policies (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    ttl integer DEFAULT 90,
    enabled boolean DEFAULT false NOT NULL
);

CREATE TABLE dependency_proxy_manifests (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    size bigint,
    file_store smallint,
    file_name text NOT NULL,
    file text NOT NULL,
    digest text NOT NULL,
    content_type text,
    status smallint DEFAULT 0 NOT NULL,
    read_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_079b293a7b CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_167a9a8a91 CHECK ((char_length(content_type) <= 255)),
    CONSTRAINT check_c579e3f586 CHECK ((char_length(file_name) <= 255)),
    CONSTRAINT check_f5d9996bf1 CHECK ((char_length(digest) <= 255))
);

CREATE SEQUENCE dependency_proxy_manifests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dependency_proxy_manifests_id_seq OWNED BY dependency_proxy_manifests.id;

CREATE TABLE deploy_keys_projects (
    id integer NOT NULL,
    deploy_key_id integer NOT NULL,
    project_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    can_push boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE deploy_keys_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deploy_keys_projects_id_seq OWNED BY deploy_keys_projects.id;

CREATE TABLE deploy_tokens (
    id integer NOT NULL,
    revoked boolean DEFAULT false,
    read_repository boolean DEFAULT false NOT NULL,
    read_registry boolean DEFAULT false NOT NULL,
    expires_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    token character varying,
    username character varying,
    token_encrypted character varying(255),
    deploy_token_type smallint DEFAULT 2 NOT NULL,
    write_registry boolean DEFAULT false NOT NULL,
    read_package_registry boolean DEFAULT false NOT NULL,
    write_package_registry boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE deploy_tokens_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deploy_tokens_id_seq OWNED BY deploy_tokens.id;

CREATE TABLE deployment_approvals (
    id bigint NOT NULL,
    deployment_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint NOT NULL
);

CREATE SEQUENCE deployment_approvals_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deployment_approvals_id_seq OWNED BY deployment_approvals.id;

CREATE TABLE deployment_clusters (
    deployment_id integer NOT NULL,
    cluster_id integer NOT NULL,
    kubernetes_namespace character varying(255)
);

CREATE TABLE deployment_merge_requests (
    deployment_id integer NOT NULL,
    merge_request_id integer NOT NULL,
    environment_id integer
);

CREATE TABLE deployments (
    id integer NOT NULL,
    iid integer NOT NULL,
    project_id integer NOT NULL,
    environment_id integer NOT NULL,
    ref character varying NOT NULL,
    tag boolean NOT NULL,
    sha character varying NOT NULL,
    user_id integer,
    deployable_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    on_stop character varying,
    finished_at timestamp with time zone,
    status smallint NOT NULL,
    cluster_id integer,
    deployable_id bigint,
    archived boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE deployments_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE deployments_id_seq OWNED BY deployments.id;

CREATE TABLE description_versions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id integer,
    merge_request_id integer,
    epic_id integer,
    description text,
    deleted_at timestamp with time zone
);

CREATE SEQUENCE description_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE description_versions_id_seq OWNED BY description_versions.id;

CREATE TABLE design_management_designs (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    issue_id integer,
    filename character varying NOT NULL,
    relative_position integer,
    iid integer,
    CONSTRAINT check_07155e2715 CHECK ((char_length((filename)::text) <= 255)),
    CONSTRAINT check_cfb92df01a CHECK ((iid IS NOT NULL))
);

CREATE SEQUENCE design_management_designs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_designs_id_seq OWNED BY design_management_designs.id;

CREATE TABLE design_management_designs_versions (
    design_id bigint NOT NULL,
    version_id bigint NOT NULL,
    event smallint DEFAULT 0 NOT NULL,
    image_v432x230 character varying(255),
    id bigint NOT NULL
);

CREATE SEQUENCE design_management_designs_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_designs_versions_id_seq OWNED BY design_management_designs_versions.id;

CREATE TABLE design_management_versions (
    id bigint NOT NULL,
    sha bytea NOT NULL,
    issue_id bigint,
    created_at timestamp with time zone NOT NULL,
    author_id integer
);

CREATE SEQUENCE design_management_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_management_versions_id_seq OWNED BY design_management_versions.id;

CREATE TABLE design_user_mentions (
    id bigint NOT NULL,
    design_id integer NOT NULL,
    note_id integer NOT NULL,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE design_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE design_user_mentions_id_seq OWNED BY design_user_mentions.id;

CREATE TABLE detached_partitions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    drop_after timestamp with time zone NOT NULL,
    table_name text NOT NULL,
    CONSTRAINT check_aecee24ba3 CHECK ((char_length(table_name) <= 63))
);

CREATE SEQUENCE detached_partitions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE detached_partitions_id_seq OWNED BY detached_partitions.id;

CREATE TABLE diff_note_positions (
    id bigint NOT NULL,
    note_id bigint NOT NULL,
    old_line integer,
    new_line integer,
    diff_content_type smallint NOT NULL,
    diff_type smallint NOT NULL,
    line_code character varying(255) NOT NULL,
    base_sha bytea NOT NULL,
    start_sha bytea NOT NULL,
    head_sha bytea NOT NULL,
    old_path text NOT NULL,
    new_path text NOT NULL
);

CREATE SEQUENCE diff_note_positions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE diff_note_positions_id_seq OWNED BY diff_note_positions.id;

CREATE TABLE dora_daily_metrics (
    id bigint NOT NULL,
    environment_id bigint NOT NULL,
    date date NOT NULL,
    deployment_frequency integer,
    lead_time_for_changes_in_seconds integer,
    CONSTRAINT dora_daily_metrics_deployment_frequency_positive CHECK ((deployment_frequency >= 0)),
    CONSTRAINT dora_daily_metrics_lead_time_for_changes_in_seconds_positive CHECK ((lead_time_for_changes_in_seconds >= 0))
);

CREATE SEQUENCE dora_daily_metrics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE dora_daily_metrics_id_seq OWNED BY dora_daily_metrics.id;

CREATE TABLE draft_notes (
    id bigint NOT NULL,
    merge_request_id integer NOT NULL,
    author_id integer NOT NULL,
    resolve_discussion boolean DEFAULT false NOT NULL,
    discussion_id character varying,
    note text NOT NULL,
    "position" text,
    original_position text,
    change_position text,
    commit_id bytea,
    line_code text,
    CONSTRAINT check_c497a94a0e CHECK ((char_length(line_code) <= 255))
);

CREATE SEQUENCE draft_notes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE draft_notes_id_seq OWNED BY draft_notes.id;

CREATE TABLE elastic_index_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    number_of_replicas smallint DEFAULT 1 NOT NULL,
    number_of_shards smallint DEFAULT 5 NOT NULL,
    alias_name text NOT NULL,
    CONSTRAINT check_c30005c325 CHECK ((char_length(alias_name) <= 255))
);

CREATE SEQUENCE elastic_index_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_index_settings_id_seq OWNED BY elastic_index_settings.id;

CREATE TABLE elastic_reindexing_slices (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    elastic_reindexing_subtask_id bigint NOT NULL,
    elastic_slice smallint DEFAULT 0 NOT NULL,
    elastic_max_slice smallint DEFAULT 0 NOT NULL,
    retry_attempt smallint DEFAULT 0 NOT NULL,
    elastic_task text,
    CONSTRAINT check_ca30e1396e CHECK ((char_length(elastic_task) <= 255))
);

CREATE SEQUENCE elastic_reindexing_slices_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_slices_id_seq OWNED BY elastic_reindexing_slices.id;

CREATE TABLE elastic_reindexing_subtasks (
    id bigint NOT NULL,
    elastic_reindexing_task_id bigint NOT NULL,
    alias_name text NOT NULL,
    index_name_from text NOT NULL,
    index_name_to text NOT NULL,
    elastic_task text,
    documents_count_target integer,
    documents_count integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_4910adc798 CHECK ((char_length(elastic_task) <= 255)),
    CONSTRAINT check_88f56216a4 CHECK ((char_length(alias_name) <= 255)),
    CONSTRAINT check_a1fbd9faa9 CHECK ((char_length(index_name_from) <= 255)),
    CONSTRAINT check_f456494bd8 CHECK ((char_length(index_name_to) <= 255))
);

CREATE SEQUENCE elastic_reindexing_subtasks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_subtasks_id_seq OWNED BY elastic_reindexing_subtasks.id;

CREATE TABLE elastic_reindexing_tasks (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    in_progress boolean DEFAULT true NOT NULL,
    error_message text,
    delete_original_index_at timestamp with time zone,
    max_slices_running smallint DEFAULT 60 NOT NULL,
    slice_multiplier smallint DEFAULT 2 NOT NULL,
    CONSTRAINT check_7f64acda8e CHECK ((char_length(error_message) <= 255))
);

CREATE SEQUENCE elastic_reindexing_tasks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE elastic_reindexing_tasks_id_seq OWNED BY elastic_reindexing_tasks.id;

CREATE TABLE elasticsearch_indexed_namespaces (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id integer NOT NULL
);

CREATE TABLE elasticsearch_indexed_projects (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL
);

CREATE TABLE emails (
    id integer NOT NULL,
    user_id integer NOT NULL,
    email character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    confirmation_token character varying,
    confirmed_at timestamp with time zone,
    confirmation_sent_at timestamp with time zone
);

CREATE SEQUENCE emails_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE emails_id_seq OWNED BY emails.id;

CREATE TABLE environments (
    id integer NOT NULL,
    project_id integer NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    external_url character varying,
    environment_type character varying,
    state character varying DEFAULT 'available'::character varying NOT NULL,
    slug character varying NOT NULL,
    auto_stop_at timestamp with time zone,
    auto_delete_at timestamp with time zone,
    tier smallint
);

CREATE SEQUENCE environments_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE environments_id_seq OWNED BY environments.id;

CREATE TABLE epic_issues (
    id integer NOT NULL,
    epic_id integer NOT NULL,
    issue_id integer NOT NULL,
    relative_position integer
);

CREATE SEQUENCE epic_issues_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epic_issues_id_seq OWNED BY epic_issues.id;

CREATE TABLE epic_metrics (
    id integer NOT NULL,
    epic_id integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE epic_metrics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epic_metrics_id_seq OWNED BY epic_metrics.id;

CREATE TABLE epic_user_mentions (
    id bigint NOT NULL,
    epic_id integer NOT NULL,
    note_id integer,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE epic_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epic_user_mentions_id_seq OWNED BY epic_user_mentions.id;

CREATE TABLE epics (
    id integer NOT NULL,
    group_id integer NOT NULL,
    author_id integer NOT NULL,
    assignee_id integer,
    iid integer NOT NULL,
    cached_markdown_version integer,
    updated_by_id integer,
    last_edited_by_id integer,
    lock_version integer DEFAULT 0,
    start_date date,
    end_date date,
    last_edited_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    title character varying NOT NULL,
    title_html character varying NOT NULL,
    description text,
    description_html text,
    start_date_sourcing_milestone_id integer,
    due_date_sourcing_milestone_id integer,
    start_date_fixed date,
    due_date_fixed date,
    start_date_is_fixed boolean,
    due_date_is_fixed boolean,
    closed_by_id integer,
    closed_at timestamp without time zone,
    parent_id integer,
    relative_position integer,
    start_date_sourcing_epic_id integer,
    due_date_sourcing_epic_id integer,
    state_id smallint DEFAULT 1,
    external_key character varying(255),
    confidential boolean DEFAULT false NOT NULL,
    color text DEFAULT '#1068bf'::text,
    CONSTRAINT check_57ee003890 CHECK ((state_id IS NOT NULL)),
    CONSTRAINT check_ca608c40b3 CHECK ((char_length(color) <= 7))
);

CREATE SEQUENCE epics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE epics_id_seq OWNED BY epics.id;

CREATE TABLE error_tracking_client_keys (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    active boolean DEFAULT true NOT NULL,
    public_key text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_840b719790 CHECK ((char_length(public_key) <= 255))
);

CREATE SEQUENCE error_tracking_client_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_client_keys_id_seq OWNED BY error_tracking_client_keys.id;

CREATE TABLE error_tracking_error_events (
    id bigint NOT NULL,
    error_id bigint NOT NULL,
    description text NOT NULL,
    environment text,
    level text,
    occurred_at timestamp with time zone NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_92ecc3077b CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_c67d5b8007 CHECK ((char_length(level) <= 255)),
    CONSTRAINT check_f4b52474ad CHECK ((char_length(environment) <= 255))
);

CREATE SEQUENCE error_tracking_error_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_error_events_id_seq OWNED BY error_tracking_error_events.id;

CREATE TABLE error_tracking_errors (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    description text NOT NULL,
    actor text NOT NULL,
    first_seen_at timestamp with time zone DEFAULT now() NOT NULL,
    last_seen_at timestamp with time zone DEFAULT now() NOT NULL,
    platform text,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    events_count bigint DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_18a758e537 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b5cb4d3888 CHECK ((char_length(actor) <= 255)),
    CONSTRAINT check_c739788b12 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_fe99886883 CHECK ((char_length(platform) <= 255))
);

CREATE SEQUENCE error_tracking_errors_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE error_tracking_errors_id_seq OWNED BY error_tracking_errors.id;

CREATE TABLE events (
    project_id integer,
    author_id integer NOT NULL,
    target_id integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    action smallint NOT NULL,
    target_type character varying,
    group_id bigint,
    fingerprint bytea,
    id bigint NOT NULL,
    CONSTRAINT check_97e06e05ad CHECK ((octet_length(fingerprint) <= 128))
);

CREATE SEQUENCE events_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE events_id_seq OWNED BY events.id;

CREATE TABLE evidences (
    id bigint NOT NULL,
    release_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    summary_sha bytea,
    summary jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE evidences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE evidences_id_seq OWNED BY evidences.id;

CREATE TABLE experiment_subjects (
    id bigint NOT NULL,
    experiment_id bigint NOT NULL,
    user_id bigint,
    project_id bigint,
    variant smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    converted_at timestamp with time zone,
    context jsonb DEFAULT '{}'::jsonb NOT NULL,
    namespace_id bigint,
    CONSTRAINT check_f6411bc4b5 CHECK ((num_nonnulls(user_id, namespace_id, project_id) = 1))
);

CREATE SEQUENCE experiment_subjects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE experiment_subjects_id_seq OWNED BY experiment_subjects.id;

CREATE TABLE experiment_users (
    id bigint NOT NULL,
    experiment_id bigint NOT NULL,
    user_id bigint NOT NULL,
    group_type smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    converted_at timestamp with time zone,
    context jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE experiment_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE experiment_users_id_seq OWNED BY experiment_users.id;

CREATE TABLE experiments (
    id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_e2dda25ed0 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE experiments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE experiments_id_seq OWNED BY experiments.id;

CREATE TABLE external_approval_rules (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_url text NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_1c64b53ea5 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b634ca168d CHECK ((char_length(external_url) <= 255))
);

CREATE SEQUENCE external_approval_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_approval_rules_id_seq OWNED BY external_approval_rules.id;

CREATE TABLE external_approval_rules_protected_branches (
    id bigint NOT NULL,
    external_approval_rule_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL
);

CREATE SEQUENCE external_approval_rules_protected_branches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_approval_rules_protected_branches_id_seq OWNED BY external_approval_rules_protected_branches.id;

CREATE TABLE external_pull_requests (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    pull_request_iid integer NOT NULL,
    status smallint NOT NULL,
    source_branch character varying(255) NOT NULL,
    target_branch character varying(255) NOT NULL,
    source_repository character varying(255) NOT NULL,
    target_repository character varying(255) NOT NULL,
    source_sha bytea NOT NULL,
    target_sha bytea NOT NULL
);

CREATE SEQUENCE external_pull_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_pull_requests_id_seq OWNED BY external_pull_requests.id;

CREATE TABLE external_status_checks (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    external_url text NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_7e3b9eb41a CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_ae0dec3f61 CHECK ((char_length(external_url) <= 255))
);

CREATE SEQUENCE external_status_checks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_status_checks_id_seq OWNED BY external_status_checks.id;

CREATE TABLE external_status_checks_protected_branches (
    id bigint NOT NULL,
    external_status_check_id bigint NOT NULL,
    protected_branch_id bigint NOT NULL
);

CREATE SEQUENCE external_status_checks_protected_branches_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE external_status_checks_protected_branches_id_seq OWNED BY external_status_checks_protected_branches.id;

CREATE TABLE feature_gates (
    id integer NOT NULL,
    feature_key character varying NOT NULL,
    key character varying NOT NULL,
    value character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE feature_gates_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE feature_gates_id_seq OWNED BY feature_gates.id;

CREATE TABLE features (
    id integer NOT NULL,
    key character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE features_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE features_id_seq OWNED BY features.id;

CREATE TABLE fork_network_members (
    id integer NOT NULL,
    fork_network_id integer NOT NULL,
    project_id integer NOT NULL,
    forked_from_project_id integer
);

CREATE SEQUENCE fork_network_members_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE fork_network_members_id_seq OWNED BY fork_network_members.id;

CREATE TABLE fork_networks (
    id integer NOT NULL,
    root_project_id integer,
    deleted_root_project_name character varying
);

CREATE SEQUENCE fork_networks_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE fork_networks_id_seq OWNED BY fork_networks.id;

CREATE TABLE geo_cache_invalidation_events (
    id bigint NOT NULL,
    key character varying NOT NULL
);

CREATE SEQUENCE geo_cache_invalidation_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_cache_invalidation_events_id_seq OWNED BY geo_cache_invalidation_events.id;

CREATE TABLE geo_container_repository_updated_events (
    id bigint NOT NULL,
    container_repository_id integer NOT NULL
);

CREATE SEQUENCE geo_container_repository_updated_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_container_repository_updated_events_id_seq OWNED BY geo_container_repository_updated_events.id;

CREATE TABLE geo_event_log (
    id bigint NOT NULL,
    created_at timestamp without time zone NOT NULL,
    repository_updated_event_id bigint,
    repository_deleted_event_id bigint,
    repository_renamed_event_id bigint,
    repositories_changed_event_id bigint,
    repository_created_event_id bigint,
    hashed_storage_migrated_event_id bigint,
    lfs_object_deleted_event_id bigint,
    hashed_storage_attachments_event_id bigint,
    job_artifact_deleted_event_id bigint,
    reset_checksum_event_id bigint,
    cache_invalidation_event_id bigint,
    container_repository_updated_event_id bigint,
    geo_event_id integer
);

CREATE SEQUENCE geo_event_log_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_event_log_id_seq OWNED BY geo_event_log.id;

CREATE TABLE geo_events (
    id bigint NOT NULL,
    replicable_name character varying(255) NOT NULL,
    event_name character varying(255) NOT NULL,
    payload jsonb DEFAULT '{}'::jsonb NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE geo_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_events_id_seq OWNED BY geo_events.id;

CREATE TABLE geo_hashed_storage_attachments_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    old_attachments_path text NOT NULL,
    new_attachments_path text NOT NULL
);

CREATE SEQUENCE geo_hashed_storage_attachments_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_hashed_storage_attachments_events_id_seq OWNED BY geo_hashed_storage_attachments_events.id;

CREATE TABLE geo_hashed_storage_migrated_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    repository_storage_name text NOT NULL,
    old_disk_path text NOT NULL,
    new_disk_path text NOT NULL,
    old_wiki_disk_path text NOT NULL,
    new_wiki_disk_path text NOT NULL,
    old_storage_version smallint,
    new_storage_version smallint NOT NULL,
    old_design_disk_path text,
    new_design_disk_path text
);

CREATE SEQUENCE geo_hashed_storage_migrated_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_hashed_storage_migrated_events_id_seq OWNED BY geo_hashed_storage_migrated_events.id;

CREATE TABLE geo_job_artifact_deleted_events (
    id bigint NOT NULL,
    file_path character varying NOT NULL,
    job_artifact_id bigint NOT NULL
);

CREATE SEQUENCE geo_job_artifact_deleted_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_job_artifact_deleted_events_id_seq OWNED BY geo_job_artifact_deleted_events.id;

CREATE TABLE geo_lfs_object_deleted_events (
    id bigint NOT NULL,
    lfs_object_id integer NOT NULL,
    oid character varying NOT NULL,
    file_path character varying NOT NULL
);

CREATE SEQUENCE geo_lfs_object_deleted_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_lfs_object_deleted_events_id_seq OWNED BY geo_lfs_object_deleted_events.id;

CREATE TABLE geo_node_namespace_links (
    id integer NOT NULL,
    geo_node_id integer NOT NULL,
    namespace_id integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE geo_node_namespace_links_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_node_namespace_links_id_seq OWNED BY geo_node_namespace_links.id;

CREATE TABLE geo_node_statuses (
    id integer NOT NULL,
    geo_node_id integer NOT NULL,
    db_replication_lag_seconds integer,
    repositories_synced_count integer,
    repositories_failed_count integer,
    lfs_objects_count integer,
    lfs_objects_synced_count integer,
    lfs_objects_failed_count integer,
    last_event_id integer,
    last_event_date timestamp without time zone,
    cursor_last_event_id integer,
    cursor_last_event_date timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    last_successful_status_check_at timestamp without time zone,
    status_message character varying,
    replication_slots_count integer,
    replication_slots_used_count integer,
    replication_slots_max_retained_wal_bytes bigint,
    wikis_synced_count integer,
    wikis_failed_count integer,
    job_artifacts_count integer,
    job_artifacts_synced_count integer,
    job_artifacts_failed_count integer,
    version character varying,
    revision character varying,
    repositories_verified_count integer,
    repositories_verification_failed_count integer,
    wikis_verified_count integer,
    wikis_verification_failed_count integer,
    lfs_objects_synced_missing_on_primary_count integer,
    job_artifacts_synced_missing_on_primary_count integer,
    repositories_checksummed_count integer,
    repositories_checksum_failed_count integer,
    repositories_checksum_mismatch_count integer,
    wikis_checksummed_count integer,
    wikis_checksum_failed_count integer,
    wikis_checksum_mismatch_count integer,
    storage_configuration_digest bytea,
    repositories_retrying_verification_count integer,
    wikis_retrying_verification_count integer,
    projects_count integer,
    container_repositories_count integer,
    container_repositories_synced_count integer,
    container_repositories_failed_count integer,
    container_repositories_registry_count integer,
    design_repositories_count integer,
    design_repositories_synced_count integer,
    design_repositories_failed_count integer,
    design_repositories_registry_count integer,
    status jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE geo_node_statuses_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_node_statuses_id_seq OWNED BY geo_node_statuses.id;

CREATE TABLE geo_nodes (
    id integer NOT NULL,
    "primary" boolean DEFAULT false NOT NULL,
    oauth_application_id integer,
    enabled boolean DEFAULT true NOT NULL,
    access_key character varying,
    encrypted_secret_access_key character varying,
    encrypted_secret_access_key_iv character varying,
    clone_url_prefix character varying,
    files_max_capacity integer DEFAULT 10 NOT NULL,
    repos_max_capacity integer DEFAULT 25 NOT NULL,
    url character varying NOT NULL,
    selective_sync_type character varying,
    selective_sync_shards text,
    verification_max_capacity integer DEFAULT 100 NOT NULL,
    minimum_reverification_interval integer DEFAULT 7 NOT NULL,
    internal_url character varying,
    name character varying NOT NULL,
    container_repositories_max_capacity integer DEFAULT 10 NOT NULL,
    sync_object_storage boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone,
    updated_at timestamp with time zone
);

CREATE SEQUENCE geo_nodes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_nodes_id_seq OWNED BY geo_nodes.id;

CREATE TABLE geo_repositories_changed_events (
    id bigint NOT NULL,
    geo_node_id integer NOT NULL
);

CREATE SEQUENCE geo_repositories_changed_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_repositories_changed_events_id_seq OWNED BY geo_repositories_changed_events.id;

CREATE TABLE geo_repository_created_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    repository_storage_name text NOT NULL,
    repo_path text NOT NULL,
    wiki_path text,
    project_name text NOT NULL
);

CREATE SEQUENCE geo_repository_created_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_repository_created_events_id_seq OWNED BY geo_repository_created_events.id;

CREATE TABLE geo_repository_deleted_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    repository_storage_name text NOT NULL,
    deleted_path text NOT NULL,
    deleted_wiki_path text,
    deleted_project_name text NOT NULL
);

CREATE SEQUENCE geo_repository_deleted_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_repository_deleted_events_id_seq OWNED BY geo_repository_deleted_events.id;

CREATE TABLE geo_repository_renamed_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    repository_storage_name text NOT NULL,
    old_path_with_namespace text NOT NULL,
    new_path_with_namespace text NOT NULL,
    old_wiki_path_with_namespace text NOT NULL,
    new_wiki_path_with_namespace text NOT NULL,
    old_path text NOT NULL,
    new_path text NOT NULL
);

CREATE SEQUENCE geo_repository_renamed_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_repository_renamed_events_id_seq OWNED BY geo_repository_renamed_events.id;

CREATE TABLE geo_repository_updated_events (
    id bigint NOT NULL,
    branches_affected integer NOT NULL,
    tags_affected integer NOT NULL,
    project_id integer NOT NULL,
    source smallint NOT NULL,
    new_branch boolean DEFAULT false NOT NULL,
    remove_branch boolean DEFAULT false NOT NULL,
    ref text
);

CREATE SEQUENCE geo_repository_updated_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_repository_updated_events_id_seq OWNED BY geo_repository_updated_events.id;

CREATE TABLE geo_reset_checksum_events (
    id bigint NOT NULL,
    project_id integer NOT NULL
);

CREATE SEQUENCE geo_reset_checksum_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE geo_reset_checksum_events_id_seq OWNED BY geo_reset_checksum_events.id;

CREATE TABLE gitlab_subscription_histories (
    id bigint NOT NULL,
    gitlab_subscription_created_at timestamp with time zone,
    gitlab_subscription_updated_at timestamp with time zone,
    start_date date,
    end_date date,
    trial_ends_on date,
    namespace_id integer,
    hosted_plan_id integer,
    max_seats_used integer,
    seats integer,
    trial boolean,
    change_type smallint,
    gitlab_subscription_id bigint NOT NULL,
    created_at timestamp with time zone,
    trial_starts_on date,
    auto_renew boolean,
    trial_extension_type smallint
);

CREATE SEQUENCE gitlab_subscription_histories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gitlab_subscription_histories_id_seq OWNED BY gitlab_subscription_histories.id;

CREATE TABLE gitlab_subscriptions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    end_date date,
    trial_ends_on date,
    namespace_id integer,
    hosted_plan_id integer,
    max_seats_used integer DEFAULT 0,
    seats integer DEFAULT 0,
    trial boolean DEFAULT false,
    trial_starts_on date,
    auto_renew boolean,
    seats_in_use integer DEFAULT 0 NOT NULL,
    seats_owed integer DEFAULT 0 NOT NULL,
    trial_extension_type smallint,
    CONSTRAINT check_77fea3f0e7 CHECK ((namespace_id IS NOT NULL))
);

CREATE SEQUENCE gitlab_subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gitlab_subscriptions_id_seq OWNED BY gitlab_subscriptions.id;

CREATE TABLE gpg_key_subkeys (
    id integer NOT NULL,
    gpg_key_id integer NOT NULL,
    keyid bytea,
    fingerprint bytea
);

CREATE SEQUENCE gpg_key_subkeys_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_key_subkeys_id_seq OWNED BY gpg_key_subkeys.id;

CREATE TABLE gpg_keys (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id integer,
    primary_keyid bytea,
    fingerprint bytea,
    key text
);

CREATE SEQUENCE gpg_keys_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_keys_id_seq OWNED BY gpg_keys.id;

CREATE TABLE gpg_signatures (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer,
    gpg_key_id integer,
    commit_sha bytea,
    gpg_key_primary_keyid bytea,
    gpg_key_user_name text,
    gpg_key_user_email text,
    verification_status smallint DEFAULT 0 NOT NULL,
    gpg_key_subkey_id integer
);

CREATE SEQUENCE gpg_signatures_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE gpg_signatures_id_seq OWNED BY gpg_signatures.id;

CREATE TABLE grafana_integrations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_token character varying(255) NOT NULL,
    encrypted_token_iv character varying(255) NOT NULL,
    grafana_url character varying(1024) NOT NULL,
    enabled boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE grafana_integrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE grafana_integrations_id_seq OWNED BY grafana_integrations.id;

CREATE TABLE group_custom_attributes (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id integer NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE group_custom_attributes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_custom_attributes_id_seq OWNED BY group_custom_attributes.id;

CREATE TABLE group_deletion_schedules (
    group_id bigint NOT NULL,
    user_id bigint NOT NULL,
    marked_for_deletion_on date NOT NULL
);

CREATE TABLE group_deploy_keys (
    id bigint NOT NULL,
    user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    last_used_at timestamp with time zone,
    expires_at timestamp with time zone,
    key text NOT NULL,
    title text,
    fingerprint text NOT NULL,
    fingerprint_sha256 bytea,
    CONSTRAINT check_cc0365908d CHECK ((char_length(title) <= 255)),
    CONSTRAINT check_e4526dcf91 CHECK ((char_length(fingerprint) <= 255)),
    CONSTRAINT check_f58fa0a0f7 CHECK ((char_length(key) <= 4096))
);

CREATE TABLE group_deploy_keys_groups (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    group_deploy_key_id bigint NOT NULL,
    can_push boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE group_deploy_keys_groups_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_deploy_keys_groups_id_seq OWNED BY group_deploy_keys_groups.id;

CREATE SEQUENCE group_deploy_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_deploy_keys_id_seq OWNED BY group_deploy_keys.id;

CREATE TABLE group_deploy_tokens (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    deploy_token_id bigint NOT NULL
);

CREATE SEQUENCE group_deploy_tokens_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_deploy_tokens_id_seq OWNED BY group_deploy_tokens.id;

CREATE TABLE group_group_links (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    shared_group_id bigint NOT NULL,
    shared_with_group_id bigint NOT NULL,
    expires_at date,
    group_access smallint DEFAULT 30 NOT NULL
);

CREATE SEQUENCE group_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_group_links_id_seq OWNED BY group_group_links.id;

CREATE TABLE group_import_states (
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid text,
    last_error text,
    user_id bigint,
    CONSTRAINT check_87b58f6b30 CHECK ((char_length(last_error) <= 255)),
    CONSTRAINT check_96558fff96 CHECK ((char_length(jid) <= 100))
);

CREATE SEQUENCE group_import_states_group_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_import_states_group_id_seq OWNED BY group_import_states.group_id;

CREATE TABLE group_merge_request_approval_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    allow_author_approval boolean DEFAULT false NOT NULL,
    allow_committer_approval boolean DEFAULT false NOT NULL,
    allow_overrides_to_approver_list_per_merge_request boolean DEFAULT false NOT NULL,
    retain_approvals_on_push boolean DEFAULT false NOT NULL,
    require_password_to_approve boolean DEFAULT false NOT NULL
);

CREATE TABLE group_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    CONSTRAINT group_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT group_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE group_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE group_repository_storage_moves_id_seq OWNED BY group_repository_storage_moves.id;

CREATE TABLE group_wiki_repositories (
    shard_id bigint NOT NULL,
    group_id bigint NOT NULL,
    disk_path text NOT NULL,
    CONSTRAINT check_07f1c81806 CHECK ((char_length(disk_path) <= 80))
);

CREATE TABLE historical_data (
    id integer NOT NULL,
    date date,
    active_user_count integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    recorded_at timestamp with time zone,
    CONSTRAINT check_640e8cf66c CHECK ((recorded_at IS NOT NULL))
);

CREATE SEQUENCE historical_data_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE historical_data_id_seq OWNED BY historical_data.id;

CREATE TABLE identities (
    id integer NOT NULL,
    extern_uid character varying,
    provider character varying,
    user_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    saml_provider_id integer,
    secondary_extern_uid character varying
);

CREATE SEQUENCE identities_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE identities_id_seq OWNED BY identities.id;

CREATE TABLE import_export_uploads (
    id integer NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer,
    import_file text,
    export_file text,
    group_id bigint,
    remote_import_url text,
    CONSTRAINT check_58f0d37481 CHECK ((char_length(remote_import_url) <= 512))
);

CREATE SEQUENCE import_export_uploads_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_export_uploads_id_seq OWNED BY import_export_uploads.id;

CREATE TABLE import_failures (
    id bigint NOT NULL,
    relation_index integer,
    project_id bigint,
    created_at timestamp with time zone NOT NULL,
    relation_key character varying(64),
    exception_class character varying(128),
    correlation_id_value character varying(128),
    exception_message character varying(255),
    retry_count integer,
    group_id integer,
    source character varying(128)
);

CREATE SEQUENCE import_failures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE import_failures_id_seq OWNED BY import_failures.id;

CREATE TABLE in_product_marketing_emails (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    cta_clicked_at timestamp with time zone,
    track smallint NOT NULL,
    series smallint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE in_product_marketing_emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE in_product_marketing_emails_id_seq OWNED BY in_product_marketing_emails.id;

CREATE TABLE incident_management_escalation_policies (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    name text NOT NULL,
    description text,
    CONSTRAINT check_510b2a5258 CHECK ((char_length(description) <= 160)),
    CONSTRAINT check_9a26365850 CHECK ((char_length(name) <= 72))
);

CREATE SEQUENCE incident_management_escalation_policies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_escalation_policies_id_seq OWNED BY incident_management_escalation_policies.id;

CREATE TABLE incident_management_escalation_rules (
    id bigint NOT NULL,
    policy_id bigint NOT NULL,
    oncall_schedule_id bigint,
    status smallint NOT NULL,
    elapsed_time_seconds integer NOT NULL,
    is_removed boolean DEFAULT false NOT NULL,
    user_id bigint,
    CONSTRAINT escalation_rules_one_of_oncall_schedule_or_user CHECK ((num_nonnulls(oncall_schedule_id, user_id) = 1))
);

CREATE SEQUENCE incident_management_escalation_rules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_escalation_rules_id_seq OWNED BY incident_management_escalation_rules.id;

CREATE TABLE incident_management_issuable_escalation_statuses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id bigint NOT NULL,
    policy_id bigint,
    escalations_started_at timestamp with time zone,
    resolved_at timestamp with time zone,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE incident_management_issuable_escalation_statuses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_issuable_escalation_statuses_id_seq OWNED BY incident_management_issuable_escalation_statuses.id;

CREATE TABLE incident_management_oncall_participants (
    id bigint NOT NULL,
    oncall_rotation_id bigint NOT NULL,
    user_id bigint NOT NULL,
    color_palette smallint NOT NULL,
    color_weight smallint NOT NULL,
    is_removed boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE incident_management_oncall_participants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_participants_id_seq OWNED BY incident_management_oncall_participants.id;

CREATE TABLE incident_management_oncall_rotations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    oncall_schedule_id bigint NOT NULL,
    length integer NOT NULL,
    length_unit smallint NOT NULL,
    starts_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    active_period_start time without time zone,
    active_period_end time without time zone,
    ends_at timestamp with time zone,
    CONSTRAINT check_5209fb5d02 CHECK ((char_length(name) <= 200))
);

CREATE SEQUENCE incident_management_oncall_rotations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_rotations_id_seq OWNED BY incident_management_oncall_rotations.id;

CREATE TABLE incident_management_oncall_schedules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    iid integer NOT NULL,
    name text NOT NULL,
    description text,
    timezone text,
    CONSTRAINT check_7ed1fd5aa7 CHECK ((char_length(description) <= 1000)),
    CONSTRAINT check_cc77cbb103 CHECK ((char_length(timezone) <= 100)),
    CONSTRAINT check_e6ef43a664 CHECK ((char_length(name) <= 200))
);

CREATE SEQUENCE incident_management_oncall_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_schedules_id_seq OWNED BY incident_management_oncall_schedules.id;

CREATE TABLE incident_management_oncall_shifts (
    id bigint NOT NULL,
    rotation_id bigint NOT NULL,
    participant_id bigint NOT NULL,
    starts_at timestamp with time zone NOT NULL,
    ends_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE incident_management_oncall_shifts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_oncall_shifts_id_seq OWNED BY incident_management_oncall_shifts.id;

CREATE TABLE incident_management_timeline_events (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    occurred_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    author_id bigint,
    issue_id bigint NOT NULL,
    updated_by_user_id bigint,
    promoted_from_note_id bigint,
    cached_markdown_version integer,
    editable boolean DEFAULT false NOT NULL,
    note text NOT NULL,
    note_html text NOT NULL,
    action text NOT NULL,
    CONSTRAINT check_18fd072206 CHECK ((char_length(action) <= 128)),
    CONSTRAINT check_3875ed0aac CHECK ((char_length(note) <= 10000)),
    CONSTRAINT check_94a235d6a4 CHECK ((char_length(note_html) <= 10000))
);

CREATE SEQUENCE incident_management_timeline_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE incident_management_timeline_events_id_seq OWNED BY incident_management_timeline_events.id;

CREATE TABLE index_statuses (
    id integer NOT NULL,
    project_id integer NOT NULL,
    indexed_at timestamp without time zone,
    note text,
    last_commit character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    last_wiki_commit bytea,
    wiki_indexed_at timestamp with time zone
);

CREATE SEQUENCE index_statuses_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE index_statuses_id_seq OWNED BY index_statuses.id;

CREATE TABLE insights (
    id integer NOT NULL,
    namespace_id integer NOT NULL,
    project_id integer NOT NULL
);

CREATE SEQUENCE insights_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE insights_id_seq OWNED BY insights.id;

CREATE TABLE integrations (
    id integer NOT NULL,
    type character varying,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    active boolean DEFAULT false NOT NULL,
    properties text,
    template boolean DEFAULT false,
    push_events boolean DEFAULT true,
    issues_events boolean DEFAULT true,
    merge_requests_events boolean DEFAULT true,
    tag_push_events boolean DEFAULT true,
    note_events boolean DEFAULT true NOT NULL,
    category character varying DEFAULT 'common'::character varying NOT NULL,
    wiki_page_events boolean DEFAULT true,
    pipeline_events boolean DEFAULT false NOT NULL,
    confidential_issues_events boolean DEFAULT true NOT NULL,
    commit_events boolean DEFAULT true NOT NULL,
    job_events boolean DEFAULT false NOT NULL,
    confidential_note_events boolean DEFAULT true,
    deployment_events boolean DEFAULT false NOT NULL,
    comment_on_event_enabled boolean DEFAULT true NOT NULL,
    instance boolean DEFAULT false NOT NULL,
    comment_detail smallint,
    inherit_from_id bigint,
    alert_events boolean,
    group_id bigint,
    vulnerability_events boolean DEFAULT false NOT NULL,
    type_new text,
    archive_trace_events boolean DEFAULT false NOT NULL,
    CONSTRAINT check_a948a0aa7e CHECK ((char_length(type_new) <= 255))
);

CREATE SEQUENCE integrations_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE integrations_id_seq OWNED BY integrations.id;

CREATE TABLE internal_ids (
    id bigint NOT NULL,
    project_id integer,
    usage integer NOT NULL,
    last_value integer NOT NULL,
    namespace_id integer
);

CREATE SEQUENCE internal_ids_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE internal_ids_id_seq OWNED BY internal_ids.id;

CREATE TABLE ip_restrictions (
    id bigint NOT NULL,
    group_id integer NOT NULL,
    range character varying NOT NULL
);

CREATE SEQUENCE ip_restrictions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ip_restrictions_id_seq OWNED BY ip_restrictions.id;

CREATE TABLE issuable_metric_images (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    file text NOT NULL,
    url text,
    CONSTRAINT check_5b3011e234 CHECK ((char_length(url) <= 255)),
    CONSTRAINT check_7ed527062f CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE issuable_metric_images_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_metric_images_id_seq OWNED BY issuable_metric_images.id;

CREATE TABLE issuable_severities (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    severity smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE issuable_severities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_severities_id_seq OWNED BY issuable_severities.id;

CREATE TABLE issuable_slas (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    due_at timestamp with time zone NOT NULL,
    label_applied boolean DEFAULT false NOT NULL,
    issuable_closed boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE issuable_slas_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issuable_slas_id_seq OWNED BY issuable_slas.id;

CREATE TABLE issue_assignees (
    user_id integer NOT NULL,
    issue_id integer NOT NULL
);

CREATE TABLE issue_customer_relations_contacts (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    contact_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE issue_customer_relations_contacts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_customer_relations_contacts_id_seq OWNED BY issue_customer_relations_contacts.id;

CREATE TABLE issue_email_participants (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    email text NOT NULL,
    CONSTRAINT check_2c321d408d CHECK ((char_length(email) <= 255))
);

CREATE SEQUENCE issue_email_participants_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_email_participants_id_seq OWNED BY issue_email_participants.id;

CREATE TABLE issue_emails (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    email_message_id text NOT NULL,
    CONSTRAINT check_5abf3e6aea CHECK ((char_length(email_message_id) <= 1000))
);

CREATE SEQUENCE issue_emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_emails_id_seq OWNED BY issue_emails.id;

CREATE TABLE issue_links (
    id integer NOT NULL,
    source_id integer NOT NULL,
    target_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    link_type smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE issue_links_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_links_id_seq OWNED BY issue_links.id;

CREATE TABLE issue_metrics (
    id integer NOT NULL,
    issue_id integer NOT NULL,
    first_mentioned_in_commit_at timestamp without time zone,
    first_associated_with_milestone_at timestamp without time zone,
    first_added_to_board_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE issue_metrics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_metrics_id_seq OWNED BY issue_metrics.id;

CREATE TABLE issue_tracker_data (
    id bigint NOT NULL,
    service_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_project_url character varying,
    encrypted_project_url_iv character varying,
    encrypted_issues_url character varying,
    encrypted_issues_url_iv character varying,
    encrypted_new_issue_url character varying,
    encrypted_new_issue_url_iv character varying
);

CREATE SEQUENCE issue_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_tracker_data_id_seq OWNED BY issue_tracker_data.id;

CREATE TABLE issue_user_mentions (
    id bigint NOT NULL,
    issue_id integer NOT NULL,
    note_id integer,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE issue_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issue_user_mentions_id_seq OWNED BY issue_user_mentions.id;

CREATE TABLE issues (
    id integer NOT NULL,
    title character varying,
    author_id integer,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    description text,
    milestone_id integer,
    iid integer,
    updated_by_id integer,
    confidential boolean DEFAULT false NOT NULL,
    due_date date,
    moved_to_id integer,
    lock_version integer DEFAULT 0,
    title_html text,
    description_html text,
    time_estimate integer,
    relative_position integer,
    cached_markdown_version integer,
    last_edited_at timestamp without time zone,
    last_edited_by_id integer,
    discussion_locked boolean,
    closed_at timestamp with time zone,
    closed_by_id integer,
    state_id smallint DEFAULT 1 NOT NULL,
    service_desk_reply_to character varying,
    weight integer,
    duplicated_to_id integer,
    promoted_to_epic_id integer,
    health_status smallint,
    sprint_id bigint,
    external_key character varying(255),
    blocking_issues_count integer DEFAULT 0 NOT NULL,
    issue_type smallint DEFAULT 0 NOT NULL,
    upvotes_count integer DEFAULT 0 NOT NULL,
    work_item_type_id bigint
);

CREATE SEQUENCE issues_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE issues_id_seq OWNED BY issues.id;

CREATE TABLE issues_prometheus_alert_events (
    issue_id bigint NOT NULL,
    prometheus_alert_event_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE issues_self_managed_prometheus_alert_events (
    issue_id bigint NOT NULL,
    self_managed_prometheus_alert_event_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE TABLE iterations_cadences (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    last_run_date date,
    duration_in_weeks integer,
    iterations_in_advance integer,
    active boolean DEFAULT true NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    title text NOT NULL,
    roll_over boolean DEFAULT false NOT NULL,
    description text,
    CONSTRAINT check_5c5d2b44bd CHECK ((char_length(description) <= 5000)),
    CONSTRAINT check_fedff82d3b CHECK ((char_length(title) <= 255))
);

CREATE SEQUENCE iterations_cadences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE iterations_cadences_id_seq OWNED BY iterations_cadences.id;

CREATE TABLE jira_connect_installations (
    id bigint NOT NULL,
    client_key character varying,
    encrypted_shared_secret character varying,
    encrypted_shared_secret_iv character varying,
    base_url character varying,
    instance_url text,
    CONSTRAINT check_4c6abed669 CHECK ((char_length(instance_url) <= 255))
);

CREATE SEQUENCE jira_connect_installations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_connect_installations_id_seq OWNED BY jira_connect_installations.id;

CREATE TABLE jira_connect_subscriptions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    jira_connect_installation_id bigint NOT NULL,
    namespace_id integer NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE jira_connect_subscriptions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_connect_subscriptions_id_seq OWNED BY jira_connect_subscriptions.id;

CREATE TABLE jira_imports (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint,
    label_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    finished_at timestamp with time zone,
    jira_project_xid bigint NOT NULL,
    total_issue_count integer DEFAULT 0 NOT NULL,
    imported_issues_count integer DEFAULT 0 NOT NULL,
    failed_to_import_count integer DEFAULT 0 NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid character varying(255),
    jira_project_key character varying(255) NOT NULL,
    jira_project_name character varying(255) NOT NULL,
    scheduled_at timestamp with time zone,
    error_message text,
    CONSTRAINT check_9ed451c5b1 CHECK ((char_length(error_message) <= 1000))
);

CREATE SEQUENCE jira_imports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_imports_id_seq OWNED BY jira_imports.id;

CREATE TABLE jira_tracker_data (
    id bigint NOT NULL,
    service_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_url character varying,
    encrypted_url_iv character varying,
    encrypted_api_url character varying,
    encrypted_api_url_iv character varying,
    encrypted_username character varying,
    encrypted_username_iv character varying,
    encrypted_password character varying,
    encrypted_password_iv character varying,
    jira_issue_transition_id character varying,
    project_key text,
    issues_enabled boolean DEFAULT false NOT NULL,
    deployment_type smallint DEFAULT 0 NOT NULL,
    vulnerabilities_issuetype text,
    vulnerabilities_enabled boolean DEFAULT false NOT NULL,
    jira_issue_transition_automatic boolean DEFAULT false NOT NULL,
    CONSTRAINT check_0bf84b76e9 CHECK ((char_length(vulnerabilities_issuetype) <= 255)),
    CONSTRAINT check_214cf6a48b CHECK ((char_length(project_key) <= 255))
);

CREATE SEQUENCE jira_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE jira_tracker_data_id_seq OWNED BY jira_tracker_data.id;

CREATE TABLE keys (
    id integer NOT NULL,
    user_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    key text,
    title character varying,
    type character varying,
    fingerprint character varying,
    public boolean DEFAULT false NOT NULL,
    last_used_at timestamp without time zone,
    fingerprint_sha256 bytea,
    expires_at timestamp with time zone,
    expiry_notification_delivered_at timestamp with time zone,
    before_expiry_notification_delivered_at timestamp with time zone
);

CREATE SEQUENCE keys_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE keys_id_seq OWNED BY keys.id;

CREATE TABLE label_links (
    id integer NOT NULL,
    label_id integer,
    target_id integer,
    target_type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE label_links_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE label_links_id_seq OWNED BY label_links.id;

CREATE TABLE label_priorities (
    id integer NOT NULL,
    project_id integer NOT NULL,
    label_id integer NOT NULL,
    priority integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE label_priorities_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE label_priorities_id_seq OWNED BY label_priorities.id;

CREATE TABLE labels (
    id integer NOT NULL,
    title character varying,
    color character varying,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    template boolean DEFAULT false,
    description character varying,
    description_html text,
    type character varying,
    group_id integer,
    cached_markdown_version integer
);

CREATE SEQUENCE labels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE labels_id_seq OWNED BY labels.id;

CREATE TABLE ldap_group_links (
    id integer NOT NULL,
    cn character varying,
    group_access integer NOT NULL,
    group_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    provider character varying,
    filter character varying
);

CREATE SEQUENCE ldap_group_links_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE ldap_group_links_id_seq OWNED BY ldap_group_links.id;

CREATE TABLE lfs_file_locks (
    id integer NOT NULL,
    project_id integer NOT NULL,
    user_id integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    path character varying(511)
);

CREATE SEQUENCE lfs_file_locks_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_file_locks_id_seq OWNED BY lfs_file_locks.id;

CREATE TABLE lfs_object_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    lfs_object_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_efe45a8ab3 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE lfs_object_states_lfs_object_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_object_states_lfs_object_id_seq OWNED BY lfs_object_states.lfs_object_id;

CREATE TABLE lfs_objects (
    id integer NOT NULL,
    oid character varying NOT NULL,
    size bigint NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    file character varying,
    file_store integer DEFAULT 1,
    CONSTRAINT check_eecfc5717d CHECK ((file_store IS NOT NULL))
);

CREATE SEQUENCE lfs_objects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_objects_id_seq OWNED BY lfs_objects.id;

CREATE TABLE lfs_objects_projects (
    id integer NOT NULL,
    lfs_object_id integer NOT NULL,
    project_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    repository_type smallint
);

CREATE SEQUENCE lfs_objects_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lfs_objects_projects_id_seq OWNED BY lfs_objects_projects.id;

CREATE TABLE licenses (
    id integer NOT NULL,
    data text NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    cloud boolean DEFAULT false,
    last_synced_at timestamp with time zone
);

CREATE SEQUENCE licenses_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE licenses_id_seq OWNED BY licenses.id;

CREATE TABLE list_user_preferences (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    list_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    collapsed boolean
);

CREATE SEQUENCE list_user_preferences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE list_user_preferences_id_seq OWNED BY list_user_preferences.id;

CREATE TABLE lists (
    id integer NOT NULL,
    board_id integer NOT NULL,
    label_id integer,
    list_type integer DEFAULT 1 NOT NULL,
    "position" integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    milestone_id integer,
    user_id integer,
    max_issue_count integer DEFAULT 0 NOT NULL,
    max_issue_weight integer DEFAULT 0 NOT NULL,
    limit_metric character varying(20),
    iteration_id bigint
);

CREATE SEQUENCE lists_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE lists_id_seq OWNED BY lists.id;

CREATE TABLE member_tasks (
    id bigint NOT NULL,
    member_id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    tasks smallint[] DEFAULT '{}'::smallint[] NOT NULL
);

CREATE SEQUENCE member_tasks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE member_tasks_id_seq OWNED BY member_tasks.id;

CREATE TABLE members (
    id integer NOT NULL,
    access_level integer NOT NULL,
    source_id integer NOT NULL,
    source_type character varying NOT NULL,
    user_id integer,
    notification_level integer NOT NULL,
    type character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    created_by_id integer,
    invite_email character varying,
    invite_token character varying,
    invite_accepted_at timestamp without time zone,
    requested_at timestamp without time zone,
    expires_at date,
    ldap boolean DEFAULT false NOT NULL,
    override boolean DEFAULT false NOT NULL,
    invite_email_success boolean DEFAULT true NOT NULL,
    state smallint DEFAULT 0
);

CREATE SEQUENCE members_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE members_id_seq OWNED BY members.id;

CREATE TABLE merge_request_assignees (
    id integer NOT NULL,
    user_id integer NOT NULL,
    merge_request_id integer NOT NULL,
    created_at timestamp with time zone,
    state smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE merge_request_assignees_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_assignees_id_seq OWNED BY merge_request_assignees.id;

CREATE TABLE merge_request_blocks (
    id bigint NOT NULL,
    blocking_merge_request_id integer NOT NULL,
    blocked_merge_request_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE merge_request_blocks_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_blocks_id_seq OWNED BY merge_request_blocks.id;

CREATE TABLE merge_request_cleanup_schedules (
    merge_request_id bigint NOT NULL,
    scheduled_at timestamp with time zone NOT NULL,
    completed_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    failed_count integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE merge_request_cleanup_schedules_merge_request_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_cleanup_schedules_merge_request_id_seq OWNED BY merge_request_cleanup_schedules.merge_request_id;

CREATE TABLE merge_request_context_commit_diff_files (
    merge_request_context_commit_id bigint NOT NULL,
    sha bytea NOT NULL,
    relative_order integer NOT NULL,
    a_mode character varying(255) NOT NULL,
    b_mode character varying(255) NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    too_large boolean NOT NULL,
    "binary" boolean,
    new_path text NOT NULL,
    old_path text NOT NULL,
    diff text
);

CREATE TABLE merge_request_context_commits (
    id bigint NOT NULL,
    merge_request_id bigint,
    authored_date timestamp with time zone,
    committed_date timestamp with time zone,
    sha bytea NOT NULL,
    relative_order integer NOT NULL,
    author_name text,
    author_email text,
    committer_name text,
    committer_email text,
    message text,
    trailers jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE merge_request_context_commits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_context_commits_id_seq OWNED BY merge_request_context_commits.id;

CREATE TABLE merge_request_diff_commit_users (
    id bigint NOT NULL,
    name text,
    email text,
    CONSTRAINT check_147358fc42 CHECK ((char_length(name) <= 512)),
    CONSTRAINT check_f5fa206cf7 CHECK ((char_length(email) <= 512)),
    CONSTRAINT merge_request_diff_commit_users_name_or_email_existence CHECK (((COALESCE(name, ''::text) <> ''::text) OR (COALESCE(email, ''::text) <> ''::text)))
);

CREATE SEQUENCE merge_request_diff_commit_users_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diff_commit_users_id_seq OWNED BY merge_request_diff_commit_users.id;

CREATE TABLE merge_request_diff_commits (
    authored_date timestamp with time zone,
    committed_date timestamp with time zone,
    merge_request_diff_id integer NOT NULL,
    relative_order integer NOT NULL,
    sha bytea NOT NULL,
    message text,
    trailers jsonb DEFAULT '{}'::jsonb NOT NULL,
    commit_author_id bigint,
    committer_id bigint
);

CREATE TABLE merge_request_diff_details (
    merge_request_diff_id bigint NOT NULL,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    CONSTRAINT check_81429e3622 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE merge_request_diff_details_merge_request_diff_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diff_details_merge_request_diff_id_seq OWNED BY merge_request_diff_details.merge_request_diff_id;

CREATE TABLE merge_request_diff_files (
    merge_request_diff_id integer NOT NULL,
    relative_order integer NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    too_large boolean NOT NULL,
    a_mode character varying NOT NULL,
    b_mode character varying NOT NULL,
    new_path text NOT NULL,
    old_path text NOT NULL,
    diff text,
    "binary" boolean,
    external_diff_offset integer,
    external_diff_size integer
);

CREATE TABLE merge_request_diffs (
    id integer NOT NULL,
    state character varying,
    merge_request_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    base_commit_sha character varying,
    real_size character varying,
    head_commit_sha character varying,
    start_commit_sha character varying,
    commits_count integer,
    external_diff character varying,
    external_diff_store integer DEFAULT 1,
    stored_externally boolean,
    files_count smallint,
    sorted boolean DEFAULT false NOT NULL,
    diff_type smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_93ee616ac9 CHECK ((external_diff_store IS NOT NULL))
);

CREATE SEQUENCE merge_request_diffs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_diffs_id_seq OWNED BY merge_request_diffs.id;

CREATE TABLE merge_request_metrics (
    id integer NOT NULL,
    merge_request_id integer NOT NULL,
    latest_build_started_at timestamp without time zone,
    latest_build_finished_at timestamp without time zone,
    first_deployed_to_production_at timestamp without time zone,
    merged_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    pipeline_id integer,
    merged_by_id integer,
    latest_closed_by_id integer,
    latest_closed_at timestamp with time zone,
    first_comment_at timestamp with time zone,
    first_commit_at timestamp with time zone,
    last_commit_at timestamp with time zone,
    diff_size integer,
    modified_paths_size integer,
    commits_count integer,
    first_approved_at timestamp with time zone,
    first_reassigned_at timestamp with time zone,
    added_lines integer,
    removed_lines integer,
    target_project_id integer,
    CONSTRAINT check_e03d0900bf CHECK ((target_project_id IS NOT NULL))
);

CREATE SEQUENCE merge_request_metrics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_metrics_id_seq OWNED BY merge_request_metrics.id;

CREATE TABLE merge_request_reviewers (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    state smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE merge_request_reviewers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_reviewers_id_seq OWNED BY merge_request_reviewers.id;

CREATE TABLE merge_request_user_mentions (
    id bigint NOT NULL,
    merge_request_id integer NOT NULL,
    note_id integer,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE merge_request_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_request_user_mentions_id_seq OWNED BY merge_request_user_mentions.id;

CREATE TABLE merge_requests (
    id integer NOT NULL,
    target_branch character varying NOT NULL,
    source_branch character varying NOT NULL,
    source_project_id integer,
    author_id integer,
    assignee_id integer,
    title character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    milestone_id integer,
    merge_status character varying DEFAULT 'unchecked'::character varying NOT NULL,
    target_project_id integer NOT NULL,
    iid integer,
    description text,
    updated_by_id integer,
    merge_error text,
    merge_params text,
    merge_when_pipeline_succeeds boolean DEFAULT false NOT NULL,
    merge_user_id integer,
    merge_commit_sha character varying,
    rebase_commit_sha character varying,
    in_progress_merge_commit_sha character varying,
    lock_version integer DEFAULT 0,
    title_html text,
    description_html text,
    time_estimate integer,
    cached_markdown_version integer,
    last_edited_at timestamp without time zone,
    last_edited_by_id integer,
    head_pipeline_id integer,
    merge_jid character varying,
    discussion_locked boolean,
    latest_merge_request_diff_id integer,
    allow_maintainer_to_push boolean DEFAULT true,
    squash boolean DEFAULT false NOT NULL,
    state_id smallint DEFAULT 1 NOT NULL,
    approvals_before_merge integer,
    rebase_jid character varying,
    squash_commit_sha bytea,
    sprint_id bigint,
    merge_ref_sha bytea,
    draft boolean DEFAULT false NOT NULL
);

CREATE TABLE merge_requests_closing_issues (
    id integer NOT NULL,
    merge_request_id integer NOT NULL,
    issue_id integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE merge_requests_closing_issues_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_closing_issues_id_seq OWNED BY merge_requests_closing_issues.id;

CREATE TABLE merge_requests_compliance_violations (
    id bigint NOT NULL,
    violating_user_id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    reason smallint NOT NULL
);

CREATE SEQUENCE merge_requests_compliance_violations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_compliance_violations_id_seq OWNED BY merge_requests_compliance_violations.id;

CREATE SEQUENCE merge_requests_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_requests_id_seq OWNED BY merge_requests.id;

CREATE TABLE merge_trains (
    id bigint NOT NULL,
    merge_request_id integer NOT NULL,
    user_id integer NOT NULL,
    pipeline_id integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    target_project_id integer NOT NULL,
    target_branch text NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    merged_at timestamp with time zone,
    duration integer
);

CREATE SEQUENCE merge_trains_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE merge_trains_id_seq OWNED BY merge_trains.id;

CREATE TABLE metrics_dashboard_annotations (
    id bigint NOT NULL,
    starting_at timestamp with time zone NOT NULL,
    ending_at timestamp with time zone,
    environment_id bigint,
    cluster_id bigint,
    dashboard_path character varying(255) NOT NULL,
    panel_xid character varying(255),
    description text NOT NULL
);

CREATE SEQUENCE metrics_dashboard_annotations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE metrics_dashboard_annotations_id_seq OWNED BY metrics_dashboard_annotations.id;

CREATE TABLE metrics_users_starred_dashboards (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    user_id bigint NOT NULL,
    dashboard_path text NOT NULL,
    CONSTRAINT check_79a84a0f57 CHECK ((char_length(dashboard_path) <= 255))
);

CREATE SEQUENCE metrics_users_starred_dashboards_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE metrics_users_starred_dashboards_id_seq OWNED BY metrics_users_starred_dashboards.id;

CREATE TABLE milestone_releases (
    milestone_id bigint NOT NULL,
    release_id bigint NOT NULL
);

CREATE TABLE milestones (
    id integer NOT NULL,
    title character varying NOT NULL,
    project_id integer,
    description text,
    due_date date,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    state character varying,
    iid integer,
    title_html text,
    description_html text,
    start_date date,
    cached_markdown_version integer,
    group_id integer
);

CREATE SEQUENCE milestones_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE milestones_id_seq OWNED BY milestones.id;

CREATE TABLE namespace_admin_notes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id bigint NOT NULL,
    note text,
    CONSTRAINT check_e9d2e71b5d CHECK ((char_length(note) <= 1000))
);

CREATE SEQUENCE namespace_admin_notes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_admin_notes_id_seq OWNED BY namespace_admin_notes.id;

CREATE TABLE namespace_aggregation_schedules (
    namespace_id integer NOT NULL
);

CREATE SEQUENCE namespace_aggregation_schedules_namespace_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_aggregation_schedules_namespace_id_seq OWNED BY namespace_aggregation_schedules.namespace_id;

CREATE TABLE namespace_limits (
    additional_purchased_storage_size bigint DEFAULT 0 NOT NULL,
    additional_purchased_storage_ends_on date,
    namespace_id integer NOT NULL,
    temporary_storage_increase_ends_on date
);

CREATE TABLE namespace_package_settings (
    namespace_id bigint NOT NULL,
    maven_duplicates_allowed boolean DEFAULT true NOT NULL,
    maven_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    generic_duplicates_allowed boolean DEFAULT true NOT NULL,
    generic_duplicate_exception_regex text DEFAULT ''::text NOT NULL,
    CONSTRAINT check_31340211b1 CHECK ((char_length(generic_duplicate_exception_regex) <= 255)),
    CONSTRAINT check_d63274b2b6 CHECK ((char_length(maven_duplicate_exception_regex) <= 255))
);

CREATE TABLE namespace_root_storage_statistics (
    namespace_id integer NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    lfs_objects_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint DEFAULT 0 NOT NULL,
    build_artifacts_size bigint DEFAULT 0 NOT NULL,
    storage_size bigint DEFAULT 0 NOT NULL,
    packages_size bigint DEFAULT 0 NOT NULL,
    snippets_size bigint DEFAULT 0 NOT NULL,
    pipeline_artifacts_size bigint DEFAULT 0 NOT NULL,
    uploads_size bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE namespace_root_storage_statistics_namespace_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_root_storage_statistics_namespace_id_seq OWNED BY namespace_root_storage_statistics.namespace_id;

CREATE TABLE namespace_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    namespace_id integer NOT NULL,
    prevent_forking_outside_group boolean DEFAULT false NOT NULL,
    allow_mfa_for_subgroups boolean DEFAULT true NOT NULL,
    default_branch_name text,
    repository_read_only boolean DEFAULT false NOT NULL,
    delayed_project_removal boolean,
    lock_delayed_project_removal boolean DEFAULT false NOT NULL,
    resource_access_token_creation_allowed boolean DEFAULT true NOT NULL,
    prevent_sharing_groups_outside_hierarchy boolean DEFAULT false NOT NULL,
    new_user_signups_cap integer,
    setup_for_company boolean,
    jobs_to_be_done smallint,
    CONSTRAINT check_0ba93c78c7 CHECK ((char_length(default_branch_name) <= 255))
);

CREATE TABLE namespace_statistics (
    id integer NOT NULL,
    namespace_id integer NOT NULL,
    shared_runners_seconds integer DEFAULT 0 NOT NULL,
    shared_runners_seconds_last_reset timestamp without time zone,
    storage_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE namespace_statistics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespace_statistics_id_seq OWNED BY namespace_statistics.id;

CREATE TABLE namespaces (
    id integer NOT NULL,
    name character varying NOT NULL,
    path character varying NOT NULL,
    owner_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    type character varying DEFAULT 'User'::character varying NOT NULL,
    description character varying DEFAULT ''::character varying NOT NULL,
    avatar character varying,
    share_with_group_lock boolean DEFAULT false,
    visibility_level integer DEFAULT 20 NOT NULL,
    request_access_enabled boolean DEFAULT true NOT NULL,
    description_html text,
    lfs_enabled boolean,
    parent_id integer,
    require_two_factor_authentication boolean DEFAULT false NOT NULL,
    two_factor_grace_period integer DEFAULT 48 NOT NULL,
    cached_markdown_version integer,
    runners_token character varying,
    project_creation_level integer,
    runners_token_encrypted character varying,
    auto_devops_enabled boolean,
    custom_project_templates_group_id integer,
    file_template_project_id integer,
    ldap_sync_error character varying,
    ldap_sync_last_successful_update_at timestamp without time zone,
    ldap_sync_last_sync_at timestamp without time zone,
    ldap_sync_last_update_at timestamp without time zone,
    repository_size_limit bigint,
    saml_discovery_token character varying,
    shared_runners_minutes_limit integer,
    extra_shared_runners_minutes_limit integer,
    ldap_sync_status character varying DEFAULT 'ready'::character varying NOT NULL,
    membership_lock boolean DEFAULT false,
    last_ci_minutes_notification_at timestamp with time zone,
    last_ci_minutes_usage_notification_level integer,
    subgroup_creation_level integer DEFAULT 1,
    emails_disabled boolean,
    max_pages_size integer,
    max_artifacts_size integer,
    mentions_disabled boolean,
    default_branch_protection smallint,
    unlock_membership_to_ldap boolean,
    max_personal_access_token_lifetime integer,
    push_rule_id bigint,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    allow_descendants_override_disabled_shared_runners boolean DEFAULT false NOT NULL,
    traversal_ids integer[] DEFAULT '{}'::integer[] NOT NULL,
    tmp_project_id integer
);

CREATE SEQUENCE namespaces_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespaces_id_seq OWNED BY namespaces.id;

CREATE TABLE namespaces_sync_events (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL
);

CREATE SEQUENCE namespaces_sync_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE namespaces_sync_events_id_seq OWNED BY namespaces_sync_events.id;

CREATE TABLE note_diff_files (
    id integer NOT NULL,
    diff_note_id integer NOT NULL,
    diff text NOT NULL,
    new_file boolean NOT NULL,
    renamed_file boolean NOT NULL,
    deleted_file boolean NOT NULL,
    a_mode character varying NOT NULL,
    b_mode character varying NOT NULL,
    new_path text NOT NULL,
    old_path text NOT NULL
);

CREATE SEQUENCE note_diff_files_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE note_diff_files_id_seq OWNED BY note_diff_files.id;

CREATE TABLE notes (
    id integer NOT NULL,
    note text,
    noteable_type character varying,
    author_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id integer,
    attachment character varying,
    line_code character varying,
    commit_id character varying,
    noteable_id integer,
    system boolean DEFAULT false NOT NULL,
    st_diff text,
    updated_by_id integer,
    type character varying,
    "position" text,
    original_position text,
    resolved_at timestamp without time zone,
    resolved_by_id integer,
    discussion_id character varying,
    note_html text,
    cached_markdown_version integer,
    change_position text,
    resolved_by_push boolean,
    review_id bigint,
    confidential boolean,
    last_edited_at timestamp with time zone
);

CREATE SEQUENCE notes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE notes_id_seq OWNED BY notes.id;

CREATE TABLE notification_settings (
    id integer NOT NULL,
    user_id integer NOT NULL,
    source_type character varying,
    source_id integer,
    level integer DEFAULT 0 NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    new_note boolean,
    new_issue boolean,
    reopen_issue boolean,
    close_issue boolean,
    reassign_issue boolean,
    new_merge_request boolean,
    reopen_merge_request boolean,
    close_merge_request boolean,
    reassign_merge_request boolean,
    merge_merge_request boolean,
    failed_pipeline boolean,
    success_pipeline boolean,
    push_to_merge_request boolean,
    issue_due boolean,
    notification_email character varying,
    new_epic boolean,
    new_release boolean,
    fixed_pipeline boolean,
    moved_project boolean DEFAULT true NOT NULL,
    change_reviewer_merge_request boolean,
    merge_when_pipeline_succeeds boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE notification_settings_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE notification_settings_id_seq OWNED BY notification_settings.id;

CREATE TABLE oauth_access_grants (
    id integer NOT NULL,
    resource_owner_id integer NOT NULL,
    application_id integer NOT NULL,
    token character varying NOT NULL,
    expires_in integer NOT NULL,
    redirect_uri text NOT NULL,
    created_at timestamp without time zone NOT NULL,
    revoked_at timestamp without time zone,
    scopes character varying,
    code_challenge text,
    code_challenge_method text,
    CONSTRAINT oauth_access_grants_code_challenge CHECK ((char_length(code_challenge) <= 128)),
    CONSTRAINT oauth_access_grants_code_challenge_method CHECK ((char_length(code_challenge_method) <= 5))
);

CREATE SEQUENCE oauth_access_grants_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_access_grants_id_seq OWNED BY oauth_access_grants.id;

CREATE TABLE oauth_access_tokens (
    id integer NOT NULL,
    resource_owner_id integer,
    application_id integer,
    token character varying NOT NULL,
    refresh_token character varying,
    expires_in integer,
    revoked_at timestamp without time zone,
    created_at timestamp without time zone NOT NULL,
    scopes character varying
);

CREATE SEQUENCE oauth_access_tokens_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_access_tokens_id_seq OWNED BY oauth_access_tokens.id;

CREATE TABLE oauth_applications (
    id integer NOT NULL,
    name character varying NOT NULL,
    uid character varying NOT NULL,
    secret character varying NOT NULL,
    redirect_uri text NOT NULL,
    scopes character varying DEFAULT ''::character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    owner_id integer,
    owner_type character varying,
    trusted boolean DEFAULT false NOT NULL,
    confidential boolean DEFAULT true NOT NULL,
    expire_access_tokens boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE oauth_applications_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_applications_id_seq OWNED BY oauth_applications.id;

CREATE TABLE oauth_openid_requests (
    id integer NOT NULL,
    access_grant_id integer NOT NULL,
    nonce character varying NOT NULL
);

CREATE SEQUENCE oauth_openid_requests_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE oauth_openid_requests_id_seq OWNED BY oauth_openid_requests.id;

CREATE TABLE onboarding_progresses (
    id bigint NOT NULL,
    namespace_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    git_pull_at timestamp with time zone,
    git_write_at timestamp with time zone,
    merge_request_created_at timestamp with time zone,
    pipeline_created_at timestamp with time zone,
    user_added_at timestamp with time zone,
    trial_started_at timestamp with time zone,
    subscription_created_at timestamp with time zone,
    required_mr_approvals_enabled_at timestamp with time zone,
    code_owners_enabled_at timestamp with time zone,
    scoped_label_created_at timestamp with time zone,
    security_scan_enabled_at timestamp with time zone,
    issue_auto_closed_at timestamp with time zone,
    repository_imported_at timestamp with time zone,
    repository_mirrored_at timestamp with time zone,
    issue_created_at timestamp with time zone
);

CREATE SEQUENCE onboarding_progresses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE onboarding_progresses_id_seq OWNED BY onboarding_progresses.id;

CREATE TABLE operations_feature_flag_scopes (
    id bigint NOT NULL,
    feature_flag_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean NOT NULL,
    environment_scope character varying DEFAULT '*'::character varying NOT NULL,
    strategies jsonb DEFAULT '[{"name": "default", "parameters": {}}]'::jsonb NOT NULL
);

CREATE SEQUENCE operations_feature_flag_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flag_scopes_id_seq OWNED BY operations_feature_flag_scopes.id;

CREATE TABLE operations_feature_flags (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    active boolean NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    description text,
    iid integer NOT NULL,
    version smallint DEFAULT 1 NOT NULL
);

CREATE TABLE operations_feature_flags_clients (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    token_encrypted character varying
);

CREATE SEQUENCE operations_feature_flags_clients_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_clients_id_seq OWNED BY operations_feature_flags_clients.id;

CREATE SEQUENCE operations_feature_flags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_id_seq OWNED BY operations_feature_flags.id;

CREATE TABLE operations_feature_flags_issues (
    id bigint NOT NULL,
    feature_flag_id bigint NOT NULL,
    issue_id bigint NOT NULL
);

CREATE SEQUENCE operations_feature_flags_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_feature_flags_issues_id_seq OWNED BY operations_feature_flags_issues.id;

CREATE TABLE operations_scopes (
    id bigint NOT NULL,
    strategy_id bigint NOT NULL,
    environment_scope character varying(255) NOT NULL
);

CREATE SEQUENCE operations_scopes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_scopes_id_seq OWNED BY operations_scopes.id;

CREATE TABLE operations_strategies (
    id bigint NOT NULL,
    feature_flag_id bigint NOT NULL,
    name character varying(255) NOT NULL,
    parameters jsonb DEFAULT '{}'::jsonb NOT NULL
);

CREATE SEQUENCE operations_strategies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_strategies_id_seq OWNED BY operations_strategies.id;

CREATE TABLE operations_strategies_user_lists (
    id bigint NOT NULL,
    strategy_id bigint NOT NULL,
    user_list_id bigint NOT NULL
);

CREATE SEQUENCE operations_strategies_user_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_strategies_user_lists_id_seq OWNED BY operations_strategies_user_lists.id;

CREATE TABLE operations_user_lists (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    iid integer NOT NULL,
    name character varying(255) NOT NULL,
    user_xids text DEFAULT ''::text NOT NULL
);

CREATE SEQUENCE operations_user_lists_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE operations_user_lists_id_seq OWNED BY operations_user_lists.id;

CREATE TABLE packages_build_infos (
    id bigint NOT NULL,
    package_id integer NOT NULL,
    pipeline_id integer
);

CREATE SEQUENCE packages_build_infos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_build_infos_id_seq OWNED BY packages_build_infos.id;

CREATE TABLE packages_composer_cache_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    delete_at timestamp with time zone,
    namespace_id integer,
    file_store smallint DEFAULT 1 NOT NULL,
    file text NOT NULL,
    file_sha256 bytea NOT NULL,
    CONSTRAINT check_84f5ba81f5 CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_composer_cache_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_composer_cache_files_id_seq OWNED BY packages_composer_cache_files.id;

CREATE TABLE packages_composer_metadata (
    package_id bigint NOT NULL,
    target_sha bytea NOT NULL,
    composer_json jsonb DEFAULT '{}'::jsonb NOT NULL,
    version_cache_sha bytea
);

CREATE TABLE packages_conan_file_metadata (
    id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    recipe_revision character varying(255) DEFAULT '0'::character varying NOT NULL,
    package_revision character varying(255),
    conan_package_reference character varying(255),
    conan_file_type smallint NOT NULL
);

CREATE SEQUENCE packages_conan_file_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_file_metadata_id_seq OWNED BY packages_conan_file_metadata.id;

CREATE TABLE packages_conan_metadata (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_username character varying(255) NOT NULL,
    package_channel character varying(255) NOT NULL
);

CREATE SEQUENCE packages_conan_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_conan_metadata_id_seq OWNED BY packages_conan_metadata.id;

CREATE TABLE packages_debian_file_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_file_id bigint NOT NULL,
    file_type smallint NOT NULL,
    component text,
    architecture text,
    fields jsonb,
    CONSTRAINT check_2ebedda4b6 CHECK ((char_length(component) <= 255)),
    CONSTRAINT check_e6e1fffcca CHECK ((char_length(architecture) <= 255))
);

CREATE TABLE packages_debian_group_architectures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_ddb220164a CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE packages_debian_group_architectures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_architectures_id_seq OWNED BY packages_debian_group_architectures.id;

CREATE TABLE packages_debian_group_component_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_id bigint NOT NULL,
    architecture_id bigint,
    size integer NOT NULL,
    file_type smallint NOT NULL,
    compression_type smallint,
    file_store smallint DEFAULT 1 NOT NULL,
    file text NOT NULL,
    file_md5 bytea NOT NULL,
    file_sha256 bytea NOT NULL,
    CONSTRAINT check_839e1685bc CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_group_component_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_component_files_id_seq OWNED BY packages_debian_group_component_files.id;

CREATE TABLE packages_debian_group_components (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_a9bc7d85be CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE packages_debian_group_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_components_id_seq OWNED BY packages_debian_group_components.id;

CREATE TABLE packages_debian_group_distribution_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL,
    encrypted_passphrase text NOT NULL,
    encrypted_passphrase_iv text NOT NULL,
    public_key text NOT NULL,
    fingerprint text NOT NULL,
    CONSTRAINT check_bc95dc3fbe CHECK ((char_length(fingerprint) <= 255)),
    CONSTRAINT check_f708183491 CHECK ((char_length(public_key) <= 524288))
);

CREATE SEQUENCE packages_debian_group_distribution_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_distribution_keys_id_seq OWNED BY packages_debian_group_distribution_keys.id;

CREATE TABLE packages_debian_group_distributions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id bigint NOT NULL,
    creator_id bigint,
    valid_time_duration_seconds integer,
    file_store smallint DEFAULT 1 NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    automatic_upgrades boolean DEFAULT false NOT NULL,
    codename text NOT NULL,
    suite text,
    origin text,
    label text,
    version text,
    description text,
    file text,
    file_signature text,
    signed_file text,
    signed_file_store smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_0007e0bf61 CHECK ((char_length(signed_file) <= 255)),
    CONSTRAINT check_310ac457b8 CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_3d6f87fc31 CHECK ((char_length(file_signature) <= 4096)),
    CONSTRAINT check_3fdadf4a0c CHECK ((char_length(version) <= 255)),
    CONSTRAINT check_590e18405a CHECK ((char_length(codename) <= 255)),
    CONSTRAINT check_b057cd840a CHECK ((char_length(origin) <= 255)),
    CONSTRAINT check_be5ed8d307 CHECK ((char_length(file) <= 255)),
    CONSTRAINT check_d3244bfc0b CHECK ((char_length(label) <= 255)),
    CONSTRAINT check_e7c928a24b CHECK ((char_length(suite) <= 255))
);

CREATE SEQUENCE packages_debian_group_distributions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_group_distributions_id_seq OWNED BY packages_debian_group_distributions.id;

CREATE TABLE packages_debian_project_architectures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_9c2e1c99d8 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE packages_debian_project_architectures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_architectures_id_seq OWNED BY packages_debian_project_architectures.id;

CREATE TABLE packages_debian_project_component_files (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    component_id bigint NOT NULL,
    architecture_id bigint,
    size integer NOT NULL,
    file_type smallint NOT NULL,
    compression_type smallint,
    file_store smallint DEFAULT 1 NOT NULL,
    file text NOT NULL,
    file_md5 bytea NOT NULL,
    file_sha256 bytea NOT NULL,
    CONSTRAINT check_e5af03fa2d CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_project_component_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_component_files_id_seq OWNED BY packages_debian_project_component_files.id;

CREATE TABLE packages_debian_project_components (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_517559f298 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE packages_debian_project_components_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_components_id_seq OWNED BY packages_debian_project_components.id;

CREATE TABLE packages_debian_project_distribution_keys (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    distribution_id bigint NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL,
    encrypted_passphrase text NOT NULL,
    encrypted_passphrase_iv text NOT NULL,
    public_key text NOT NULL,
    fingerprint text NOT NULL,
    CONSTRAINT check_9e8a5eef0a CHECK ((char_length(fingerprint) <= 255)),
    CONSTRAINT check_d188f6547f CHECK ((char_length(public_key) <= 524288))
);

CREATE SEQUENCE packages_debian_project_distribution_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_distribution_keys_id_seq OWNED BY packages_debian_project_distribution_keys.id;

CREATE TABLE packages_debian_project_distributions (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    creator_id bigint,
    valid_time_duration_seconds integer,
    file_store smallint DEFAULT 1 NOT NULL,
    automatic boolean DEFAULT true NOT NULL,
    automatic_upgrades boolean DEFAULT false NOT NULL,
    codename text NOT NULL,
    suite text,
    origin text,
    label text,
    version text,
    description text,
    file text,
    file_signature text,
    signed_file text,
    signed_file_store smallint DEFAULT 1 NOT NULL,
    CONSTRAINT check_6177ccd4a6 CHECK ((char_length(origin) <= 255)),
    CONSTRAINT check_6f6b55a4c4 CHECK ((char_length(label) <= 255)),
    CONSTRAINT check_834dabadb6 CHECK ((char_length(codename) <= 255)),
    CONSTRAINT check_96965792c2 CHECK ((char_length(version) <= 255)),
    CONSTRAINT check_9e5e22b7ff CHECK ((char_length(signed_file) <= 255)),
    CONSTRAINT check_a56ae58a17 CHECK ((char_length(suite) <= 255)),
    CONSTRAINT check_a5a2ac6af2 CHECK ((char_length(file_signature) <= 4096)),
    CONSTRAINT check_b93154339f CHECK ((char_length(description) <= 255)),
    CONSTRAINT check_cb4ac9599e CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE packages_debian_project_distributions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_project_distributions_id_seq OWNED BY packages_debian_project_distributions.id;

CREATE TABLE packages_debian_publications (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    distribution_id bigint NOT NULL
);

CREATE SEQUENCE packages_debian_publications_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_debian_publications_id_seq OWNED BY packages_debian_publications.id;

CREATE TABLE packages_dependencies (
    id bigint NOT NULL,
    name character varying(255) NOT NULL,
    version_pattern character varying(255) NOT NULL
);

CREATE SEQUENCE packages_dependencies_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_dependencies_id_seq OWNED BY packages_dependencies.id;

CREATE TABLE packages_dependency_links (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    dependency_id bigint NOT NULL,
    dependency_type smallint NOT NULL
);

CREATE SEQUENCE packages_dependency_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_dependency_links_id_seq OWNED BY packages_dependency_links.id;

CREATE TABLE packages_events (
    id bigint NOT NULL,
    event_type smallint NOT NULL,
    event_scope smallint NOT NULL,
    originator_type smallint NOT NULL,
    originator bigint,
    created_at timestamp with time zone NOT NULL,
    package_id bigint
);

CREATE SEQUENCE packages_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_events_id_seq OWNED BY packages_events.id;

CREATE TABLE packages_helm_file_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_file_id bigint NOT NULL,
    channel text NOT NULL,
    metadata jsonb,
    CONSTRAINT check_06e8d100af CHECK ((char_length(channel) <= 255))
);

CREATE TABLE packages_maven_metadata (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    app_group character varying NOT NULL,
    app_name character varying NOT NULL,
    app_version character varying,
    path character varying(512) NOT NULL
);

CREATE SEQUENCE packages_maven_metadata_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_maven_metadata_id_seq OWNED BY packages_maven_metadata.id;

CREATE TABLE packages_npm_metadata (
    package_id bigint NOT NULL,
    package_json jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT chk_rails_e5cbc301ae CHECK ((char_length((package_json)::text) < 20000))
);

CREATE TABLE packages_nuget_dependency_link_metadata (
    dependency_link_id bigint NOT NULL,
    target_framework text NOT NULL,
    CONSTRAINT packages_nuget_dependency_link_metadata_target_framework_constr CHECK ((char_length(target_framework) <= 255))
);

CREATE TABLE packages_nuget_metadata (
    package_id bigint NOT NULL,
    license_url text,
    project_url text,
    icon_url text,
    CONSTRAINT packages_nuget_metadata_icon_url_constraint CHECK ((char_length(icon_url) <= 255)),
    CONSTRAINT packages_nuget_metadata_license_url_constraint CHECK ((char_length(license_url) <= 255)),
    CONSTRAINT packages_nuget_metadata_project_url_constraint CHECK ((char_length(project_url) <= 255))
);

CREATE TABLE packages_package_file_build_infos (
    id bigint NOT NULL,
    package_file_id bigint NOT NULL,
    pipeline_id bigint
);

CREATE SEQUENCE packages_package_file_build_infos_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_package_file_build_infos_id_seq OWNED BY packages_package_file_build_infos.id;

CREATE TABLE packages_package_files (
    id bigint NOT NULL,
    package_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    size bigint,
    file_store integer DEFAULT 1,
    file_md5 bytea,
    file_sha1 bytea,
    file_name character varying NOT NULL,
    file text NOT NULL,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_failure character varying(255),
    verification_retry_count integer,
    file_sha256 bytea,
    verification_checksum bytea,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    CONSTRAINT check_4c5e6bb0b3 CHECK ((file_store IS NOT NULL))
);

CREATE SEQUENCE packages_package_files_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_package_files_id_seq OWNED BY packages_package_files.id;

CREATE TABLE packages_packages (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    version character varying,
    package_type smallint NOT NULL,
    creator_id integer,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE packages_packages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_packages_id_seq OWNED BY packages_packages.id;

CREATE TABLE packages_pypi_metadata (
    package_id bigint NOT NULL,
    required_python text,
    CONSTRAINT check_0d9aed55b2 CHECK ((required_python IS NOT NULL)),
    CONSTRAINT check_379019d5da CHECK ((char_length(required_python) <= 255))
);

CREATE TABLE packages_rubygems_metadata (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    package_id bigint NOT NULL,
    authors text,
    files text,
    summary text,
    description text,
    email text,
    homepage text,
    licenses text,
    metadata text,
    author text,
    bindir text,
    cert_chain text,
    executables text,
    extensions text,
    extra_rdoc_files text,
    platform text,
    post_install_message text,
    rdoc_options text,
    require_paths text,
    required_ruby_version text,
    required_rubygems_version text,
    requirements text,
    rubygems_version text,
    signing_key text,
    CONSTRAINT check_0154a18c82 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_22814c771b CHECK ((char_length(email) <= 255)),
    CONSTRAINT check_242293030e CHECK ((char_length(extensions) <= 255)),
    CONSTRAINT check_27619a7922 CHECK ((char_length(rubygems_version) <= 255)),
    CONSTRAINT check_3d1b6f3a39 CHECK ((char_length(post_install_message) <= 255)),
    CONSTRAINT check_545f7606f9 CHECK ((char_length(required_rubygems_version) <= 255)),
    CONSTRAINT check_5988451714 CHECK ((char_length(executables) <= 255)),
    CONSTRAINT check_5f9c84ea17 CHECK ((char_length(platform) <= 255)),
    CONSTRAINT check_64f1cecf05 CHECK ((char_length(requirements) <= 255)),
    CONSTRAINT check_6ac7043c50 CHECK ((char_length(extra_rdoc_files) <= 255)),
    CONSTRAINT check_6ff3abe325 CHECK ((char_length(cert_chain) <= 255)),
    CONSTRAINT check_7cb01436df CHECK ((char_length(licenses) <= 255)),
    CONSTRAINT check_8be21d92e7 CHECK ((char_length(summary) <= 1024)),
    CONSTRAINT check_946cb96acb CHECK ((char_length(homepage) <= 255)),
    CONSTRAINT check_9824fc9efc CHECK ((char_length(bindir) <= 255)),
    CONSTRAINT check_994b68eb64 CHECK ((char_length(authors) <= 255)),
    CONSTRAINT check_9d42fa48ae CHECK ((char_length(signing_key) <= 255)),
    CONSTRAINT check_b0f4f8c853 CHECK ((char_length(files) <= 255)),
    CONSTRAINT check_b7b296b420 CHECK ((char_length(author) <= 255)),
    CONSTRAINT check_bf16b21a47 CHECK ((char_length(rdoc_options) <= 255)),
    CONSTRAINT check_ca641a3354 CHECK ((char_length(required_ruby_version) <= 255)),
    CONSTRAINT check_ea02f4800f CHECK ((char_length(metadata) <= 30000)),
    CONSTRAINT check_f76bad1a9a CHECK ((char_length(require_paths) <= 255))
);

CREATE TABLE packages_tags (
    id bigint NOT NULL,
    package_id integer NOT NULL,
    name character varying(255) NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE packages_tags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE packages_tags_id_seq OWNED BY packages_tags.id;

CREATE TABLE pages_deployment_states (
    pages_deployment_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_15217e8c3a CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE pages_deployment_states_pages_deployment_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_deployment_states_pages_deployment_id_seq OWNED BY pages_deployment_states.pages_deployment_id;

CREATE TABLE pages_deployments (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    ci_build_id bigint,
    file_store smallint NOT NULL,
    file text NOT NULL,
    file_count integer NOT NULL,
    file_sha256 bytea NOT NULL,
    size bigint,
    CONSTRAINT check_5f9132a958 CHECK ((size IS NOT NULL)),
    CONSTRAINT check_f0fe8032dd CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE pages_deployments_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_deployments_id_seq OWNED BY pages_deployments.id;

CREATE TABLE pages_domain_acme_orders (
    id bigint NOT NULL,
    pages_domain_id integer NOT NULL,
    expires_at timestamp with time zone NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    url character varying NOT NULL,
    challenge_token character varying NOT NULL,
    challenge_file_content text NOT NULL,
    encrypted_private_key text NOT NULL,
    encrypted_private_key_iv text NOT NULL
);

CREATE SEQUENCE pages_domain_acme_orders_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_domain_acme_orders_id_seq OWNED BY pages_domain_acme_orders.id;

CREATE TABLE pages_domains (
    id integer NOT NULL,
    project_id integer,
    certificate text,
    encrypted_key text,
    encrypted_key_iv character varying,
    encrypted_key_salt character varying,
    domain character varying,
    verified_at timestamp with time zone,
    verification_code character varying NOT NULL,
    enabled_until timestamp with time zone,
    remove_at timestamp with time zone,
    auto_ssl_enabled boolean DEFAULT false NOT NULL,
    certificate_valid_not_before timestamp with time zone,
    certificate_valid_not_after timestamp with time zone,
    certificate_source smallint DEFAULT 0 NOT NULL,
    wildcard boolean DEFAULT false NOT NULL,
    usage smallint DEFAULT 0 NOT NULL,
    scope smallint DEFAULT 2,
    auto_ssl_failed boolean DEFAULT false NOT NULL,
    CONSTRAINT check_ab7cf26a46 CHECK ((scope IS NOT NULL))
);

CREATE SEQUENCE pages_domains_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pages_domains_id_seq OWNED BY pages_domains.id;

CREATE TABLE partitioned_foreign_keys (
    id bigint NOT NULL,
    cascade_delete boolean DEFAULT true NOT NULL,
    from_table text NOT NULL,
    from_column text NOT NULL,
    to_table text NOT NULL,
    to_column text NOT NULL,
    CONSTRAINT check_2c2e02a62b CHECK ((char_length(from_column) <= 63)),
    CONSTRAINT check_40738efb57 CHECK ((char_length(to_table) <= 63)),
    CONSTRAINT check_741676d405 CHECK ((char_length(from_table) <= 63)),
    CONSTRAINT check_7e98be694f CHECK ((char_length(to_column) <= 63))
);

CREATE SEQUENCE partitioned_foreign_keys_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE partitioned_foreign_keys_id_seq OWNED BY partitioned_foreign_keys.id;

CREATE TABLE path_locks (
    id integer NOT NULL,
    path character varying NOT NULL,
    project_id integer,
    user_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE path_locks_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE path_locks_id_seq OWNED BY path_locks.id;

CREATE TABLE personal_access_tokens (
    id integer NOT NULL,
    user_id integer NOT NULL,
    name character varying NOT NULL,
    revoked boolean DEFAULT false,
    expires_at date,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    scopes character varying DEFAULT '--- []
'::character varying NOT NULL,
    impersonation boolean DEFAULT false NOT NULL,
    token_digest character varying,
    expire_notification_delivered boolean DEFAULT false NOT NULL,
    last_used_at timestamp with time zone,
    after_expiry_notification_delivered boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE personal_access_tokens_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE personal_access_tokens_id_seq OWNED BY personal_access_tokens.id;

CREATE TABLE plan_limits (
    plan_id bigint NOT NULL,
    ci_active_pipelines integer DEFAULT 0 NOT NULL,
    ci_pipeline_size integer DEFAULT 0 NOT NULL,
    ci_active_jobs integer DEFAULT 0 NOT NULL,
    id bigint NOT NULL,
    project_hooks integer DEFAULT 100 NOT NULL,
    group_hooks integer DEFAULT 50 NOT NULL,
    ci_project_subscriptions integer DEFAULT 2 NOT NULL,
    ci_pipeline_schedules integer DEFAULT 10 NOT NULL,
    offset_pagination_limit integer DEFAULT 50000 NOT NULL,
    ci_instance_level_variables integer DEFAULT 25 NOT NULL,
    ci_max_artifact_size_lsif integer DEFAULT 100 NOT NULL,
    ci_max_artifact_size_archive integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metadata integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_trace integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_junit integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_sast integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_dependency_scanning integer DEFAULT 350 NOT NULL,
    ci_max_artifact_size_container_scanning integer DEFAULT 150 NOT NULL,
    ci_max_artifact_size_dast integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_codequality integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_license_management integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_license_scanning integer DEFAULT 100 NOT NULL,
    ci_max_artifact_size_performance integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metrics integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_metrics_referee integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_network_referee integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_dotenv integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cobertura integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_terraform integer DEFAULT 5 NOT NULL,
    ci_max_artifact_size_accessibility integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cluster_applications integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_secret_detection integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_requirements integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_coverage_fuzzing integer DEFAULT 0 NOT NULL,
    storage_size_limit integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_browser_performance integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_load_performance integer DEFAULT 0 NOT NULL,
    ci_needs_size_limit integer DEFAULT 50 NOT NULL,
    conan_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    maven_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    npm_max_file_size bigint DEFAULT 524288000 NOT NULL,
    nuget_max_file_size bigint DEFAULT 524288000 NOT NULL,
    pypi_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    generic_packages_max_file_size bigint DEFAULT '5368709120'::bigint NOT NULL,
    project_feature_flags integer DEFAULT 200 NOT NULL,
    golang_max_file_size bigint DEFAULT 104857600 NOT NULL,
    debian_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    ci_max_artifact_size_api_fuzzing integer DEFAULT 0 NOT NULL,
    daily_invites integer DEFAULT 0 NOT NULL,
    ci_pipeline_deployments integer DEFAULT 500 NOT NULL,
    pull_mirror_interval_seconds integer DEFAULT 300 NOT NULL,
    rubygems_max_file_size bigint DEFAULT '3221225472'::bigint NOT NULL,
    terraform_module_max_file_size bigint DEFAULT 1073741824 NOT NULL,
    helm_max_file_size bigint DEFAULT 5242880 NOT NULL,
    ci_registered_group_runners integer DEFAULT 1000 NOT NULL,
    ci_registered_project_runners integer DEFAULT 1000 NOT NULL,
    web_hook_calls integer DEFAULT 0 NOT NULL,
    ci_daily_pipeline_schedule_triggers integer DEFAULT 0 NOT NULL,
    pages_file_entries integer DEFAULT 200000 NOT NULL,
    ci_max_artifact_size_running_container_scanning integer DEFAULT 0 NOT NULL,
    ci_max_artifact_size_cluster_image_scanning integer DEFAULT 0 NOT NULL,
    ci_jobs_trace_size_limit integer DEFAULT 100 NOT NULL,
    dast_profile_schedules integer DEFAULT 1 NOT NULL,
    external_audit_event_destinations integer DEFAULT 5 NOT NULL,
    dotenv_variables integer DEFAULT 20 NOT NULL,
    dotenv_size integer DEFAULT 5120 NOT NULL,
    pipeline_triggers integer DEFAULT 25000 NOT NULL
);

CREATE SEQUENCE plan_limits_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE plan_limits_id_seq OWNED BY plan_limits.id;

CREATE TABLE plans (
    id integer NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    name character varying,
    title character varying
);

CREATE SEQUENCE plans_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE plans_id_seq OWNED BY plans.id;

CREATE TABLE pool_repositories (
    id bigint NOT NULL,
    shard_id integer NOT NULL,
    disk_path character varying,
    state character varying,
    source_project_id integer
);

CREATE SEQUENCE pool_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE pool_repositories_id_seq OWNED BY pool_repositories.id;

CREATE TABLE postgres_async_indexes (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name text NOT NULL,
    definition text NOT NULL,
    table_name text NOT NULL,
    CONSTRAINT check_083b21157b CHECK ((char_length(definition) <= 2048)),
    CONSTRAINT check_b732c6cd1d CHECK ((char_length(name) <= 63)),
    CONSTRAINT check_e64ff4359e CHECK ((char_length(table_name) <= 63))
);

CREATE SEQUENCE postgres_async_indexes_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_async_indexes_id_seq OWNED BY postgres_async_indexes.id;

CREATE VIEW postgres_foreign_keys AS
 SELECT pg_constraint.oid,
    pg_constraint.conname AS name,
    (((constrained_namespace.nspname)::text || '.'::text) || (constrained_table.relname)::text) AS constrained_table_identifier,
    (((referenced_namespace.nspname)::text || '.'::text) || (referenced_table.relname)::text) AS referenced_table_identifier
   FROM ((((pg_constraint
     JOIN pg_class constrained_table ON ((constrained_table.oid = pg_constraint.conrelid)))
     JOIN pg_class referenced_table ON ((referenced_table.oid = pg_constraint.confrelid)))
     JOIN pg_namespace constrained_namespace ON ((constrained_table.relnamespace = constrained_namespace.oid)))
     JOIN pg_namespace referenced_namespace ON ((referenced_table.relnamespace = referenced_namespace.oid)))
  WHERE (pg_constraint.contype = 'f'::"char");

CREATE VIEW postgres_index_bloat_estimates AS
 SELECT (((relation_stats.nspname)::text || '.'::text) || (relation_stats.idxname)::text) AS identifier,
    (
        CASE
            WHEN ((relation_stats.relpages)::double precision > relation_stats.est_pages_ff) THEN ((relation_stats.bs)::double precision * ((relation_stats.relpages)::double precision - relation_stats.est_pages_ff))
            ELSE (0)::double precision
        END)::bigint AS bloat_size_bytes
   FROM ( SELECT COALESCE(((1)::double precision + ceil((rows_hdr_pdg_stats.reltuples / floor((((((rows_hdr_pdg_stats.bs - (rows_hdr_pdg_stats.pageopqdata)::numeric) - (rows_hdr_pdg_stats.pagehdr)::numeric) * (rows_hdr_pdg_stats.fillfactor)::numeric))::double precision / ((100)::double precision * (((4)::numeric + rows_hdr_pdg_stats.nulldatahdrwidth))::double precision)))))), (0)::double precision) AS est_pages_ff,
            rows_hdr_pdg_stats.bs,
            rows_hdr_pdg_stats.nspname,
            rows_hdr_pdg_stats.tblname,
            rows_hdr_pdg_stats.idxname,
            rows_hdr_pdg_stats.relpages,
            rows_hdr_pdg_stats.is_na
           FROM ( SELECT rows_data_stats.maxalign,
                    rows_data_stats.bs,
                    rows_data_stats.nspname,
                    rows_data_stats.tblname,
                    rows_data_stats.idxname,
                    rows_data_stats.reltuples,
                    rows_data_stats.relpages,
                    rows_data_stats.idxoid,
                    rows_data_stats.fillfactor,
                    (((((((rows_data_stats.index_tuple_hdr_bm + rows_data_stats.maxalign) -
                        CASE
                            WHEN ((rows_data_stats.index_tuple_hdr_bm % rows_data_stats.maxalign) = 0) THEN rows_data_stats.maxalign
                            ELSE (rows_data_stats.index_tuple_hdr_bm % rows_data_stats.maxalign)
                        END))::double precision + rows_data_stats.nulldatawidth) + (rows_data_stats.maxalign)::double precision) - (
                        CASE
                            WHEN (rows_data_stats.nulldatawidth = (0)::double precision) THEN 0
                            WHEN (((rows_data_stats.nulldatawidth)::integer % rows_data_stats.maxalign) = 0) THEN rows_data_stats.maxalign
                            ELSE ((rows_data_stats.nulldatawidth)::integer % rows_data_stats.maxalign)
                        END)::double precision))::numeric AS nulldatahdrwidth,
                    rows_data_stats.pagehdr,
                    rows_data_stats.pageopqdata,
                    rows_data_stats.is_na
                   FROM ( SELECT n.nspname,
                            i.tblname,
                            i.idxname,
                            i.reltuples,
                            i.relpages,
                            i.idxoid,
                            i.fillfactor,
                            (current_setting('block_size'::text))::numeric AS bs,
                                CASE
                                    WHEN ((version() ~ 'mingw32'::text) OR (version() ~ '64-bit|x86_64|ppc64|ia64|amd64'::text)) THEN 8
                                    ELSE 4
                                END AS maxalign,
                            24 AS pagehdr,
                            16 AS pageopqdata,
                                CASE
                                    WHEN (max(COALESCE(s.null_frac, (0)::real)) = (0)::double precision) THEN 2
                                    ELSE (2 + (((32 + 8) - 1) / 8))
                                END AS index_tuple_hdr_bm,
                            sum((((1)::double precision - COALESCE(s.null_frac, (0)::real)) * (COALESCE(s.avg_width, 1024))::double precision)) AS nulldatawidth,
                            (max(
                                CASE
                                    WHEN (i.atttypid = ('name'::regtype)::oid) THEN 1
                                    ELSE 0
                                END) > 0) AS is_na
                           FROM ((( SELECT ct.relname AS tblname,
                                    ct.relnamespace,
                                    ic.idxname,
                                    ic.attpos,
                                    ic.indkey,
                                    ic.indkey[ic.attpos] AS indkey,
                                    ic.reltuples,
                                    ic.relpages,
                                    ic.tbloid,
                                    ic.idxoid,
                                    ic.fillfactor,
                                    COALESCE(a1.attnum, a2.attnum) AS attnum,
                                    COALESCE(a1.attname, a2.attname) AS attname,
                                    COALESCE(a1.atttypid, a2.atttypid) AS atttypid,
CASE
 WHEN (a1.attnum IS NULL) THEN ic.idxname
 ELSE ct.relname
END AS attrelname
                                   FROM (((( SELECT idx_data.idxname,
    idx_data.reltuples,
    idx_data.relpages,
    idx_data.tbloid,
    idx_data.idxoid,
    idx_data.fillfactor,
    idx_data.indkey,
    generate_series(1, (idx_data.indnatts)::integer) AS attpos
   FROM ( SELECT ci.relname AS idxname,
      ci.reltuples,
      ci.relpages,
      i_1.indrelid AS tbloid,
      i_1.indexrelid AS idxoid,
      COALESCE((("substring"(array_to_string(ci.reloptions, ' '::text), 'fillfactor=([0-9]+)'::text))::smallint)::integer, 90) AS fillfactor,
      i_1.indnatts,
      (string_to_array(textin(int2vectorout(i_1.indkey)), ' '::text))::integer[] AS indkey
     FROM (pg_index i_1
       JOIN pg_class ci ON ((ci.oid = i_1.indexrelid)))
    WHERE ((ci.relam = ( SELECT pg_am.oid
       FROM pg_am
      WHERE (pg_am.amname = 'btree'::name))) AND (ci.relpages > 0))) idx_data) ic
                                     JOIN pg_class ct ON ((ct.oid = ic.tbloid)))
                                     LEFT JOIN pg_attribute a1 ON (((ic.indkey[ic.attpos] <> 0) AND (a1.attrelid = ic.tbloid) AND (a1.attnum = ic.indkey[ic.attpos]))))
                                     LEFT JOIN pg_attribute a2 ON (((ic.indkey[ic.attpos] = 0) AND (a2.attrelid = ic.idxoid) AND (a2.attnum = ic.attpos))))) i(tblname, relnamespace, idxname, attpos, indkey, indkey_1, reltuples, relpages, tbloid, idxoid, fillfactor, attnum, attname, atttypid, attrelname)
                             JOIN pg_namespace n ON ((n.oid = i.relnamespace)))
                             JOIN pg_stats s ON (((s.schemaname = n.nspname) AND (s.tablename = i.attrelname) AND (s.attname = i.attname))))
                          GROUP BY n.nspname, i.tblname, i.idxname, i.reltuples, i.relpages, i.idxoid, i.fillfactor, (current_setting('block_size'::text))::numeric,
                                CASE
                                    WHEN ((version() ~ 'mingw32'::text) OR (version() ~ '64-bit|x86_64|ppc64|ia64|amd64'::text)) THEN 8
                                    ELSE 4
                                END, 24::integer, 16::integer) rows_data_stats) rows_hdr_pdg_stats) relation_stats
  WHERE ((relation_stats.nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])) AND (NOT relation_stats.is_na))
  ORDER BY relation_stats.nspname, relation_stats.tblname, relation_stats.idxname;

CREATE VIEW postgres_indexes AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (i.relname)::text) AS identifier,
    pg_index.indexrelid,
    pg_namespace.nspname AS schema,
    i.relname AS name,
    pg_indexes.tablename,
    a.amname AS type,
    pg_index.indisunique AS "unique",
    pg_index.indisvalid AS valid_index,
    i.relispartition AS partitioned,
    pg_index.indisexclusion AS exclusion,
    (pg_index.indexprs IS NOT NULL) AS expression,
    (pg_index.indpred IS NOT NULL) AS partial,
    pg_indexes.indexdef AS definition,
    pg_relation_size((i.oid)::regclass) AS ondisk_size_bytes
   FROM ((((pg_index
     JOIN pg_class i ON ((i.oid = pg_index.indexrelid)))
     JOIN pg_namespace ON ((i.relnamespace = pg_namespace.oid)))
     JOIN pg_indexes ON (((i.relname = pg_indexes.indexname) AND (pg_namespace.nspname = pg_indexes.schemaname))))
     JOIN pg_am a ON ((i.relam = a.oid)))
  WHERE ((pg_namespace.nspname <> 'pg_catalog'::name) AND (pg_namespace.nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])));

CREATE VIEW postgres_partitioned_tables AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text) AS identifier,
    pg_class.oid,
    pg_namespace.nspname AS schema,
    pg_class.relname AS name,
        CASE partitioned_tables.partstrat
            WHEN 'l'::"char" THEN 'list'::text
            WHEN 'r'::"char" THEN 'range'::text
            WHEN 'h'::"char" THEN 'hash'::text
            ELSE NULL::text
        END AS strategy,
    array_agg(pg_attribute.attname) AS key_columns
   FROM (((( SELECT pg_partitioned_table.partrelid,
            pg_partitioned_table.partstrat,
            unnest(pg_partitioned_table.partattrs) AS column_position
           FROM pg_partitioned_table) partitioned_tables
     JOIN pg_class ON ((partitioned_tables.partrelid = pg_class.oid)))
     JOIN pg_namespace ON ((pg_class.relnamespace = pg_namespace.oid)))
     JOIN pg_attribute ON (((pg_attribute.attrelid = pg_class.oid) AND (pg_attribute.attnum = partitioned_tables.column_position))))
  WHERE (pg_namespace.nspname = "current_schema"())
  GROUP BY (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text), pg_class.oid, pg_namespace.nspname, pg_class.relname,
        CASE partitioned_tables.partstrat
            WHEN 'l'::"char" THEN 'list'::text
            WHEN 'r'::"char" THEN 'range'::text
            WHEN 'h'::"char" THEN 'hash'::text
            ELSE NULL::text
        END;

CREATE VIEW postgres_partitions AS
 SELECT (((pg_namespace.nspname)::text || '.'::text) || (pg_class.relname)::text) AS identifier,
    pg_class.oid,
    pg_namespace.nspname AS schema,
    pg_class.relname AS name,
    (((parent_namespace.nspname)::text || '.'::text) || (parent_class.relname)::text) AS parent_identifier,
    pg_get_expr(pg_class.relpartbound, pg_inherits.inhrelid) AS condition
   FROM ((((pg_class
     JOIN pg_namespace ON ((pg_namespace.oid = pg_class.relnamespace)))
     JOIN pg_inherits ON ((pg_class.oid = pg_inherits.inhrelid)))
     JOIN pg_class parent_class ON ((pg_inherits.inhparent = parent_class.oid)))
     JOIN pg_namespace parent_namespace ON ((parent_class.relnamespace = parent_namespace.oid)))
  WHERE (pg_class.relispartition AND (pg_namespace.nspname = ANY (ARRAY["current_schema"(), 'gitlab_partitions_dynamic'::name, 'gitlab_partitions_static'::name])));

CREATE TABLE postgres_reindex_actions (
    id bigint NOT NULL,
    action_start timestamp with time zone NOT NULL,
    action_end timestamp with time zone,
    ondisk_size_bytes_start bigint NOT NULL,
    ondisk_size_bytes_end bigint,
    state smallint DEFAULT 0 NOT NULL,
    index_identifier text NOT NULL,
    bloat_estimate_bytes_start bigint,
    CONSTRAINT check_f12527622c CHECK ((char_length(index_identifier) <= 255))
);

CREATE SEQUENCE postgres_reindex_actions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_reindex_actions_id_seq OWNED BY postgres_reindex_actions.id;

CREATE TABLE postgres_reindex_queued_actions (
    id bigint NOT NULL,
    index_identifier text NOT NULL,
    state smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone DEFAULT now() NOT NULL,
    updated_at timestamp with time zone DEFAULT now() NOT NULL,
    CONSTRAINT check_e4b01395c0 CHECK ((char_length(index_identifier) <= 255))
);

CREATE SEQUENCE postgres_reindex_queued_actions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE postgres_reindex_queued_actions_id_seq OWNED BY postgres_reindex_queued_actions.id;

CREATE TABLE programming_languages (
    id integer NOT NULL,
    name character varying NOT NULL,
    color character varying NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE programming_languages_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE programming_languages_id_seq OWNED BY programming_languages.id;

CREATE TABLE project_access_tokens (
    personal_access_token_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE TABLE project_alerting_settings (
    project_id integer NOT NULL,
    encrypted_token character varying NOT NULL,
    encrypted_token_iv character varying NOT NULL
);

CREATE TABLE project_aliases (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    name character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_aliases_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_aliases_id_seq OWNED BY project_aliases.id;

CREATE TABLE project_authorizations (
    user_id integer NOT NULL,
    project_id integer NOT NULL,
    access_level integer NOT NULL
);

CREATE TABLE project_auto_devops (
    id integer NOT NULL,
    project_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean,
    deploy_strategy integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_auto_devops_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_auto_devops_id_seq OWNED BY project_auto_devops.id;

CREATE TABLE project_ci_cd_settings (
    id integer NOT NULL,
    project_id integer NOT NULL,
    group_runners_enabled boolean DEFAULT true NOT NULL,
    merge_pipelines_enabled boolean,
    default_git_depth integer,
    forward_deployment_enabled boolean,
    merge_trains_enabled boolean DEFAULT false,
    auto_rollback_enabled boolean DEFAULT false NOT NULL,
    keep_latest_artifact boolean DEFAULT true NOT NULL,
    restrict_user_defined_variables boolean DEFAULT false NOT NULL,
    job_token_scope_enabled boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE project_ci_cd_settings_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_ci_cd_settings_id_seq OWNED BY project_ci_cd_settings.id;

CREATE TABLE project_ci_feature_usages (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    feature smallint NOT NULL,
    default_branch boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE project_ci_feature_usages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_ci_feature_usages_id_seq OWNED BY project_ci_feature_usages.id;

CREATE TABLE project_compliance_framework_settings (
    project_id bigint NOT NULL,
    framework_id bigint,
    CONSTRAINT check_d348de9e2d CHECK ((framework_id IS NOT NULL))
);

CREATE SEQUENCE project_compliance_framework_settings_project_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_compliance_framework_settings_project_id_seq OWNED BY project_compliance_framework_settings.project_id;

CREATE TABLE project_custom_attributes (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE project_custom_attributes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_custom_attributes_id_seq OWNED BY project_custom_attributes.id;

CREATE TABLE project_daily_statistics (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    fetch_count integer NOT NULL,
    date date
);

CREATE SEQUENCE project_daily_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_daily_statistics_id_seq OWNED BY project_daily_statistics.id;

CREATE TABLE project_deploy_tokens (
    id integer NOT NULL,
    project_id integer NOT NULL,
    deploy_token_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_deploy_tokens_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_deploy_tokens_id_seq OWNED BY project_deploy_tokens.id;

CREATE TABLE project_error_tracking_settings (
    project_id integer NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    api_url character varying,
    encrypted_token character varying,
    encrypted_token_iv character varying,
    project_name character varying,
    organization_name character varying,
    integrated boolean DEFAULT true NOT NULL
);

CREATE TABLE project_export_jobs (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    status smallint DEFAULT 0 NOT NULL,
    jid character varying(100) NOT NULL
);

CREATE SEQUENCE project_export_jobs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_export_jobs_id_seq OWNED BY project_export_jobs.id;

CREATE TABLE project_feature_usages (
    project_id integer NOT NULL,
    jira_dvcs_cloud_last_sync_at timestamp without time zone,
    jira_dvcs_server_last_sync_at timestamp without time zone
);

CREATE TABLE project_features (
    id integer NOT NULL,
    project_id integer NOT NULL,
    merge_requests_access_level integer,
    issues_access_level integer,
    wiki_access_level integer,
    snippets_access_level integer DEFAULT 20 NOT NULL,
    builds_access_level integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    repository_access_level integer DEFAULT 20 NOT NULL,
    pages_access_level integer NOT NULL,
    forking_access_level integer,
    metrics_dashboard_access_level integer,
    analytics_access_level integer DEFAULT 20 NOT NULL,
    requirements_access_level integer DEFAULT 20 NOT NULL,
    operations_access_level integer DEFAULT 20 NOT NULL,
    security_and_compliance_access_level integer DEFAULT 10 NOT NULL,
    container_registry_access_level integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_features_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_features_id_seq OWNED BY project_features.id;

CREATE TABLE project_group_links (
    id integer NOT NULL,
    project_id integer NOT NULL,
    group_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    group_access integer DEFAULT 30 NOT NULL,
    expires_at date
);

CREATE SEQUENCE project_group_links_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_group_links_id_seq OWNED BY project_group_links.id;

CREATE TABLE project_import_data (
    id integer NOT NULL,
    project_id integer,
    data text,
    encrypted_credentials text,
    encrypted_credentials_iv character varying,
    encrypted_credentials_salt character varying
);

CREATE SEQUENCE project_import_data_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_import_data_id_seq OWNED BY project_import_data.id;

CREATE TABLE project_incident_management_settings (
    project_id integer NOT NULL,
    create_issue boolean DEFAULT false NOT NULL,
    send_email boolean DEFAULT false NOT NULL,
    issue_template_key text,
    pagerduty_active boolean DEFAULT false NOT NULL,
    encrypted_pagerduty_token bytea,
    encrypted_pagerduty_token_iv bytea,
    auto_close_incident boolean DEFAULT true NOT NULL,
    sla_timer boolean DEFAULT false,
    sla_timer_minutes integer,
    CONSTRAINT pagerduty_token_iv_length_constraint CHECK ((octet_length(encrypted_pagerduty_token_iv) <= 12)),
    CONSTRAINT pagerduty_token_length_constraint CHECK ((octet_length(encrypted_pagerduty_token) <= 255))
);

CREATE TABLE project_metrics_settings (
    project_id integer NOT NULL,
    external_dashboard_url character varying,
    dashboard_timezone smallint DEFAULT 0 NOT NULL
);

CREATE TABLE project_mirror_data (
    id integer NOT NULL,
    project_id integer NOT NULL,
    status character varying,
    jid character varying,
    last_error text,
    last_successful_update_at timestamp with time zone,
    last_update_at timestamp with time zone,
    last_update_scheduled_at timestamp without time zone,
    last_update_started_at timestamp without time zone,
    next_execution_timestamp timestamp without time zone,
    retry_count integer DEFAULT 0 NOT NULL,
    correlation_id_value character varying(128)
);

CREATE SEQUENCE project_mirror_data_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_mirror_data_id_seq OWNED BY project_mirror_data.id;

CREATE TABLE project_pages_metadata (
    project_id bigint NOT NULL,
    deployed boolean DEFAULT false NOT NULL,
    artifacts_archive_id bigint,
    pages_deployment_id bigint
);

CREATE TABLE project_repositories (
    id bigint NOT NULL,
    shard_id integer NOT NULL,
    disk_path character varying NOT NULL,
    project_id integer NOT NULL
);

CREATE SEQUENCE project_repositories_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repositories_id_seq OWNED BY project_repositories.id;

CREATE TABLE project_repository_states (
    id integer NOT NULL,
    project_id integer NOT NULL,
    repository_verification_checksum bytea,
    wiki_verification_checksum bytea,
    last_repository_verification_failure character varying,
    last_wiki_verification_failure character varying,
    repository_retry_at timestamp with time zone,
    wiki_retry_at timestamp with time zone,
    repository_retry_count integer,
    wiki_retry_count integer,
    last_repository_verification_ran_at timestamp with time zone,
    last_wiki_verification_ran_at timestamp with time zone
);

CREATE SEQUENCE project_repository_states_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repository_states_id_seq OWNED BY project_repository_states.id;

CREATE TABLE project_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    CONSTRAINT project_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT project_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE project_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_repository_storage_moves_id_seq OWNED BY project_repository_storage_moves.id;

CREATE TABLE project_security_settings (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    auto_fix_container_scanning boolean DEFAULT true NOT NULL,
    auto_fix_dast boolean DEFAULT true NOT NULL,
    auto_fix_dependency_scanning boolean DEFAULT true NOT NULL,
    auto_fix_sast boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE project_security_settings_project_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_security_settings_project_id_seq OWNED BY project_security_settings.project_id;

CREATE TABLE project_settings (
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    show_default_award_emojis boolean DEFAULT true NOT NULL,
    allow_merge_on_skipped_pipeline boolean,
    push_rule_id bigint,
    squash_option smallint DEFAULT 3,
    has_confluence boolean DEFAULT false NOT NULL,
    cve_id_request_enabled boolean DEFAULT true NOT NULL,
    has_vulnerabilities boolean DEFAULT false NOT NULL,
    prevent_merge_without_jira_issue boolean DEFAULT false NOT NULL,
    mr_default_target_self boolean DEFAULT false NOT NULL,
    previous_default_branch text,
    warn_about_potentially_unwanted_characters boolean DEFAULT true NOT NULL,
    merge_commit_template text,
    has_shimo boolean DEFAULT false NOT NULL,
    squash_commit_template text,
    CONSTRAINT check_3a03e7557a CHECK ((char_length(previous_default_branch) <= 4096)),
    CONSTRAINT check_b09644994b CHECK ((char_length(squash_commit_template) <= 500)),
    CONSTRAINT check_eaf7cfb6a7 CHECK ((char_length(merge_commit_template) <= 500))
);

CREATE TABLE project_statistics (
    id integer NOT NULL,
    project_id integer NOT NULL,
    namespace_id integer NOT NULL,
    commit_count bigint DEFAULT 0 NOT NULL,
    storage_size bigint DEFAULT 0 NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    lfs_objects_size bigint DEFAULT 0 NOT NULL,
    build_artifacts_size bigint DEFAULT 0 NOT NULL,
    shared_runners_seconds bigint DEFAULT 0 NOT NULL,
    shared_runners_seconds_last_reset timestamp without time zone,
    packages_size bigint DEFAULT 0 NOT NULL,
    wiki_size bigint,
    snippets_size bigint,
    pipeline_artifacts_size bigint DEFAULT 0 NOT NULL,
    uploads_size bigint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE project_statistics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_statistics_id_seq OWNED BY project_statistics.id;

CREATE TABLE project_topics (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    topic_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE project_topics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_topics_id_seq OWNED BY project_topics.id;

CREATE TABLE project_tracing_settings (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    external_url character varying NOT NULL
);

CREATE SEQUENCE project_tracing_settings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE project_tracing_settings_id_seq OWNED BY project_tracing_settings.id;

CREATE TABLE projects (
    id integer NOT NULL,
    name character varying,
    path character varying,
    description text,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    creator_id integer,
    namespace_id integer NOT NULL,
    last_activity_at timestamp without time zone,
    import_url character varying,
    visibility_level integer DEFAULT 0 NOT NULL,
    archived boolean DEFAULT false NOT NULL,
    avatar character varying,
    star_count integer DEFAULT 0 NOT NULL,
    merge_requests_rebase_enabled boolean DEFAULT false,
    import_type character varying,
    import_source character varying,
    merge_requests_ff_only_enabled boolean DEFAULT false NOT NULL,
    shared_runners_enabled boolean DEFAULT true NOT NULL,
    runners_token character varying,
    build_coverage_regex character varying,
    build_allow_git_fetch boolean DEFAULT true NOT NULL,
    build_timeout integer DEFAULT 3600 NOT NULL,
    pending_delete boolean DEFAULT false,
    public_builds boolean DEFAULT true NOT NULL,
    last_repository_check_failed boolean,
    last_repository_check_at timestamp without time zone,
    only_allow_merge_if_pipeline_succeeds boolean DEFAULT false NOT NULL,
    has_external_issue_tracker boolean,
    repository_storage character varying DEFAULT 'default'::character varying NOT NULL,
    repository_read_only boolean,
    request_access_enabled boolean DEFAULT true NOT NULL,
    has_external_wiki boolean,
    ci_config_path character varying,
    lfs_enabled boolean,
    description_html text,
    only_allow_merge_if_all_discussions_are_resolved boolean,
    printing_merge_request_link_enabled boolean DEFAULT true NOT NULL,
    auto_cancel_pending_pipelines integer DEFAULT 1 NOT NULL,
    cached_markdown_version integer,
    delete_error text,
    last_repository_updated_at timestamp without time zone,
    storage_version smallint,
    resolve_outdated_diff_discussions boolean,
    external_authorization_classification_label character varying,
    jobs_cache_index integer,
    pages_https_only boolean DEFAULT true,
    remote_mirror_available_overridden boolean,
    pool_repository_id bigint,
    runners_token_encrypted character varying,
    bfg_object_map character varying,
    detected_repository_languages boolean,
    disable_overriding_approvers_per_merge_request boolean,
    external_webhook_token character varying,
    issues_template text,
    merge_requests_author_approval boolean DEFAULT false,
    merge_requests_disable_committers_approval boolean,
    merge_requests_template text,
    mirror_last_successful_update_at timestamp without time zone,
    mirror_last_update_at timestamp without time zone,
    mirror_overwrites_diverged_branches boolean,
    mirror_user_id integer,
    only_mirror_protected_branches boolean,
    packages_enabled boolean,
    pull_mirror_available_overridden boolean,
    repository_size_limit bigint,
    require_password_to_approve boolean,
    mirror boolean DEFAULT false NOT NULL,
    mirror_trigger_builds boolean DEFAULT false NOT NULL,
    reset_approvals_on_push boolean DEFAULT true,
    service_desk_enabled boolean DEFAULT true,
    approvals_before_merge integer DEFAULT 0 NOT NULL,
    emails_disabled boolean,
    max_pages_size integer,
    max_artifacts_size integer,
    pull_mirror_branch_prefix character varying(50),
    marked_for_deletion_at date,
    marked_for_deletion_by_user_id integer,
    remove_source_branch_after_merge boolean,
    suggestion_commit_message character varying(255),
    autoclose_referenced_issues boolean,
    project_namespace_id bigint
);

CREATE SEQUENCE projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_id_seq OWNED BY projects.id;

CREATE TABLE projects_sync_events (
    id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE SEQUENCE projects_sync_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE projects_sync_events_id_seq OWNED BY projects_sync_events.id;

CREATE TABLE prometheus_alert_events (
    id bigint NOT NULL,
    project_id integer NOT NULL,
    prometheus_alert_id integer NOT NULL,
    started_at timestamp with time zone NOT NULL,
    ended_at timestamp with time zone,
    status smallint,
    payload_key character varying
);

CREATE SEQUENCE prometheus_alert_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE prometheus_alert_events_id_seq OWNED BY prometheus_alert_events.id;

CREATE TABLE prometheus_alerts (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    threshold double precision NOT NULL,
    operator integer NOT NULL,
    environment_id integer NOT NULL,
    project_id integer NOT NULL,
    prometheus_metric_id integer NOT NULL,
    runbook_url text,
    CONSTRAINT check_cb76d7e629 CHECK ((char_length(runbook_url) <= 255))
);

CREATE SEQUENCE prometheus_alerts_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE prometheus_alerts_id_seq OWNED BY prometheus_alerts.id;

CREATE TABLE prometheus_metrics (
    id integer NOT NULL,
    project_id integer,
    title character varying NOT NULL,
    query character varying NOT NULL,
    y_label character varying NOT NULL,
    unit character varying NOT NULL,
    legend character varying,
    "group" integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    common boolean DEFAULT false NOT NULL,
    identifier character varying,
    dashboard_path text,
    CONSTRAINT check_0ad9f01463 CHECK ((char_length(dashboard_path) <= 2048))
);

CREATE SEQUENCE prometheus_metrics_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE prometheus_metrics_id_seq OWNED BY prometheus_metrics.id;

CREATE TABLE protected_branch_merge_access_levels (
    id integer NOT NULL,
    protected_branch_id integer NOT NULL,
    access_level integer DEFAULT 40,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    group_id integer,
    user_id integer
);

CREATE SEQUENCE protected_branch_merge_access_levels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_merge_access_levels_id_seq OWNED BY protected_branch_merge_access_levels.id;

CREATE TABLE protected_branch_push_access_levels (
    id integer NOT NULL,
    protected_branch_id integer NOT NULL,
    access_level integer DEFAULT 40,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    group_id integer,
    user_id integer,
    deploy_key_id integer
);

CREATE SEQUENCE protected_branch_push_access_levels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_push_access_levels_id_seq OWNED BY protected_branch_push_access_levels.id;

CREATE TABLE protected_branch_unprotect_access_levels (
    id integer NOT NULL,
    protected_branch_id integer NOT NULL,
    access_level integer DEFAULT 40,
    user_id integer,
    group_id integer
);

CREATE SEQUENCE protected_branch_unprotect_access_levels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branch_unprotect_access_levels_id_seq OWNED BY protected_branch_unprotect_access_levels.id;

CREATE TABLE protected_branches (
    id integer NOT NULL,
    project_id integer NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    code_owner_approval_required boolean DEFAULT false NOT NULL,
    allow_force_push boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE protected_branches_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_branches_id_seq OWNED BY protected_branches.id;

CREATE TABLE protected_environment_deploy_access_levels (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    access_level integer DEFAULT 40,
    protected_environment_id integer NOT NULL,
    user_id integer,
    group_id integer
);

CREATE SEQUENCE protected_environment_deploy_access_levels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_environment_deploy_access_levels_id_seq OWNED BY protected_environment_deploy_access_levels.id;

CREATE TABLE protected_environments (
    id integer NOT NULL,
    project_id integer,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    name character varying NOT NULL,
    group_id bigint,
    required_approval_count integer DEFAULT 0 NOT NULL,
    CONSTRAINT protected_environments_project_or_group_existence CHECK (((project_id IS NULL) <> (group_id IS NULL)))
);

CREATE SEQUENCE protected_environments_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_environments_id_seq OWNED BY protected_environments.id;

CREATE TABLE protected_tag_create_access_levels (
    id integer NOT NULL,
    protected_tag_id integer NOT NULL,
    access_level integer DEFAULT 40,
    user_id integer,
    group_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE protected_tag_create_access_levels_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_tag_create_access_levels_id_seq OWNED BY protected_tag_create_access_levels.id;

CREATE TABLE protected_tags (
    id integer NOT NULL,
    project_id integer NOT NULL,
    name character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE protected_tags_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE protected_tags_id_seq OWNED BY protected_tags.id;

CREATE TABLE push_event_payloads (
    commit_count bigint NOT NULL,
    action smallint NOT NULL,
    ref_type smallint NOT NULL,
    commit_from bytea,
    commit_to bytea,
    ref text,
    commit_title character varying(70),
    ref_count integer,
    event_id bigint NOT NULL
);

CREATE TABLE push_rules (
    id integer NOT NULL,
    force_push_regex character varying,
    delete_branch_regex character varying,
    commit_message_regex character varying,
    deny_delete_tag boolean,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    author_email_regex character varying,
    member_check boolean DEFAULT false NOT NULL,
    file_name_regex character varying,
    is_sample boolean DEFAULT false,
    max_file_size integer DEFAULT 0 NOT NULL,
    prevent_secrets boolean DEFAULT false NOT NULL,
    branch_name_regex character varying,
    reject_unsigned_commits boolean,
    commit_committer_check boolean,
    regexp_uses_re2 boolean DEFAULT true,
    commit_message_negative_regex character varying
);

CREATE SEQUENCE push_rules_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE push_rules_id_seq OWNED BY push_rules.id;

CREATE TABLE raw_usage_data (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    recorded_at timestamp with time zone NOT NULL,
    sent_at timestamp with time zone,
    payload jsonb NOT NULL,
    version_usage_data_id_value bigint
);

CREATE SEQUENCE raw_usage_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE raw_usage_data_id_seq OWNED BY raw_usage_data.id;

CREATE TABLE redirect_routes (
    id integer NOT NULL,
    source_id integer NOT NULL,
    source_type character varying NOT NULL,
    path character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE redirect_routes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE redirect_routes_id_seq OWNED BY redirect_routes.id;

CREATE TABLE release_links (
    id bigint NOT NULL,
    release_id integer NOT NULL,
    url character varying NOT NULL,
    name character varying NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    filepath character varying(128),
    link_type smallint DEFAULT 0
);

CREATE SEQUENCE release_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE release_links_id_seq OWNED BY release_links.id;

CREATE TABLE releases (
    id integer NOT NULL,
    tag character varying,
    description text,
    project_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    description_html text,
    cached_markdown_version integer,
    author_id integer,
    name character varying,
    sha character varying,
    released_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE releases_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE releases_id_seq OWNED BY releases.id;

CREATE TABLE remote_mirrors (
    id integer NOT NULL,
    project_id integer,
    url character varying,
    enabled boolean DEFAULT false,
    update_status character varying,
    last_update_at timestamp without time zone,
    last_successful_update_at timestamp without time zone,
    last_update_started_at timestamp without time zone,
    last_error character varying,
    only_protected_branches boolean DEFAULT false NOT NULL,
    remote_name character varying,
    encrypted_credentials text,
    encrypted_credentials_iv character varying,
    encrypted_credentials_salt character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    error_notification_sent boolean,
    keep_divergent_refs boolean
);

CREATE SEQUENCE remote_mirrors_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE remote_mirrors_id_seq OWNED BY remote_mirrors.id;

CREATE TABLE repository_languages (
    project_id integer NOT NULL,
    programming_language_id integer NOT NULL,
    share double precision NOT NULL
);

CREATE TABLE required_code_owners_sections (
    id bigint NOT NULL,
    protected_branch_id bigint NOT NULL,
    name text NOT NULL,
    CONSTRAINT check_e58d53741e CHECK ((char_length(name) <= 1024))
);

CREATE SEQUENCE required_code_owners_sections_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE required_code_owners_sections_id_seq OWNED BY required_code_owners_sections.id;

CREATE TABLE requirements (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    author_id integer,
    iid integer NOT NULL,
    cached_markdown_version integer,
    state smallint DEFAULT 1 NOT NULL,
    title character varying(255) NOT NULL,
    title_html text,
    description text,
    description_html text,
    issue_id bigint,
    CONSTRAINT check_785ae25b9d CHECK ((char_length(description) <= 10000))
);

CREATE SEQUENCE requirements_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE requirements_id_seq OWNED BY requirements.id;

CREATE TABLE requirements_management_test_reports (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    requirement_id bigint,
    author_id bigint,
    state smallint NOT NULL,
    build_id bigint,
    issue_id bigint
);

CREATE SEQUENCE requirements_management_test_reports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE requirements_management_test_reports_id_seq OWNED BY requirements_management_test_reports.id;

CREATE TABLE resource_iteration_events (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    issue_id bigint,
    merge_request_id bigint,
    iteration_id bigint,
    created_at timestamp with time zone NOT NULL,
    action smallint NOT NULL
);

CREATE SEQUENCE resource_iteration_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_iteration_events_id_seq OWNED BY resource_iteration_events.id;

CREATE TABLE resource_label_events (
    id bigint NOT NULL,
    action integer NOT NULL,
    issue_id integer,
    merge_request_id integer,
    label_id integer,
    user_id integer,
    created_at timestamp with time zone NOT NULL,
    cached_markdown_version integer,
    reference text,
    reference_html text,
    epic_id integer
);

CREATE SEQUENCE resource_label_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_label_events_id_seq OWNED BY resource_label_events.id;

CREATE TABLE resource_milestone_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint,
    merge_request_id bigint,
    milestone_id bigint,
    action smallint NOT NULL,
    state smallint NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE resource_milestone_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_milestone_events_id_seq OWNED BY resource_milestone_events.id;

CREATE TABLE resource_state_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint,
    merge_request_id bigint,
    created_at timestamp with time zone NOT NULL,
    state smallint NOT NULL,
    epic_id integer,
    source_commit text,
    close_after_error_tracking_resolve boolean DEFAULT false NOT NULL,
    close_auto_resolve_prometheus_alert boolean DEFAULT false NOT NULL,
    source_merge_request_id bigint,
    CONSTRAINT check_f0bcfaa3a2 CHECK ((char_length(source_commit) <= 40)),
    CONSTRAINT state_events_must_belong_to_issue_or_merge_request_or_epic CHECK ((((issue_id <> NULL::bigint) AND (merge_request_id IS NULL) AND (epic_id IS NULL)) OR ((issue_id IS NULL) AND (merge_request_id <> NULL::bigint) AND (epic_id IS NULL)) OR ((issue_id IS NULL) AND (merge_request_id IS NULL) AND (epic_id <> NULL::integer))))
);

CREATE SEQUENCE resource_state_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_state_events_id_seq OWNED BY resource_state_events.id;

CREATE TABLE resource_weight_events (
    id bigint NOT NULL,
    user_id bigint,
    issue_id bigint NOT NULL,
    weight integer,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE resource_weight_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE resource_weight_events_id_seq OWNED BY resource_weight_events.id;

CREATE TABLE reviews (
    id bigint NOT NULL,
    author_id integer,
    merge_request_id integer NOT NULL,
    project_id integer NOT NULL,
    created_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE reviews_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE reviews_id_seq OWNED BY reviews.id;

CREATE TABLE routes (
    id integer NOT NULL,
    source_id integer NOT NULL,
    source_type character varying NOT NULL,
    path character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying
);

CREATE SEQUENCE routes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE routes_id_seq OWNED BY routes.id;

CREATE TABLE saml_group_links (
    id bigint NOT NULL,
    access_level smallint NOT NULL,
    group_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    saml_group_name text NOT NULL,
    CONSTRAINT check_1b3fc49d1e CHECK ((char_length(saml_group_name) <= 255))
);

CREATE SEQUENCE saml_group_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saml_group_links_id_seq OWNED BY saml_group_links.id;

CREATE TABLE saml_providers (
    id integer NOT NULL,
    group_id integer NOT NULL,
    enabled boolean NOT NULL,
    certificate_fingerprint character varying NOT NULL,
    sso_url character varying NOT NULL,
    enforced_sso boolean DEFAULT false NOT NULL,
    enforced_group_managed_accounts boolean DEFAULT false NOT NULL,
    prohibited_outer_forks boolean DEFAULT true NOT NULL,
    default_membership_role smallint DEFAULT 10 NOT NULL,
    git_check_enforced boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE saml_providers_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE saml_providers_id_seq OWNED BY saml_providers.id;

CREATE TABLE scim_identities (
    id bigint NOT NULL,
    group_id bigint NOT NULL,
    user_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    active boolean DEFAULT false,
    extern_uid character varying(255) NOT NULL
);

CREATE SEQUENCE scim_identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scim_identities_id_seq OWNED BY scim_identities.id;

CREATE TABLE scim_oauth_access_tokens (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    group_id integer NOT NULL,
    token_encrypted character varying NOT NULL
);

CREATE SEQUENCE scim_oauth_access_tokens_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE scim_oauth_access_tokens_id_seq OWNED BY scim_oauth_access_tokens.id;

CREATE TABLE security_findings (
    id bigint NOT NULL,
    scan_id bigint NOT NULL,
    scanner_id bigint NOT NULL,
    severity smallint NOT NULL,
    confidence smallint NOT NULL,
    project_fingerprint text,
    deduplicated boolean DEFAULT false NOT NULL,
    "position" integer,
    uuid uuid,
    overridden_uuid uuid,
    CONSTRAINT check_b9508c6df8 CHECK ((char_length(project_fingerprint) <= 40))
);

CREATE SEQUENCE security_findings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_findings_id_seq OWNED BY security_findings.id;

CREATE TABLE security_orchestration_policy_configurations (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    security_policy_management_project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    configured_at timestamp with time zone
);

COMMENT ON TABLE security_orchestration_policy_configurations IS '{"owner":"group::container security","description":"Configuration used to store relationship between project and security policy repository"}';

CREATE SEQUENCE security_orchestration_policy_configurations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_orchestration_policy_configurations_id_seq OWNED BY security_orchestration_policy_configurations.id;

CREATE TABLE security_orchestration_policy_rule_schedules (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    next_run_at timestamp with time zone,
    security_orchestration_policy_configuration_id bigint NOT NULL,
    user_id bigint NOT NULL,
    policy_index integer NOT NULL,
    cron text NOT NULL,
    rule_index integer DEFAULT 0 NOT NULL,
    CONSTRAINT check_915825a76e CHECK ((char_length(cron) <= 255))
);

COMMENT ON TABLE security_orchestration_policy_rule_schedules IS '{"owner":"group::container security","description":"Schedules used to store relationship between project and security policy repository"}';

CREATE SEQUENCE security_orchestration_policy_rule_schedules_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_orchestration_policy_rule_schedules_id_seq OWNED BY security_orchestration_policy_rule_schedules.id;

CREATE TABLE security_scans (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    build_id bigint NOT NULL,
    scan_type smallint NOT NULL,
    info jsonb DEFAULT '{}'::jsonb NOT NULL,
    project_id bigint,
    pipeline_id bigint,
    latest boolean DEFAULT true NOT NULL,
    status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE security_scans_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE security_scans_id_seq OWNED BY security_scans.id;

CREATE TABLE self_managed_prometheus_alert_events (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    environment_id bigint,
    started_at timestamp with time zone NOT NULL,
    ended_at timestamp with time zone,
    status smallint NOT NULL,
    title character varying(255) NOT NULL,
    query_expression character varying(255),
    payload_key character varying(255) NOT NULL
);

CREATE SEQUENCE self_managed_prometheus_alert_events_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE self_managed_prometheus_alert_events_id_seq OWNED BY self_managed_prometheus_alert_events.id;

CREATE TABLE sent_notifications (
    id integer NOT NULL,
    project_id integer,
    noteable_type character varying,
    noteable_id integer,
    recipient_id integer,
    commit_id character varying,
    reply_key character varying NOT NULL,
    line_code character varying,
    note_type character varying,
    "position" text,
    in_reply_to_discussion_id character varying
);

CREATE SEQUENCE sent_notifications_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sent_notifications_id_seq OWNED BY sent_notifications.id;

CREATE TABLE sentry_issues (
    id bigint NOT NULL,
    issue_id bigint NOT NULL,
    sentry_issue_identifier bigint NOT NULL
);

CREATE SEQUENCE sentry_issues_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sentry_issues_id_seq OWNED BY sentry_issues.id;

CREATE TABLE serverless_domain_cluster (
    pages_domain_id bigint NOT NULL,
    clusters_applications_knative_id bigint NOT NULL,
    creator_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    uuid character varying(14) NOT NULL,
    encrypted_key text,
    encrypted_key_iv character varying(255),
    certificate text
);

CREATE TABLE service_desk_settings (
    project_id bigint NOT NULL,
    issue_template_key character varying(255),
    outgoing_name character varying(255),
    project_key character varying(255),
    file_template_project_id bigint
);

CREATE TABLE shards (
    id integer NOT NULL,
    name character varying NOT NULL
);

CREATE SEQUENCE shards_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE shards_id_seq OWNED BY shards.id;

CREATE TABLE slack_integrations (
    id integer NOT NULL,
    service_id integer NOT NULL,
    team_id character varying NOT NULL,
    team_name character varying NOT NULL,
    alias character varying NOT NULL,
    user_id character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE slack_integrations_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE slack_integrations_id_seq OWNED BY slack_integrations.id;

CREATE TABLE smartcard_identities (
    id bigint NOT NULL,
    user_id integer NOT NULL,
    subject character varying NOT NULL,
    issuer character varying NOT NULL
);

CREATE SEQUENCE smartcard_identities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE smartcard_identities_id_seq OWNED BY smartcard_identities.id;

CREATE TABLE snippet_repositories (
    shard_id bigint NOT NULL,
    snippet_id bigint NOT NULL,
    disk_path character varying(80) NOT NULL,
    verification_retry_count smallint,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_checksum bytea,
    verification_failure text,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_started_at timestamp with time zone,
    CONSTRAINT snippet_repositories_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE TABLE snippet_repository_storage_moves (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    snippet_id bigint NOT NULL,
    state smallint DEFAULT 1 NOT NULL,
    source_storage_name text NOT NULL,
    destination_storage_name text NOT NULL,
    CONSTRAINT snippet_repository_storage_moves_destination_storage_name CHECK ((char_length(destination_storage_name) <= 255)),
    CONSTRAINT snippet_repository_storage_moves_source_storage_name CHECK ((char_length(source_storage_name) <= 255))
);

CREATE SEQUENCE snippet_repository_storage_moves_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippet_repository_storage_moves_id_seq OWNED BY snippet_repository_storage_moves.id;

CREATE TABLE snippet_statistics (
    snippet_id bigint NOT NULL,
    repository_size bigint DEFAULT 0 NOT NULL,
    file_count bigint DEFAULT 0 NOT NULL,
    commit_count bigint DEFAULT 0 NOT NULL
);

CREATE TABLE snippet_user_mentions (
    id bigint NOT NULL,
    snippet_id integer NOT NULL,
    note_id integer,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE snippet_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippet_user_mentions_id_seq OWNED BY snippet_user_mentions.id;

CREATE TABLE snippets (
    id integer NOT NULL,
    title character varying,
    content text,
    author_id integer NOT NULL,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    file_name character varying,
    type character varying,
    visibility_level integer DEFAULT 0 NOT NULL,
    title_html text,
    content_html text,
    cached_markdown_version integer,
    description text,
    description_html text,
    encrypted_secret_token character varying(255),
    encrypted_secret_token_iv character varying(255),
    secret boolean DEFAULT false NOT NULL,
    repository_read_only boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE snippets_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE snippets_id_seq OWNED BY snippets.id;

CREATE TABLE software_license_policies (
    id integer NOT NULL,
    project_id integer NOT NULL,
    software_license_id integer NOT NULL,
    classification integer DEFAULT 0,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_8e8751b568 CHECK ((classification IS NOT NULL))
);

CREATE SEQUENCE software_license_policies_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE software_license_policies_id_seq OWNED BY software_license_policies.id;

CREATE TABLE software_licenses (
    id integer NOT NULL,
    name character varying NOT NULL,
    spdx_identifier character varying(255)
);

CREATE SEQUENCE software_licenses_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE software_licenses_id_seq OWNED BY software_licenses.id;

CREATE TABLE spam_logs (
    id integer NOT NULL,
    user_id integer,
    source_ip character varying,
    user_agent character varying,
    via_api boolean,
    noteable_type character varying,
    title character varying,
    description text,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    submitted_as_ham boolean DEFAULT false NOT NULL,
    recaptcha_verified boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE spam_logs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE spam_logs_id_seq OWNED BY spam_logs.id;

CREATE TABLE sprints (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    due_date date,
    project_id bigint,
    group_id bigint,
    iid integer NOT NULL,
    cached_markdown_version integer,
    title text NOT NULL,
    title_html text,
    description text,
    description_html text,
    state_enum smallint DEFAULT 1 NOT NULL,
    iterations_cadence_id integer,
    sequence integer,
    CONSTRAINT sprints_must_belong_to_project_or_group CHECK ((((project_id <> NULL::bigint) AND (group_id IS NULL)) OR ((group_id <> NULL::bigint) AND (project_id IS NULL)))),
    CONSTRAINT sprints_title CHECK ((char_length(title) <= 255))
);

CREATE SEQUENCE sprints_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE sprints_id_seq OWNED BY sprints.id;

CREATE TABLE status_check_responses (
    id bigint NOT NULL,
    merge_request_id bigint NOT NULL,
    external_approval_rule_id bigint,
    sha bytea NOT NULL,
    external_status_check_id bigint NOT NULL
);

CREATE SEQUENCE status_check_responses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_check_responses_id_seq OWNED BY status_check_responses.id;

CREATE TABLE status_page_published_incidents (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_id bigint NOT NULL
);

CREATE SEQUENCE status_page_published_incidents_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_page_published_incidents_id_seq OWNED BY status_page_published_incidents.id;

CREATE TABLE status_page_settings (
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    enabled boolean DEFAULT false NOT NULL,
    aws_s3_bucket_name character varying(63) NOT NULL,
    aws_region character varying(255) NOT NULL,
    aws_access_key character varying(255) NOT NULL,
    encrypted_aws_secret_key character varying(255) NOT NULL,
    encrypted_aws_secret_key_iv character varying(255) NOT NULL,
    status_page_url text,
    CONSTRAINT check_75a79cd992 CHECK ((char_length(status_page_url) <= 1024))
);

CREATE SEQUENCE status_page_settings_project_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE status_page_settings_project_id_seq OWNED BY status_page_settings.project_id;

CREATE TABLE subscriptions (
    id integer NOT NULL,
    user_id integer,
    subscribable_type character varying,
    subscribable_id integer,
    subscribed boolean,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    project_id integer
);

CREATE SEQUENCE subscriptions_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE subscriptions_id_seq OWNED BY subscriptions.id;

CREATE TABLE suggestions (
    id bigint NOT NULL,
    note_id integer NOT NULL,
    relative_order smallint NOT NULL,
    applied boolean DEFAULT false NOT NULL,
    commit_id character varying,
    from_content text NOT NULL,
    to_content text NOT NULL,
    lines_above integer DEFAULT 0 NOT NULL,
    lines_below integer DEFAULT 0 NOT NULL,
    outdated boolean DEFAULT false NOT NULL
);

CREATE SEQUENCE suggestions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE suggestions_id_seq OWNED BY suggestions.id;

CREATE TABLE system_note_metadata (
    id integer NOT NULL,
    note_id integer NOT NULL,
    commit_count integer,
    action character varying,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    description_version_id bigint
);

CREATE SEQUENCE system_note_metadata_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE system_note_metadata_id_seq OWNED BY system_note_metadata.id;

CREATE TABLE taggings (
    tag_id integer,
    taggable_type character varying,
    tagger_id integer,
    tagger_type character varying,
    context character varying,
    created_at timestamp without time zone,
    id bigint NOT NULL,
    taggable_id bigint
);

CREATE SEQUENCE taggings_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE taggings_id_seq OWNED BY taggings.id;

CREATE TABLE tags (
    id integer NOT NULL,
    name character varying,
    taggings_count integer DEFAULT 0
);

CREATE SEQUENCE tags_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE tags_id_seq OWNED BY tags.id;

CREATE TABLE term_agreements (
    id integer NOT NULL,
    term_id integer NOT NULL,
    user_id integer NOT NULL,
    accepted boolean DEFAULT false NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE term_agreements_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE term_agreements_id_seq OWNED BY term_agreements.id;

CREATE TABLE terraform_state_versions (
    id bigint NOT NULL,
    terraform_state_id bigint NOT NULL,
    created_by_user_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    version integer NOT NULL,
    file_store smallint NOT NULL,
    file text NOT NULL,
    verification_retry_count smallint,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    verification_checksum bytea,
    verification_failure text,
    ci_build_id bigint,
    verification_started_at timestamp with time zone,
    verification_state smallint DEFAULT 0 NOT NULL,
    CONSTRAINT check_0824bb7bbd CHECK ((char_length(file) <= 255)),
    CONSTRAINT tf_state_versions_verification_failure_text_limit CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE terraform_state_versions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE terraform_state_versions_id_seq OWNED BY terraform_state_versions.id;

CREATE TABLE terraform_states (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    file character varying(255),
    lock_xid character varying(255),
    locked_at timestamp with time zone,
    locked_by_user_id bigint,
    uuid character varying(32) NOT NULL,
    name character varying(255) NOT NULL,
    versioning_enabled boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE terraform_states_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE terraform_states_id_seq OWNED BY terraform_states.id;

CREATE TABLE timelogs (
    id integer NOT NULL,
    time_spent integer NOT NULL,
    user_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    issue_id integer,
    merge_request_id integer,
    spent_at timestamp with time zone,
    note_id integer,
    project_id integer,
    summary text,
    CONSTRAINT check_271d321699 CHECK ((char_length(summary) <= 255))
);

CREATE SEQUENCE timelogs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE timelogs_id_seq OWNED BY timelogs.id;

CREATE TABLE todos (
    id integer NOT NULL,
    user_id integer NOT NULL,
    project_id integer,
    target_type character varying NOT NULL,
    target_id integer,
    author_id integer NOT NULL,
    action integer NOT NULL,
    state character varying NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    note_id integer,
    commit_id character varying,
    group_id integer,
    resolved_by_action smallint
);

CREATE SEQUENCE todos_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE todos_id_seq OWNED BY todos.id;

CREATE TABLE token_with_ivs (
    id bigint NOT NULL,
    hashed_token bytea NOT NULL,
    hashed_plaintext_token bytea NOT NULL,
    iv bytea NOT NULL
);

CREATE SEQUENCE token_with_ivs_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE token_with_ivs_id_seq OWNED BY token_with_ivs.id;

CREATE TABLE topics (
    id bigint NOT NULL,
    name text NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    avatar text,
    description text,
    total_projects_count bigint DEFAULT 0 NOT NULL,
    CONSTRAINT check_26753fb43a CHECK ((char_length(avatar) <= 255)),
    CONSTRAINT check_5d1a07c8c8 CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_7a90d4c757 CHECK ((char_length(name) <= 255))
);

CREATE SEQUENCE topics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE topics_id_seq OWNED BY topics.id;

CREATE TABLE trending_projects (
    id integer NOT NULL,
    project_id integer NOT NULL
);

CREATE SEQUENCE trending_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE trending_projects_id_seq OWNED BY trending_projects.id;

CREATE TABLE u2f_registrations (
    id integer NOT NULL,
    certificate text,
    key_handle character varying,
    public_key character varying,
    counter integer,
    user_id integer,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL,
    name character varying
);

CREATE SEQUENCE u2f_registrations_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE u2f_registrations_id_seq OWNED BY u2f_registrations.id;

CREATE TABLE upcoming_reconciliations (
    id bigint NOT NULL,
    namespace_id bigint,
    next_reconciliation_date date NOT NULL,
    display_alert_from date NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE upcoming_reconciliations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE upcoming_reconciliations_id_seq OWNED BY upcoming_reconciliations.id;

CREATE TABLE upload_states (
    verification_started_at timestamp with time zone,
    verification_retry_at timestamp with time zone,
    verified_at timestamp with time zone,
    upload_id bigint NOT NULL,
    verification_state smallint DEFAULT 0 NOT NULL,
    verification_retry_count smallint,
    verification_checksum bytea,
    verification_failure text,
    CONSTRAINT check_7396dc8591 CHECK ((char_length(verification_failure) <= 255))
);

CREATE SEQUENCE upload_states_upload_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE upload_states_upload_id_seq OWNED BY upload_states.upload_id;

CREATE TABLE uploads (
    id integer NOT NULL,
    size bigint NOT NULL,
    path character varying(511) NOT NULL,
    checksum character varying(64),
    model_type character varying,
    model_id integer,
    uploader character varying NOT NULL,
    created_at timestamp without time zone NOT NULL,
    store integer DEFAULT 1,
    mount_point character varying,
    secret character varying,
    CONSTRAINT check_5e9547379c CHECK ((store IS NOT NULL))
);

CREATE SEQUENCE uploads_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE uploads_id_seq OWNED BY uploads.id;

CREATE TABLE user_agent_details (
    id integer NOT NULL,
    user_agent character varying NOT NULL,
    ip_address character varying NOT NULL,
    subject_id integer NOT NULL,
    subject_type character varying NOT NULL,
    submitted boolean DEFAULT false NOT NULL,
    created_at timestamp without time zone NOT NULL,
    updated_at timestamp without time zone NOT NULL
);

CREATE SEQUENCE user_agent_details_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_agent_details_id_seq OWNED BY user_agent_details.id;

CREATE TABLE user_callouts (
    id integer NOT NULL,
    feature_name integer NOT NULL,
    user_id integer NOT NULL,
    dismissed_at timestamp with time zone
);

CREATE SEQUENCE user_callouts_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_callouts_id_seq OWNED BY user_callouts.id;

CREATE TABLE user_canonical_emails (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    canonical_email character varying NOT NULL
);

CREATE SEQUENCE user_canonical_emails_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_canonical_emails_id_seq OWNED BY user_canonical_emails.id;

CREATE TABLE user_credit_card_validations (
    user_id bigint NOT NULL,
    credit_card_validated_at timestamp with time zone NOT NULL,
    expiration_date date,
    last_digits smallint,
    holder_name text,
    network text,
    CONSTRAINT check_1765e2b30f CHECK ((char_length(network) <= 32)),
    CONSTRAINT check_3eea080c91 CHECK (((last_digits >= 0) AND (last_digits <= 9999))),
    CONSTRAINT check_eafe45d88b CHECK ((char_length(holder_name) <= 26))
);

CREATE TABLE user_custom_attributes (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id integer NOT NULL,
    key character varying NOT NULL,
    value character varying NOT NULL
);

CREATE SEQUENCE user_custom_attributes_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_custom_attributes_id_seq OWNED BY user_custom_attributes.id;

CREATE TABLE user_details (
    user_id bigint NOT NULL,
    job_title character varying(200) DEFAULT ''::character varying NOT NULL,
    bio character varying(255) DEFAULT ''::character varying NOT NULL,
    webauthn_xid text,
    provisioned_by_group_id bigint,
    other_role text,
    pronouns text,
    pronunciation text,
    registration_objective smallint,
    phone text,
    CONSTRAINT check_245664af82 CHECK ((char_length(webauthn_xid) <= 100)),
    CONSTRAINT check_a73b398c60 CHECK ((char_length(phone) <= 32)),
    CONSTRAINT check_b132136b01 CHECK ((char_length(other_role) <= 100)),
    CONSTRAINT check_eeeaf8d4f0 CHECK ((char_length(pronouns) <= 50)),
    CONSTRAINT check_f932ed37db CHECK ((char_length(pronunciation) <= 255))
);

COMMENT ON COLUMN user_details.phone IS 'JiHu-specific column';

CREATE SEQUENCE user_details_user_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_details_user_id_seq OWNED BY user_details.user_id;

CREATE TABLE user_follow_users (
    follower_id integer NOT NULL,
    followee_id integer NOT NULL
);

CREATE TABLE user_group_callouts (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    group_id bigint NOT NULL,
    feature_name smallint NOT NULL,
    dismissed_at timestamp with time zone
);

CREATE SEQUENCE user_group_callouts_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_group_callouts_id_seq OWNED BY user_group_callouts.id;

CREATE TABLE user_highest_roles (
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    highest_access_level integer
);

CREATE TABLE user_interacted_projects (
    user_id integer NOT NULL,
    project_id integer NOT NULL
);

CREATE TABLE user_permission_export_uploads (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id bigint NOT NULL,
    file_store integer,
    status smallint DEFAULT 0 NOT NULL,
    file text,
    CONSTRAINT check_1956806648 CHECK ((char_length(file) <= 255))
);

CREATE SEQUENCE user_permission_export_uploads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_permission_export_uploads_id_seq OWNED BY user_permission_export_uploads.id;

CREATE TABLE user_preferences (
    id integer NOT NULL,
    user_id integer NOT NULL,
    issue_notes_filter smallint DEFAULT 0 NOT NULL,
    merge_request_notes_filter smallint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    first_day_of_week integer,
    issues_sort character varying,
    merge_requests_sort character varying,
    timezone character varying,
    time_display_relative boolean,
    time_format_in_24h boolean,
    epic_notes_filter smallint DEFAULT 0 NOT NULL,
    epics_sort character varying,
    roadmap_epics_state integer,
    roadmaps_sort character varying,
    projects_sort character varying(64),
    show_whitespace_in_diffs boolean DEFAULT true NOT NULL,
    setup_for_company boolean,
    sourcegraph_enabled boolean,
    render_whitespace_in_code boolean,
    tab_width smallint,
    experience_level smallint,
    view_diffs_file_by_file boolean DEFAULT false NOT NULL,
    gitpod_enabled boolean DEFAULT false NOT NULL,
    markdown_surround_selection boolean DEFAULT true NOT NULL
);

CREATE SEQUENCE user_preferences_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_preferences_id_seq OWNED BY user_preferences.id;

CREATE TABLE user_statuses (
    user_id integer NOT NULL,
    cached_markdown_version integer,
    emoji character varying DEFAULT 'speech_balloon'::character varying NOT NULL,
    message character varying(100),
    message_html character varying,
    availability smallint DEFAULT 0 NOT NULL,
    clear_status_at timestamp with time zone
);

CREATE SEQUENCE user_statuses_user_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_statuses_user_id_seq OWNED BY user_statuses.user_id;

CREATE TABLE user_synced_attributes_metadata (
    id integer NOT NULL,
    name_synced boolean DEFAULT false,
    email_synced boolean DEFAULT false,
    location_synced boolean DEFAULT false,
    user_id integer NOT NULL,
    provider character varying
);

CREATE SEQUENCE user_synced_attributes_metadata_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE user_synced_attributes_metadata_id_seq OWNED BY user_synced_attributes_metadata.id;

CREATE TABLE users (
    id integer NOT NULL,
    email character varying DEFAULT ''::character varying NOT NULL,
    encrypted_password character varying DEFAULT ''::character varying NOT NULL,
    reset_password_token character varying,
    reset_password_sent_at timestamp without time zone,
    remember_created_at timestamp without time zone,
    sign_in_count integer DEFAULT 0,
    current_sign_in_at timestamp without time zone,
    last_sign_in_at timestamp without time zone,
    current_sign_in_ip character varying,
    last_sign_in_ip character varying,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    name character varying,
    admin boolean DEFAULT false NOT NULL,
    projects_limit integer NOT NULL,
    skype character varying DEFAULT ''::character varying NOT NULL,
    linkedin character varying DEFAULT ''::character varying NOT NULL,
    twitter character varying DEFAULT ''::character varying NOT NULL,
    failed_attempts integer DEFAULT 0,
    locked_at timestamp without time zone,
    username character varying,
    can_create_group boolean DEFAULT true NOT NULL,
    can_create_team boolean DEFAULT true NOT NULL,
    state character varying,
    color_scheme_id integer DEFAULT 1 NOT NULL,
    password_expires_at timestamp without time zone,
    created_by_id integer,
    last_credential_check_at timestamp without time zone,
    avatar character varying,
    confirmation_token character varying,
    confirmed_at timestamp without time zone,
    confirmation_sent_at timestamp without time zone,
    unconfirmed_email character varying,
    hide_no_ssh_key boolean DEFAULT false,
    website_url character varying DEFAULT ''::character varying NOT NULL,
    notification_email character varying,
    hide_no_password boolean DEFAULT false,
    password_automatically_set boolean DEFAULT false,
    location character varying,
    encrypted_otp_secret character varying,
    encrypted_otp_secret_iv character varying,
    encrypted_otp_secret_salt character varying,
    otp_required_for_login boolean DEFAULT false NOT NULL,
    otp_backup_codes text,
    public_email character varying,
    dashboard integer DEFAULT 0,
    project_view integer DEFAULT 0,
    consumed_timestep integer,
    layout integer DEFAULT 0,
    hide_project_limit boolean DEFAULT false,
    unlock_token character varying,
    otp_grace_period_started_at timestamp without time zone,
    external boolean DEFAULT false,
    incoming_email_token character varying,
    organization character varying,
    require_two_factor_authentication_from_group boolean DEFAULT false NOT NULL,
    two_factor_grace_period integer DEFAULT 48 NOT NULL,
    last_activity_on date,
    notified_of_own_activity boolean,
    preferred_language character varying,
    theme_id smallint,
    include_private_contributions boolean,
    feed_token character varying,
    accepted_term_id integer,
    private_profile boolean DEFAULT false NOT NULL,
    commit_email character varying,
    auditor boolean DEFAULT false NOT NULL,
    admin_email_unsubscribed_at timestamp without time zone,
    email_opted_in boolean,
    email_opted_in_at timestamp without time zone,
    email_opted_in_ip character varying,
    email_opted_in_source_id integer,
    group_view integer,
    managing_group_id integer,
    note text,
    roadmap_layout smallint,
    static_object_token character varying(255),
    first_name character varying(255),
    last_name character varying(255),
    role smallint,
    user_type smallint,
    static_object_token_encrypted text,
    CONSTRAINT check_7bde697e8e CHECK ((char_length(static_object_token_encrypted) <= 255))
);

CREATE SEQUENCE users_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_id_seq OWNED BY users.id;

CREATE TABLE users_ops_dashboard_projects (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    user_id integer NOT NULL,
    project_id integer NOT NULL
);

CREATE SEQUENCE users_ops_dashboard_projects_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_ops_dashboard_projects_id_seq OWNED BY users_ops_dashboard_projects.id;

CREATE TABLE users_security_dashboard_projects (
    user_id bigint NOT NULL,
    project_id bigint NOT NULL
);

CREATE TABLE users_star_projects (
    id integer NOT NULL,
    project_id integer NOT NULL,
    user_id integer NOT NULL,
    created_at timestamp without time zone,
    updated_at timestamp without time zone
);

CREATE SEQUENCE users_star_projects_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_star_projects_id_seq OWNED BY users_star_projects.id;

CREATE TABLE users_statistics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    without_groups_and_projects integer DEFAULT 0 NOT NULL,
    with_highest_role_guest integer DEFAULT 0 NOT NULL,
    with_highest_role_reporter integer DEFAULT 0 NOT NULL,
    with_highest_role_developer integer DEFAULT 0 NOT NULL,
    with_highest_role_maintainer integer DEFAULT 0 NOT NULL,
    with_highest_role_owner integer DEFAULT 0 NOT NULL,
    bots integer DEFAULT 0 NOT NULL,
    blocked integer DEFAULT 0 NOT NULL,
    with_highest_role_minimal_access integer DEFAULT 0 NOT NULL
);

CREATE SEQUENCE users_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE users_statistics_id_seq OWNED BY users_statistics.id;

CREATE TABLE vulnerabilities (
    id bigint NOT NULL,
    milestone_id bigint,
    epic_id bigint,
    project_id bigint NOT NULL,
    author_id bigint NOT NULL,
    updated_by_id bigint,
    last_edited_by_id bigint,
    start_date_sourcing_milestone_id bigint,
    due_date_sourcing_milestone_id bigint,
    last_edited_at timestamp with time zone,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    start_date date,
    due_date date,
    state smallint DEFAULT 1 NOT NULL,
    severity smallint NOT NULL,
    confidence smallint NOT NULL,
    severity_overridden boolean DEFAULT false,
    confidence_overridden boolean DEFAULT false,
    title character varying(255) NOT NULL,
    title_html text,
    description text,
    description_html text,
    report_type smallint NOT NULL,
    cached_markdown_version integer,
    resolved_by_id bigint,
    resolved_at timestamp with time zone,
    confirmed_by_id bigint,
    confirmed_at timestamp with time zone,
    dismissed_at timestamp with time zone,
    dismissed_by_id bigint,
    resolved_on_default_branch boolean DEFAULT false NOT NULL,
    present_on_default_branch boolean DEFAULT true NOT NULL,
    detected_at timestamp with time zone DEFAULT now()
);

CREATE SEQUENCE vulnerabilities_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerabilities_id_seq OWNED BY vulnerabilities.id;

CREATE TABLE vulnerability_exports (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    started_at timestamp with time zone,
    finished_at timestamp with time zone,
    status character varying(255) NOT NULL,
    file character varying(255),
    project_id bigint,
    author_id bigint NOT NULL,
    file_store integer,
    format smallint DEFAULT 0 NOT NULL,
    group_id integer
);

CREATE SEQUENCE vulnerability_exports_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_exports_id_seq OWNED BY vulnerability_exports.id;

CREATE TABLE vulnerability_external_issue_links (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    author_id bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    link_type smallint DEFAULT 1 NOT NULL,
    external_type smallint DEFAULT 1 NOT NULL,
    external_project_key text NOT NULL,
    external_issue_key text NOT NULL,
    CONSTRAINT check_3200604f5e CHECK ((char_length(external_issue_key) <= 255)),
    CONSTRAINT check_68cffd19b0 CHECK ((char_length(external_project_key) <= 255))
);

CREATE SEQUENCE vulnerability_external_issue_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_external_issue_links_id_seq OWNED BY vulnerability_external_issue_links.id;

CREATE TABLE vulnerability_feedback (
    id integer NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    feedback_type smallint NOT NULL,
    category smallint NOT NULL,
    project_id integer NOT NULL,
    author_id integer NOT NULL,
    pipeline_id integer,
    issue_id integer,
    project_fingerprint character varying(40) NOT NULL,
    merge_request_id integer,
    comment_author_id integer,
    comment text,
    comment_timestamp timestamp with time zone,
    finding_uuid uuid,
    dismissal_reason smallint
);

CREATE SEQUENCE vulnerability_feedback_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_feedback_id_seq OWNED BY vulnerability_feedback.id;

CREATE TABLE vulnerability_finding_evidence_assets (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_id bigint NOT NULL,
    type text,
    name text,
    url text,
    CONSTRAINT check_5adf5d69de CHECK ((char_length(type) <= 2048)),
    CONSTRAINT check_839f29d7ca CHECK ((char_length(name) <= 2048)),
    CONSTRAINT check_9272d912c0 CHECK ((char_length(url) <= 2048))
);

CREATE SEQUENCE vulnerability_finding_evidence_assets_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_assets_id_seq OWNED BY vulnerability_finding_evidence_assets.id;

CREATE TABLE vulnerability_finding_evidence_headers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_request_id bigint,
    vulnerability_finding_evidence_response_id bigint,
    name text NOT NULL,
    value text NOT NULL,
    CONSTRAINT check_01d21e8d92 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_3f9011f903 CHECK ((char_length(value) <= 8192))
);

CREATE SEQUENCE vulnerability_finding_evidence_headers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_headers_id_seq OWNED BY vulnerability_finding_evidence_headers.id;

CREATE TABLE vulnerability_finding_evidence_requests (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_id bigint,
    method text,
    url text,
    body text,
    vulnerability_finding_evidence_supporting_message_id bigint,
    CONSTRAINT check_7e37f2d01a CHECK ((char_length(body) <= 2048)),
    CONSTRAINT check_8152fbb236 CHECK ((char_length(url) <= 2048)),
    CONSTRAINT check_d9d11300f4 CHECK ((char_length(method) <= 32))
);

CREATE SEQUENCE vulnerability_finding_evidence_requests_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_requests_id_seq OWNED BY vulnerability_finding_evidence_requests.id;

CREATE TABLE vulnerability_finding_evidence_responses (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_id bigint,
    status_code integer,
    reason_phrase text,
    body text,
    vulnerability_finding_evidence_supporting_message_id bigint,
    CONSTRAINT check_58b124ab48 CHECK ((char_length(reason_phrase) <= 2048)),
    CONSTRAINT check_76bac0c32b CHECK ((char_length(body) <= 2048))
);

CREATE SEQUENCE vulnerability_finding_evidence_responses_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_responses_id_seq OWNED BY vulnerability_finding_evidence_responses.id;

CREATE TABLE vulnerability_finding_evidence_sources (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_id bigint NOT NULL,
    name text,
    url text,
    CONSTRAINT check_0fe01298d6 CHECK ((char_length(url) <= 2048)),
    CONSTRAINT check_86b537ba1a CHECK ((char_length(name) <= 2048))
);

CREATE SEQUENCE vulnerability_finding_evidence_sources_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_sources_id_seq OWNED BY vulnerability_finding_evidence_sources.id;

CREATE TABLE vulnerability_finding_evidence_supporting_messages (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_finding_evidence_id bigint NOT NULL,
    name text,
    CONSTRAINT check_fa33b9ae85 CHECK ((char_length(name) <= 2048))
);

CREATE SEQUENCE vulnerability_finding_evidence_supporting_messages_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidence_supporting_messages_id_seq OWNED BY vulnerability_finding_evidence_supporting_messages.id;

CREATE TABLE vulnerability_finding_evidences (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_occurrence_id bigint NOT NULL,
    summary text,
    data jsonb DEFAULT '{}'::jsonb NOT NULL,
    CONSTRAINT check_5773b236fb CHECK ((char_length(summary) <= 8000000))
);

CREATE SEQUENCE vulnerability_finding_evidences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_evidences_id_seq OWNED BY vulnerability_finding_evidences.id;

CREATE TABLE vulnerability_finding_links (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_occurrence_id bigint NOT NULL,
    name text,
    url text NOT NULL,
    CONSTRAINT check_55f0a95439 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_b7fe886df6 CHECK ((char_length(url) <= 2048))
);

CREATE SEQUENCE vulnerability_finding_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_links_id_seq OWNED BY vulnerability_finding_links.id;

CREATE TABLE vulnerability_finding_signatures (
    id bigint NOT NULL,
    finding_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    algorithm_type smallint NOT NULL,
    signature_sha bytea NOT NULL
);

CREATE SEQUENCE vulnerability_finding_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_finding_signatures_id_seq OWNED BY vulnerability_finding_signatures.id;

CREATE TABLE vulnerability_findings_remediations (
    id bigint NOT NULL,
    vulnerability_occurrence_id bigint,
    vulnerability_remediation_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE vulnerability_findings_remediations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_findings_remediations_id_seq OWNED BY vulnerability_findings_remediations.id;

CREATE TABLE vulnerability_flags (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    vulnerability_occurrence_id bigint NOT NULL,
    flag_type smallint DEFAULT 0 NOT NULL,
    origin text NOT NULL,
    description text NOT NULL,
    CONSTRAINT check_45e743349f CHECK ((char_length(description) <= 1024)),
    CONSTRAINT check_49c1d00032 CHECK ((char_length(origin) <= 255))
);

CREATE SEQUENCE vulnerability_flags_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_flags_id_seq OWNED BY vulnerability_flags.id;

CREATE TABLE vulnerability_historical_statistics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    total integer DEFAULT 0 NOT NULL,
    critical integer DEFAULT 0 NOT NULL,
    high integer DEFAULT 0 NOT NULL,
    medium integer DEFAULT 0 NOT NULL,
    low integer DEFAULT 0 NOT NULL,
    unknown integer DEFAULT 0 NOT NULL,
    info integer DEFAULT 0 NOT NULL,
    date date NOT NULL,
    letter_grade smallint NOT NULL
);

CREATE SEQUENCE vulnerability_historical_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_historical_statistics_id_seq OWNED BY vulnerability_historical_statistics.id;

CREATE TABLE vulnerability_identifiers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    fingerprint bytea NOT NULL,
    external_type character varying NOT NULL,
    external_id character varying NOT NULL,
    name character varying NOT NULL,
    url text
);

CREATE SEQUENCE vulnerability_identifiers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_identifiers_id_seq OWNED BY vulnerability_identifiers.id;

CREATE TABLE vulnerability_issue_links (
    id bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    link_type smallint DEFAULT 1 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL
);

CREATE SEQUENCE vulnerability_issue_links_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_issue_links_id_seq OWNED BY vulnerability_issue_links.id;

CREATE TABLE vulnerability_occurrence_identifiers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    occurrence_id bigint NOT NULL,
    identifier_id bigint NOT NULL
);

CREATE SEQUENCE vulnerability_occurrence_identifiers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_occurrence_identifiers_id_seq OWNED BY vulnerability_occurrence_identifiers.id;

CREATE TABLE vulnerability_occurrence_pipelines (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    occurrence_id bigint NOT NULL,
    pipeline_id integer NOT NULL
);

CREATE SEQUENCE vulnerability_occurrence_pipelines_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_occurrence_pipelines_id_seq OWNED BY vulnerability_occurrence_pipelines.id;

CREATE TABLE vulnerability_occurrences (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    severity smallint NOT NULL,
    confidence smallint NOT NULL,
    report_type smallint NOT NULL,
    project_id integer NOT NULL,
    scanner_id bigint NOT NULL,
    primary_identifier_id bigint NOT NULL,
    project_fingerprint bytea NOT NULL,
    location_fingerprint bytea NOT NULL,
    uuid character varying(36) NOT NULL,
    name character varying NOT NULL,
    metadata_version character varying NOT NULL,
    raw_metadata text NOT NULL,
    vulnerability_id bigint,
    details jsonb DEFAULT '{}'::jsonb NOT NULL,
    description text,
    message text,
    solution text,
    cve text,
    location jsonb,
    detection_method smallint DEFAULT 0 NOT NULL,
    migrated_to_new_structure boolean DEFAULT false NOT NULL,
    CONSTRAINT check_4a3a60f2ba CHECK ((char_length(solution) <= 7000)),
    CONSTRAINT check_ade261da6b CHECK ((char_length(description) <= 15000)),
    CONSTRAINT check_df6dd20219 CHECK ((char_length(message) <= 3000)),
    CONSTRAINT check_f602da68dd CHECK ((char_length(cve) <= 48400))
);

CREATE SEQUENCE vulnerability_occurrences_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_occurrences_id_seq OWNED BY vulnerability_occurrences.id;

CREATE TABLE vulnerability_reads (
    id bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    project_id bigint NOT NULL,
    scanner_id bigint NOT NULL,
    report_type smallint NOT NULL,
    severity smallint NOT NULL,
    state smallint NOT NULL,
    has_issues boolean DEFAULT false NOT NULL,
    resolved_on_default_branch boolean DEFAULT false NOT NULL,
    uuid uuid NOT NULL,
    location_image text,
    CONSTRAINT check_380451bdbe CHECK ((char_length(location_image) <= 2048))
);

CREATE SEQUENCE vulnerability_reads_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_reads_id_seq OWNED BY vulnerability_reads.id;

CREATE TABLE vulnerability_remediations (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    file_store smallint,
    summary text NOT NULL,
    file text NOT NULL,
    checksum bytea NOT NULL,
    project_id bigint NOT NULL,
    CONSTRAINT check_ac0ccabff3 CHECK ((char_length(summary) <= 200)),
    CONSTRAINT check_fe3325e3ba CHECK ((char_length(file) <= 255))
);

COMMENT ON COLUMN vulnerability_remediations.checksum IS 'Stores the SHA256 checksum of the attached diff file';

CREATE SEQUENCE vulnerability_remediations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_remediations_id_seq OWNED BY vulnerability_remediations.id;

CREATE TABLE vulnerability_scanners (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id integer NOT NULL,
    external_id character varying NOT NULL,
    name character varying NOT NULL,
    vendor text DEFAULT 'GitLab'::text NOT NULL
);

CREATE SEQUENCE vulnerability_scanners_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_scanners_id_seq OWNED BY vulnerability_scanners.id;

CREATE TABLE vulnerability_statistics (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    total integer DEFAULT 0 NOT NULL,
    critical integer DEFAULT 0 NOT NULL,
    high integer DEFAULT 0 NOT NULL,
    medium integer DEFAULT 0 NOT NULL,
    low integer DEFAULT 0 NOT NULL,
    unknown integer DEFAULT 0 NOT NULL,
    info integer DEFAULT 0 NOT NULL,
    letter_grade smallint NOT NULL,
    latest_pipeline_id bigint
);

CREATE SEQUENCE vulnerability_statistics_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_statistics_id_seq OWNED BY vulnerability_statistics.id;

CREATE TABLE vulnerability_user_mentions (
    id bigint NOT NULL,
    vulnerability_id bigint NOT NULL,
    note_id integer,
    mentioned_users_ids integer[],
    mentioned_projects_ids integer[],
    mentioned_groups_ids integer[]
);

CREATE SEQUENCE vulnerability_user_mentions_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE vulnerability_user_mentions_id_seq OWNED BY vulnerability_user_mentions.id;

CREATE TABLE web_hooks (
    id integer NOT NULL,
    project_id integer,
    created_at timestamp without time zone,
    updated_at timestamp without time zone,
    type character varying DEFAULT 'ProjectHook'::character varying,
    service_id integer,
    push_events boolean DEFAULT true NOT NULL,
    issues_events boolean DEFAULT false NOT NULL,
    merge_requests_events boolean DEFAULT false NOT NULL,
    tag_push_events boolean DEFAULT false,
    note_events boolean DEFAULT false NOT NULL,
    enable_ssl_verification boolean DEFAULT true,
    wiki_page_events boolean DEFAULT false NOT NULL,
    pipeline_events boolean DEFAULT false NOT NULL,
    confidential_issues_events boolean DEFAULT false NOT NULL,
    repository_update_events boolean DEFAULT false NOT NULL,
    job_events boolean DEFAULT false NOT NULL,
    confidential_note_events boolean,
    push_events_branch_filter text,
    encrypted_token character varying,
    encrypted_token_iv character varying,
    encrypted_url character varying,
    encrypted_url_iv character varying,
    group_id integer,
    deployment_events boolean DEFAULT false NOT NULL,
    feature_flag_events boolean DEFAULT false NOT NULL,
    releases_events boolean DEFAULT false NOT NULL,
    member_events boolean DEFAULT false NOT NULL,
    subgroup_events boolean DEFAULT false NOT NULL,
    recent_failures smallint DEFAULT 0 NOT NULL,
    backoff_count smallint DEFAULT 0 NOT NULL,
    disabled_until timestamp with time zone
);

CREATE SEQUENCE web_hooks_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE web_hooks_id_seq OWNED BY web_hooks.id;

CREATE TABLE webauthn_registrations (
    id bigint NOT NULL,
    user_id bigint NOT NULL,
    counter bigint DEFAULT 0 NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    credential_xid text NOT NULL,
    name text NOT NULL,
    public_key text NOT NULL,
    u2f_registration_id integer,
    CONSTRAINT check_2f02e74321 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_e54008d9ce CHECK ((char_length(credential_xid) <= 340))
);

CREATE SEQUENCE webauthn_registrations_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE webauthn_registrations_id_seq OWNED BY webauthn_registrations.id;

CREATE TABLE wiki_page_meta (
    id integer NOT NULL,
    project_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    title character varying(255) NOT NULL
);

CREATE SEQUENCE wiki_page_meta_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE wiki_page_meta_id_seq OWNED BY wiki_page_meta.id;

CREATE TABLE wiki_page_slugs (
    id integer NOT NULL,
    canonical boolean DEFAULT false NOT NULL,
    wiki_page_meta_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    slug character varying(2048) NOT NULL
);

CREATE SEQUENCE wiki_page_slugs_id_seq
    AS integer
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE wiki_page_slugs_id_seq OWNED BY wiki_page_slugs.id;

CREATE TABLE work_item_types (
    id bigint NOT NULL,
    base_type smallint DEFAULT 0 NOT NULL,
    cached_markdown_version integer,
    name text NOT NULL,
    description text,
    description_html text,
    icon_name text,
    namespace_id bigint,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    CONSTRAINT check_104d2410f6 CHECK ((char_length(name) <= 255)),
    CONSTRAINT check_fecb3a98d1 CHECK ((char_length(icon_name) <= 255))
);

CREATE SEQUENCE work_item_types_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE work_item_types_id_seq OWNED BY work_item_types.id;

CREATE TABLE x509_certificates (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    subject_key_identifier character varying(255) NOT NULL,
    subject character varying(255) NOT NULL,
    email character varying(255) NOT NULL,
    serial_number bytea NOT NULL,
    certificate_status smallint DEFAULT 0 NOT NULL,
    x509_issuer_id bigint NOT NULL
);

CREATE SEQUENCE x509_certificates_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE x509_certificates_id_seq OWNED BY x509_certificates.id;

CREATE TABLE x509_commit_signatures (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    project_id bigint NOT NULL,
    x509_certificate_id bigint NOT NULL,
    commit_sha bytea NOT NULL,
    verification_status smallint DEFAULT 0 NOT NULL
);

CREATE SEQUENCE x509_commit_signatures_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE x509_commit_signatures_id_seq OWNED BY x509_commit_signatures.id;

CREATE TABLE x509_issuers (
    id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    subject_key_identifier character varying(255) NOT NULL,
    subject character varying(255) NOT NULL,
    crl_url character varying(255) NOT NULL
);

CREATE SEQUENCE x509_issuers_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE x509_issuers_id_seq OWNED BY x509_issuers.id;

CREATE TABLE zentao_tracker_data (
    id bigint NOT NULL,
    integration_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    encrypted_url bytea,
    encrypted_url_iv bytea,
    encrypted_api_url bytea,
    encrypted_api_url_iv bytea,
    encrypted_zentao_product_xid bytea,
    encrypted_zentao_product_xid_iv bytea,
    encrypted_api_token bytea,
    encrypted_api_token_iv bytea
);

CREATE SEQUENCE zentao_tracker_data_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE zentao_tracker_data_id_seq OWNED BY zentao_tracker_data.id;

CREATE TABLE zoom_meetings (
    id bigint NOT NULL,
    project_id bigint NOT NULL,
    issue_id bigint NOT NULL,
    created_at timestamp with time zone NOT NULL,
    updated_at timestamp with time zone NOT NULL,
    issue_status smallint DEFAULT 1 NOT NULL,
    url character varying(255)
);

CREATE SEQUENCE zoom_meetings_id_seq
    START WITH 1
    INCREMENT BY 1
    NO MINVALUE
    NO MAXVALUE
    CACHE 1;

ALTER SEQUENCE zoom_meetings_id_seq OWNED BY zoom_meetings.id;

ALTER TABLE ONLY abuse_reports ALTER COLUMN id SET DEFAULT nextval('abuse_reports_id_seq'::regclass);

ALTER TABLE ONLY agent_activity_events ALTER COLUMN id SET DEFAULT nextval('agent_activity_events_id_seq'::regclass);

ALTER TABLE ONLY agent_group_authorizations ALTER COLUMN id SET DEFAULT nextval('agent_group_authorizations_id_seq'::regclass);

ALTER TABLE ONLY agent_project_authorizations ALTER COLUMN id SET DEFAULT nextval('agent_project_authorizations_id_seq'::regclass);

ALTER TABLE ONLY alert_management_alert_assignees ALTER COLUMN id SET DEFAULT nextval('alert_management_alert_assignees_id_seq'::regclass);

ALTER TABLE ONLY alert_management_alert_user_mentions ALTER COLUMN id SET DEFAULT nextval('alert_management_alert_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY alert_management_alerts ALTER COLUMN id SET DEFAULT nextval('alert_management_alerts_id_seq'::regclass);

ALTER TABLE ONLY alert_management_http_integrations ALTER COLUMN id SET DEFAULT nextval('alert_management_http_integrations_id_seq'::regclass);

ALTER TABLE ONLY allowed_email_domains ALTER COLUMN id SET DEFAULT nextval('allowed_email_domains_id_seq'::regclass);

ALTER TABLE ONLY analytics_cycle_analytics_group_stages ALTER COLUMN id SET DEFAULT nextval('analytics_cycle_analytics_group_stages_id_seq'::regclass);

ALTER TABLE ONLY analytics_cycle_analytics_group_value_streams ALTER COLUMN id SET DEFAULT nextval('analytics_cycle_analytics_group_value_streams_id_seq'::regclass);

ALTER TABLE ONLY analytics_cycle_analytics_project_stages ALTER COLUMN id SET DEFAULT nextval('analytics_cycle_analytics_project_stages_id_seq'::regclass);

ALTER TABLE ONLY analytics_cycle_analytics_project_value_streams ALTER COLUMN id SET DEFAULT nextval('analytics_cycle_analytics_project_value_streams_id_seq'::regclass);

ALTER TABLE ONLY analytics_cycle_analytics_stage_event_hashes ALTER COLUMN id SET DEFAULT nextval('analytics_cycle_analytics_stage_event_hashes_id_seq'::regclass);

ALTER TABLE ONLY analytics_devops_adoption_segments ALTER COLUMN id SET DEFAULT nextval('analytics_devops_adoption_segments_id_seq'::regclass);

ALTER TABLE ONLY analytics_devops_adoption_snapshots ALTER COLUMN id SET DEFAULT nextval('analytics_devops_adoption_snapshots_id_seq'::regclass);

ALTER TABLE ONLY analytics_usage_trends_measurements ALTER COLUMN id SET DEFAULT nextval('analytics_usage_trends_measurements_id_seq'::regclass);

ALTER TABLE ONLY appearances ALTER COLUMN id SET DEFAULT nextval('appearances_id_seq'::regclass);

ALTER TABLE ONLY application_setting_terms ALTER COLUMN id SET DEFAULT nextval('application_setting_terms_id_seq'::regclass);

ALTER TABLE ONLY application_settings ALTER COLUMN id SET DEFAULT nextval('application_settings_id_seq'::regclass);

ALTER TABLE ONLY approval_merge_request_rule_sources ALTER COLUMN id SET DEFAULT nextval('approval_merge_request_rule_sources_id_seq'::regclass);

ALTER TABLE ONLY approval_merge_request_rules ALTER COLUMN id SET DEFAULT nextval('approval_merge_request_rules_id_seq'::regclass);

ALTER TABLE ONLY approval_merge_request_rules_approved_approvers ALTER COLUMN id SET DEFAULT nextval('approval_merge_request_rules_approved_approvers_id_seq'::regclass);

ALTER TABLE ONLY approval_merge_request_rules_groups ALTER COLUMN id SET DEFAULT nextval('approval_merge_request_rules_groups_id_seq'::regclass);

ALTER TABLE ONLY approval_merge_request_rules_users ALTER COLUMN id SET DEFAULT nextval('approval_merge_request_rules_users_id_seq'::regclass);

ALTER TABLE ONLY approval_project_rules ALTER COLUMN id SET DEFAULT nextval('approval_project_rules_id_seq'::regclass);

ALTER TABLE ONLY approval_project_rules_groups ALTER COLUMN id SET DEFAULT nextval('approval_project_rules_groups_id_seq'::regclass);

ALTER TABLE ONLY approval_project_rules_users ALTER COLUMN id SET DEFAULT nextval('approval_project_rules_users_id_seq'::regclass);

ALTER TABLE ONLY approvals ALTER COLUMN id SET DEFAULT nextval('approvals_id_seq'::regclass);

ALTER TABLE ONLY approver_groups ALTER COLUMN id SET DEFAULT nextval('approver_groups_id_seq'::regclass);

ALTER TABLE ONLY approvers ALTER COLUMN id SET DEFAULT nextval('approvers_id_seq'::regclass);

ALTER TABLE ONLY atlassian_identities ALTER COLUMN user_id SET DEFAULT nextval('atlassian_identities_user_id_seq'::regclass);

ALTER TABLE ONLY audit_events ALTER COLUMN id SET DEFAULT nextval('audit_events_id_seq'::regclass);

ALTER TABLE ONLY audit_events_external_audit_event_destinations ALTER COLUMN id SET DEFAULT nextval('audit_events_external_audit_event_destinations_id_seq'::regclass);

ALTER TABLE ONLY authentication_events ALTER COLUMN id SET DEFAULT nextval('authentication_events_id_seq'::regclass);

ALTER TABLE ONLY award_emoji ALTER COLUMN id SET DEFAULT nextval('award_emoji_id_seq'::regclass);

ALTER TABLE ONLY background_migration_jobs ALTER COLUMN id SET DEFAULT nextval('background_migration_jobs_id_seq'::regclass);

ALTER TABLE ONLY badges ALTER COLUMN id SET DEFAULT nextval('badges_id_seq'::regclass);

ALTER TABLE ONLY batched_background_migration_job_transition_logs ALTER COLUMN id SET DEFAULT nextval('batched_background_migration_job_transition_logs_id_seq'::regclass);

ALTER TABLE ONLY batched_background_migration_jobs ALTER COLUMN id SET DEFAULT nextval('batched_background_migration_jobs_id_seq'::regclass);

ALTER TABLE ONLY batched_background_migrations ALTER COLUMN id SET DEFAULT nextval('batched_background_migrations_id_seq'::regclass);

ALTER TABLE ONLY board_assignees ALTER COLUMN id SET DEFAULT nextval('board_assignees_id_seq'::regclass);

ALTER TABLE ONLY board_group_recent_visits ALTER COLUMN id SET DEFAULT nextval('board_group_recent_visits_id_seq'::regclass);

ALTER TABLE ONLY board_labels ALTER COLUMN id SET DEFAULT nextval('board_labels_id_seq'::regclass);

ALTER TABLE ONLY board_project_recent_visits ALTER COLUMN id SET DEFAULT nextval('board_project_recent_visits_id_seq'::regclass);

ALTER TABLE ONLY board_user_preferences ALTER COLUMN id SET DEFAULT nextval('board_user_preferences_id_seq'::regclass);

ALTER TABLE ONLY boards ALTER COLUMN id SET DEFAULT nextval('boards_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_board_labels ALTER COLUMN id SET DEFAULT nextval('boards_epic_board_labels_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_board_positions ALTER COLUMN id SET DEFAULT nextval('boards_epic_board_positions_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_board_recent_visits ALTER COLUMN id SET DEFAULT nextval('boards_epic_board_recent_visits_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_boards ALTER COLUMN id SET DEFAULT nextval('boards_epic_boards_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_list_user_preferences ALTER COLUMN id SET DEFAULT nextval('boards_epic_list_user_preferences_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_lists ALTER COLUMN id SET DEFAULT nextval('boards_epic_lists_id_seq'::regclass);

ALTER TABLE ONLY boards_epic_user_preferences ALTER COLUMN id SET DEFAULT nextval('boards_epic_user_preferences_id_seq'::regclass);

ALTER TABLE ONLY broadcast_messages ALTER COLUMN id SET DEFAULT nextval('broadcast_messages_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_configurations ALTER COLUMN id SET DEFAULT nextval('bulk_import_configurations_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_entities ALTER COLUMN id SET DEFAULT nextval('bulk_import_entities_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_export_uploads ALTER COLUMN id SET DEFAULT nextval('bulk_import_export_uploads_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_exports ALTER COLUMN id SET DEFAULT nextval('bulk_import_exports_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_failures ALTER COLUMN id SET DEFAULT nextval('bulk_import_failures_id_seq'::regclass);

ALTER TABLE ONLY bulk_import_trackers ALTER COLUMN id SET DEFAULT nextval('bulk_import_trackers_id_seq'::regclass);

ALTER TABLE ONLY bulk_imports ALTER COLUMN id SET DEFAULT nextval('bulk_imports_id_seq'::regclass);

ALTER TABLE ONLY chat_names ALTER COLUMN id SET DEFAULT nextval('chat_names_id_seq'::regclass);

ALTER TABLE ONLY chat_teams ALTER COLUMN id SET DEFAULT nextval('chat_teams_id_seq'::regclass);

ALTER TABLE ONLY ci_build_needs ALTER COLUMN id SET DEFAULT nextval('ci_build_needs_id_seq'::regclass);

ALTER TABLE ONLY ci_build_pending_states ALTER COLUMN id SET DEFAULT nextval('ci_build_pending_states_id_seq'::regclass);

ALTER TABLE ONLY ci_build_report_results ALTER COLUMN build_id SET DEFAULT nextval('ci_build_report_results_build_id_seq'::regclass);

ALTER TABLE ONLY ci_build_trace_chunks ALTER COLUMN id SET DEFAULT nextval('ci_build_trace_chunks_id_seq'::regclass);

ALTER TABLE ONLY ci_builds ALTER COLUMN id SET DEFAULT nextval('ci_builds_id_seq'::regclass);

ALTER TABLE ONLY ci_builds_metadata ALTER COLUMN id SET DEFAULT nextval('ci_builds_metadata_id_seq'::regclass);

ALTER TABLE ONLY ci_builds_runner_session ALTER COLUMN id SET DEFAULT nextval('ci_builds_runner_session_id_seq'::regclass);

ALTER TABLE ONLY ci_daily_build_group_report_results ALTER COLUMN id SET DEFAULT nextval('ci_daily_build_group_report_results_id_seq'::regclass);

ALTER TABLE ONLY ci_deleted_objects ALTER COLUMN id SET DEFAULT nextval('ci_deleted_objects_id_seq'::regclass);

ALTER TABLE ONLY ci_freeze_periods ALTER COLUMN id SET DEFAULT nextval('ci_freeze_periods_id_seq'::regclass);

ALTER TABLE ONLY ci_group_variables ALTER COLUMN id SET DEFAULT nextval('ci_group_variables_id_seq'::regclass);

ALTER TABLE ONLY ci_instance_variables ALTER COLUMN id SET DEFAULT nextval('ci_instance_variables_id_seq'::regclass);

ALTER TABLE ONLY ci_job_artifact_states ALTER COLUMN job_artifact_id SET DEFAULT nextval('ci_job_artifact_states_job_artifact_id_seq'::regclass);

ALTER TABLE ONLY ci_job_artifacts ALTER COLUMN id SET DEFAULT nextval('ci_job_artifacts_id_seq'::regclass);

ALTER TABLE ONLY ci_job_token_project_scope_links ALTER COLUMN id SET DEFAULT nextval('ci_job_token_project_scope_links_id_seq'::regclass);

ALTER TABLE ONLY ci_job_variables ALTER COLUMN id SET DEFAULT nextval('ci_job_variables_id_seq'::regclass);

ALTER TABLE ONLY ci_minutes_additional_packs ALTER COLUMN id SET DEFAULT nextval('ci_minutes_additional_packs_id_seq'::regclass);

ALTER TABLE ONLY ci_namespace_mirrors ALTER COLUMN id SET DEFAULT nextval('ci_namespace_mirrors_id_seq'::regclass);

ALTER TABLE ONLY ci_namespace_monthly_usages ALTER COLUMN id SET DEFAULT nextval('ci_namespace_monthly_usages_id_seq'::regclass);

ALTER TABLE ONLY ci_pending_builds ALTER COLUMN id SET DEFAULT nextval('ci_pending_builds_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_artifacts ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_artifacts_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_chat_data ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_chat_data_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_messages ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_messages_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_schedule_variables ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_schedule_variables_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_schedules ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_schedules_id_seq'::regclass);

ALTER TABLE ONLY ci_pipeline_variables ALTER COLUMN id SET DEFAULT nextval('ci_pipeline_variables_id_seq'::regclass);

ALTER TABLE ONLY ci_pipelines ALTER COLUMN id SET DEFAULT nextval('ci_pipelines_id_seq'::regclass);

ALTER TABLE ONLY ci_pipelines_config ALTER COLUMN pipeline_id SET DEFAULT nextval('ci_pipelines_config_pipeline_id_seq'::regclass);

ALTER TABLE ONLY ci_platform_metrics ALTER COLUMN id SET DEFAULT nextval('ci_platform_metrics_id_seq'::regclass);

ALTER TABLE ONLY ci_project_mirrors ALTER COLUMN id SET DEFAULT nextval('ci_project_mirrors_id_seq'::regclass);

ALTER TABLE ONLY ci_project_monthly_usages ALTER COLUMN id SET DEFAULT nextval('ci_project_monthly_usages_id_seq'::regclass);

ALTER TABLE ONLY ci_refs ALTER COLUMN id SET DEFAULT nextval('ci_refs_id_seq'::regclass);

ALTER TABLE ONLY ci_resource_groups ALTER COLUMN id SET DEFAULT nextval('ci_resource_groups_id_seq'::regclass);

ALTER TABLE ONLY ci_resources ALTER COLUMN id SET DEFAULT nextval('ci_resources_id_seq'::regclass);

ALTER TABLE ONLY ci_runner_namespaces ALTER COLUMN id SET DEFAULT nextval('ci_runner_namespaces_id_seq'::regclass);

ALTER TABLE ONLY ci_runner_projects ALTER COLUMN id SET DEFAULT nextval('ci_runner_projects_id_seq'::regclass);

ALTER TABLE ONLY ci_runners ALTER COLUMN id SET DEFAULT nextval('ci_runners_id_seq'::regclass);

ALTER TABLE ONLY ci_running_builds ALTER COLUMN id SET DEFAULT nextval('ci_running_builds_id_seq'::regclass);

ALTER TABLE ONLY ci_sources_pipelines ALTER COLUMN id SET DEFAULT nextval('ci_sources_pipelines_id_seq'::regclass);

ALTER TABLE ONLY ci_sources_projects ALTER COLUMN id SET DEFAULT nextval('ci_sources_projects_id_seq'::regclass);

ALTER TABLE ONLY ci_stages ALTER COLUMN id SET DEFAULT nextval('ci_stages_id_seq'::regclass);

ALTER TABLE ONLY ci_subscriptions_projects ALTER COLUMN id SET DEFAULT nextval('ci_subscriptions_projects_id_seq'::regclass);

ALTER TABLE ONLY ci_trigger_requests ALTER COLUMN id SET DEFAULT nextval('ci_trigger_requests_id_seq'::regclass);

ALTER TABLE ONLY ci_triggers ALTER COLUMN id SET DEFAULT nextval('ci_triggers_id_seq'::regclass);

ALTER TABLE ONLY ci_unit_test_failures ALTER COLUMN id SET DEFAULT nextval('ci_unit_test_failures_id_seq'::regclass);

ALTER TABLE ONLY ci_unit_tests ALTER COLUMN id SET DEFAULT nextval('ci_unit_tests_id_seq'::regclass);

ALTER TABLE ONLY ci_variables ALTER COLUMN id SET DEFAULT nextval('ci_variables_id_seq'::regclass);

ALTER TABLE ONLY cluster_agent_tokens ALTER COLUMN id SET DEFAULT nextval('cluster_agent_tokens_id_seq'::regclass);

ALTER TABLE ONLY cluster_agents ALTER COLUMN id SET DEFAULT nextval('cluster_agents_id_seq'::regclass);

ALTER TABLE ONLY cluster_groups ALTER COLUMN id SET DEFAULT nextval('cluster_groups_id_seq'::regclass);

ALTER TABLE ONLY cluster_platforms_kubernetes ALTER COLUMN id SET DEFAULT nextval('cluster_platforms_kubernetes_id_seq'::regclass);

ALTER TABLE ONLY cluster_projects ALTER COLUMN id SET DEFAULT nextval('cluster_projects_id_seq'::regclass);

ALTER TABLE ONLY cluster_providers_aws ALTER COLUMN id SET DEFAULT nextval('cluster_providers_aws_id_seq'::regclass);

ALTER TABLE ONLY cluster_providers_gcp ALTER COLUMN id SET DEFAULT nextval('cluster_providers_gcp_id_seq'::regclass);

ALTER TABLE ONLY clusters ALTER COLUMN id SET DEFAULT nextval('clusters_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_cert_managers ALTER COLUMN id SET DEFAULT nextval('clusters_applications_cert_managers_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_cilium ALTER COLUMN id SET DEFAULT nextval('clusters_applications_cilium_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_crossplane ALTER COLUMN id SET DEFAULT nextval('clusters_applications_crossplane_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_elastic_stacks ALTER COLUMN id SET DEFAULT nextval('clusters_applications_elastic_stacks_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_helm ALTER COLUMN id SET DEFAULT nextval('clusters_applications_helm_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_ingress ALTER COLUMN id SET DEFAULT nextval('clusters_applications_ingress_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_jupyter ALTER COLUMN id SET DEFAULT nextval('clusters_applications_jupyter_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_knative ALTER COLUMN id SET DEFAULT nextval('clusters_applications_knative_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_prometheus ALTER COLUMN id SET DEFAULT nextval('clusters_applications_prometheus_id_seq'::regclass);

ALTER TABLE ONLY clusters_applications_runners ALTER COLUMN id SET DEFAULT nextval('clusters_applications_runners_id_seq'::regclass);

ALTER TABLE ONLY clusters_kubernetes_namespaces ALTER COLUMN id SET DEFAULT nextval('clusters_kubernetes_namespaces_id_seq'::regclass);

ALTER TABLE ONLY commit_user_mentions ALTER COLUMN id SET DEFAULT nextval('commit_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY compliance_management_frameworks ALTER COLUMN id SET DEFAULT nextval('compliance_management_frameworks_id_seq'::regclass);

ALTER TABLE ONLY container_repositories ALTER COLUMN id SET DEFAULT nextval('container_repositories_id_seq'::regclass);

ALTER TABLE ONLY content_blocked_states ALTER COLUMN id SET DEFAULT nextval('content_blocked_states_id_seq'::regclass);

ALTER TABLE ONLY conversational_development_index_metrics ALTER COLUMN id SET DEFAULT nextval('conversational_development_index_metrics_id_seq'::regclass);

ALTER TABLE ONLY coverage_fuzzing_corpuses ALTER COLUMN id SET DEFAULT nextval('coverage_fuzzing_corpuses_id_seq'::regclass);

ALTER TABLE ONLY csv_issue_imports ALTER COLUMN id SET DEFAULT nextval('csv_issue_imports_id_seq'::regclass);

ALTER TABLE ONLY custom_emoji ALTER COLUMN id SET DEFAULT nextval('custom_emoji_id_seq'::regclass);

ALTER TABLE ONLY customer_relations_contacts ALTER COLUMN id SET DEFAULT nextval('customer_relations_contacts_id_seq'::regclass);

ALTER TABLE ONLY customer_relations_organizations ALTER COLUMN id SET DEFAULT nextval('customer_relations_organizations_id_seq'::regclass);

ALTER TABLE ONLY dast_profile_schedules ALTER COLUMN id SET DEFAULT nextval('dast_profile_schedules_id_seq'::regclass);

ALTER TABLE ONLY dast_profiles ALTER COLUMN id SET DEFAULT nextval('dast_profiles_id_seq'::regclass);

ALTER TABLE ONLY dast_scanner_profiles ALTER COLUMN id SET DEFAULT nextval('dast_scanner_profiles_id_seq'::regclass);

ALTER TABLE ONLY dast_site_profile_secret_variables ALTER COLUMN id SET DEFAULT nextval('dast_site_profile_secret_variables_id_seq'::regclass);

ALTER TABLE ONLY dast_site_profiles ALTER COLUMN id SET DEFAULT nextval('dast_site_profiles_id_seq'::regclass);

ALTER TABLE ONLY dast_site_tokens ALTER COLUMN id SET DEFAULT nextval('dast_site_tokens_id_seq'::regclass);

ALTER TABLE ONLY dast_site_validations ALTER COLUMN id SET DEFAULT nextval('dast_site_validations_id_seq'::regclass);

ALTER TABLE ONLY dast_sites ALTER COLUMN id SET DEFAULT nextval('dast_sites_id_seq'::regclass);

ALTER TABLE ONLY dependency_proxy_blobs ALTER COLUMN id SET DEFAULT nextval('dependency_proxy_blobs_id_seq'::regclass);

ALTER TABLE ONLY dependency_proxy_group_settings ALTER COLUMN id SET DEFAULT nextval('dependency_proxy_group_settings_id_seq'::regclass);

ALTER TABLE ONLY dependency_proxy_manifests ALTER COLUMN id SET DEFAULT nextval('dependency_proxy_manifests_id_seq'::regclass);

ALTER TABLE ONLY deploy_keys_projects ALTER COLUMN id SET DEFAULT nextval('deploy_keys_projects_id_seq'::regclass);

ALTER TABLE ONLY deploy_tokens ALTER COLUMN id SET DEFAULT nextval('deploy_tokens_id_seq'::regclass);

ALTER TABLE ONLY deployment_approvals ALTER COLUMN id SET DEFAULT nextval('deployment_approvals_id_seq'::regclass);

ALTER TABLE ONLY deployments ALTER COLUMN id SET DEFAULT nextval('deployments_id_seq'::regclass);

ALTER TABLE ONLY description_versions ALTER COLUMN id SET DEFAULT nextval('description_versions_id_seq'::regclass);

ALTER TABLE ONLY design_management_designs ALTER COLUMN id SET DEFAULT nextval('design_management_designs_id_seq'::regclass);

ALTER TABLE ONLY design_management_designs_versions ALTER COLUMN id SET DEFAULT nextval('design_management_designs_versions_id_seq'::regclass);

ALTER TABLE ONLY design_management_versions ALTER COLUMN id SET DEFAULT nextval('design_management_versions_id_seq'::regclass);

ALTER TABLE ONLY design_user_mentions ALTER COLUMN id SET DEFAULT nextval('design_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY detached_partitions ALTER COLUMN id SET DEFAULT nextval('detached_partitions_id_seq'::regclass);

ALTER TABLE ONLY diff_note_positions ALTER COLUMN id SET DEFAULT nextval('diff_note_positions_id_seq'::regclass);

ALTER TABLE ONLY dora_daily_metrics ALTER COLUMN id SET DEFAULT nextval('dora_daily_metrics_id_seq'::regclass);

ALTER TABLE ONLY draft_notes ALTER COLUMN id SET DEFAULT nextval('draft_notes_id_seq'::regclass);

ALTER TABLE ONLY elastic_index_settings ALTER COLUMN id SET DEFAULT nextval('elastic_index_settings_id_seq'::regclass);

ALTER TABLE ONLY elastic_reindexing_slices ALTER COLUMN id SET DEFAULT nextval('elastic_reindexing_slices_id_seq'::regclass);

ALTER TABLE ONLY elastic_reindexing_subtasks ALTER COLUMN id SET DEFAULT nextval('elastic_reindexing_subtasks_id_seq'::regclass);

ALTER TABLE ONLY elastic_reindexing_tasks ALTER COLUMN id SET DEFAULT nextval('elastic_reindexing_tasks_id_seq'::regclass);

ALTER TABLE ONLY emails ALTER COLUMN id SET DEFAULT nextval('emails_id_seq'::regclass);

ALTER TABLE ONLY environments ALTER COLUMN id SET DEFAULT nextval('environments_id_seq'::regclass);

ALTER TABLE ONLY epic_issues ALTER COLUMN id SET DEFAULT nextval('epic_issues_id_seq'::regclass);

ALTER TABLE ONLY epic_metrics ALTER COLUMN id SET DEFAULT nextval('epic_metrics_id_seq'::regclass);

ALTER TABLE ONLY epic_user_mentions ALTER COLUMN id SET DEFAULT nextval('epic_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY epics ALTER COLUMN id SET DEFAULT nextval('epics_id_seq'::regclass);

ALTER TABLE ONLY error_tracking_client_keys ALTER COLUMN id SET DEFAULT nextval('error_tracking_client_keys_id_seq'::regclass);

ALTER TABLE ONLY error_tracking_error_events ALTER COLUMN id SET DEFAULT nextval('error_tracking_error_events_id_seq'::regclass);

ALTER TABLE ONLY error_tracking_errors ALTER COLUMN id SET DEFAULT nextval('error_tracking_errors_id_seq'::regclass);

ALTER TABLE ONLY events ALTER COLUMN id SET DEFAULT nextval('events_id_seq'::regclass);

ALTER TABLE ONLY evidences ALTER COLUMN id SET DEFAULT nextval('evidences_id_seq'::regclass);

ALTER TABLE ONLY experiment_subjects ALTER COLUMN id SET DEFAULT nextval('experiment_subjects_id_seq'::regclass);

ALTER TABLE ONLY experiment_users ALTER COLUMN id SET DEFAULT nextval('experiment_users_id_seq'::regclass);

ALTER TABLE ONLY experiments ALTER COLUMN id SET DEFAULT nextval('experiments_id_seq'::regclass);

ALTER TABLE ONLY external_approval_rules ALTER COLUMN id SET DEFAULT nextval('external_approval_rules_id_seq'::regclass);

ALTER TABLE ONLY external_approval_rules_protected_branches ALTER COLUMN id SET DEFAULT nextval('external_approval_rules_protected_branches_id_seq'::regclass);

ALTER TABLE ONLY external_pull_requests ALTER COLUMN id SET DEFAULT nextval('external_pull_requests_id_seq'::regclass);

ALTER TABLE ONLY external_status_checks ALTER COLUMN id SET DEFAULT nextval('external_status_checks_id_seq'::regclass);

ALTER TABLE ONLY external_status_checks_protected_branches ALTER COLUMN id SET DEFAULT nextval('external_status_checks_protected_branches_id_seq'::regclass);

ALTER TABLE ONLY feature_gates ALTER COLUMN id SET DEFAULT nextval('feature_gates_id_seq'::regclass);

ALTER TABLE ONLY features ALTER COLUMN id SET DEFAULT nextval('features_id_seq'::regclass);

ALTER TABLE ONLY fork_network_members ALTER COLUMN id SET DEFAULT nextval('fork_network_members_id_seq'::regclass);

ALTER TABLE ONLY fork_networks ALTER COLUMN id SET DEFAULT nextval('fork_networks_id_seq'::regclass);

ALTER TABLE ONLY geo_cache_invalidation_events ALTER COLUMN id SET DEFAULT nextval('geo_cache_invalidation_events_id_seq'::regclass);

ALTER TABLE ONLY geo_container_repository_updated_events ALTER COLUMN id SET DEFAULT nextval('geo_container_repository_updated_events_id_seq'::regclass);

ALTER TABLE ONLY geo_event_log ALTER COLUMN id SET DEFAULT nextval('geo_event_log_id_seq'::regclass);

ALTER TABLE ONLY geo_events ALTER COLUMN id SET DEFAULT nextval('geo_events_id_seq'::regclass);

ALTER TABLE ONLY geo_hashed_storage_attachments_events ALTER COLUMN id SET DEFAULT nextval('geo_hashed_storage_attachments_events_id_seq'::regclass);

ALTER TABLE ONLY geo_hashed_storage_migrated_events ALTER COLUMN id SET DEFAULT nextval('geo_hashed_storage_migrated_events_id_seq'::regclass);

ALTER TABLE ONLY geo_job_artifact_deleted_events ALTER COLUMN id SET DEFAULT nextval('geo_job_artifact_deleted_events_id_seq'::regclass);

ALTER TABLE ONLY geo_lfs_object_deleted_events ALTER COLUMN id SET DEFAULT nextval('geo_lfs_object_deleted_events_id_seq'::regclass);

ALTER TABLE ONLY geo_node_namespace_links ALTER COLUMN id SET DEFAULT nextval('geo_node_namespace_links_id_seq'::regclass);

ALTER TABLE ONLY geo_node_statuses ALTER COLUMN id SET DEFAULT nextval('geo_node_statuses_id_seq'::regclass);

ALTER TABLE ONLY geo_nodes ALTER COLUMN id SET DEFAULT nextval('geo_nodes_id_seq'::regclass);

ALTER TABLE ONLY geo_repositories_changed_events ALTER COLUMN id SET DEFAULT nextval('geo_repositories_changed_events_id_seq'::regclass);

ALTER TABLE ONLY geo_repository_created_events ALTER COLUMN id SET DEFAULT nextval('geo_repository_created_events_id_seq'::regclass);

ALTER TABLE ONLY geo_repository_deleted_events ALTER COLUMN id SET DEFAULT nextval('geo_repository_deleted_events_id_seq'::regclass);

ALTER TABLE ONLY geo_repository_renamed_events ALTER COLUMN id SET DEFAULT nextval('geo_repository_renamed_events_id_seq'::regclass);

ALTER TABLE ONLY geo_repository_updated_events ALTER COLUMN id SET DEFAULT nextval('geo_repository_updated_events_id_seq'::regclass);

ALTER TABLE ONLY geo_reset_checksum_events ALTER COLUMN id SET DEFAULT nextval('geo_reset_checksum_events_id_seq'::regclass);

ALTER TABLE ONLY gitlab_subscription_histories ALTER COLUMN id SET DEFAULT nextval('gitlab_subscription_histories_id_seq'::regclass);

ALTER TABLE ONLY gitlab_subscriptions ALTER COLUMN id SET DEFAULT nextval('gitlab_subscriptions_id_seq'::regclass);

ALTER TABLE ONLY gpg_key_subkeys ALTER COLUMN id SET DEFAULT nextval('gpg_key_subkeys_id_seq'::regclass);

ALTER TABLE ONLY gpg_keys ALTER COLUMN id SET DEFAULT nextval('gpg_keys_id_seq'::regclass);

ALTER TABLE ONLY gpg_signatures ALTER COLUMN id SET DEFAULT nextval('gpg_signatures_id_seq'::regclass);

ALTER TABLE ONLY grafana_integrations ALTER COLUMN id SET DEFAULT nextval('grafana_integrations_id_seq'::regclass);

ALTER TABLE ONLY group_custom_attributes ALTER COLUMN id SET DEFAULT nextval('group_custom_attributes_id_seq'::regclass);

ALTER TABLE ONLY group_deploy_keys ALTER COLUMN id SET DEFAULT nextval('group_deploy_keys_id_seq'::regclass);

ALTER TABLE ONLY group_deploy_keys_groups ALTER COLUMN id SET DEFAULT nextval('group_deploy_keys_groups_id_seq'::regclass);

ALTER TABLE ONLY group_deploy_tokens ALTER COLUMN id SET DEFAULT nextval('group_deploy_tokens_id_seq'::regclass);

ALTER TABLE ONLY group_group_links ALTER COLUMN id SET DEFAULT nextval('group_group_links_id_seq'::regclass);

ALTER TABLE ONLY group_import_states ALTER COLUMN group_id SET DEFAULT nextval('group_import_states_group_id_seq'::regclass);

ALTER TABLE ONLY group_repository_storage_moves ALTER COLUMN id SET DEFAULT nextval('group_repository_storage_moves_id_seq'::regclass);

ALTER TABLE ONLY historical_data ALTER COLUMN id SET DEFAULT nextval('historical_data_id_seq'::regclass);

ALTER TABLE ONLY identities ALTER COLUMN id SET DEFAULT nextval('identities_id_seq'::regclass);

ALTER TABLE ONLY import_export_uploads ALTER COLUMN id SET DEFAULT nextval('import_export_uploads_id_seq'::regclass);

ALTER TABLE ONLY import_failures ALTER COLUMN id SET DEFAULT nextval('import_failures_id_seq'::regclass);

ALTER TABLE ONLY in_product_marketing_emails ALTER COLUMN id SET DEFAULT nextval('in_product_marketing_emails_id_seq'::regclass);

ALTER TABLE ONLY incident_management_escalation_policies ALTER COLUMN id SET DEFAULT nextval('incident_management_escalation_policies_id_seq'::regclass);

ALTER TABLE ONLY incident_management_escalation_rules ALTER COLUMN id SET DEFAULT nextval('incident_management_escalation_rules_id_seq'::regclass);

ALTER TABLE ONLY incident_management_issuable_escalation_statuses ALTER COLUMN id SET DEFAULT nextval('incident_management_issuable_escalation_statuses_id_seq'::regclass);

ALTER TABLE ONLY incident_management_oncall_participants ALTER COLUMN id SET DEFAULT nextval('incident_management_oncall_participants_id_seq'::regclass);

ALTER TABLE ONLY incident_management_oncall_rotations ALTER COLUMN id SET DEFAULT nextval('incident_management_oncall_rotations_id_seq'::regclass);

ALTER TABLE ONLY incident_management_oncall_schedules ALTER COLUMN id SET DEFAULT nextval('incident_management_oncall_schedules_id_seq'::regclass);

ALTER TABLE ONLY incident_management_oncall_shifts ALTER COLUMN id SET DEFAULT nextval('incident_management_oncall_shifts_id_seq'::regclass);

ALTER TABLE ONLY incident_management_pending_alert_escalations ALTER COLUMN id SET DEFAULT nextval('incident_management_pending_alert_escalations_id_seq'::regclass);

ALTER TABLE ONLY incident_management_pending_issue_escalations ALTER COLUMN id SET DEFAULT nextval('incident_management_pending_issue_escalations_id_seq'::regclass);

ALTER TABLE ONLY incident_management_timeline_events ALTER COLUMN id SET DEFAULT nextval('incident_management_timeline_events_id_seq'::regclass);

ALTER TABLE ONLY index_statuses ALTER COLUMN id SET DEFAULT nextval('index_statuses_id_seq'::regclass);

ALTER TABLE ONLY insights ALTER COLUMN id SET DEFAULT nextval('insights_id_seq'::regclass);

ALTER TABLE ONLY integrations ALTER COLUMN id SET DEFAULT nextval('integrations_id_seq'::regclass);

ALTER TABLE ONLY internal_ids ALTER COLUMN id SET DEFAULT nextval('internal_ids_id_seq'::regclass);

ALTER TABLE ONLY ip_restrictions ALTER COLUMN id SET DEFAULT nextval('ip_restrictions_id_seq'::regclass);

ALTER TABLE ONLY issuable_metric_images ALTER COLUMN id SET DEFAULT nextval('issuable_metric_images_id_seq'::regclass);

ALTER TABLE ONLY issuable_severities ALTER COLUMN id SET DEFAULT nextval('issuable_severities_id_seq'::regclass);

ALTER TABLE ONLY issuable_slas ALTER COLUMN id SET DEFAULT nextval('issuable_slas_id_seq'::regclass);

ALTER TABLE ONLY issue_customer_relations_contacts ALTER COLUMN id SET DEFAULT nextval('issue_customer_relations_contacts_id_seq'::regclass);

ALTER TABLE ONLY issue_email_participants ALTER COLUMN id SET DEFAULT nextval('issue_email_participants_id_seq'::regclass);

ALTER TABLE ONLY issue_emails ALTER COLUMN id SET DEFAULT nextval('issue_emails_id_seq'::regclass);

ALTER TABLE ONLY issue_links ALTER COLUMN id SET DEFAULT nextval('issue_links_id_seq'::regclass);

ALTER TABLE ONLY issue_metrics ALTER COLUMN id SET DEFAULT nextval('issue_metrics_id_seq'::regclass);

ALTER TABLE ONLY issue_tracker_data ALTER COLUMN id SET DEFAULT nextval('issue_tracker_data_id_seq'::regclass);

ALTER TABLE ONLY issue_user_mentions ALTER COLUMN id SET DEFAULT nextval('issue_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY issues ALTER COLUMN id SET DEFAULT nextval('issues_id_seq'::regclass);

ALTER TABLE ONLY iterations_cadences ALTER COLUMN id SET DEFAULT nextval('iterations_cadences_id_seq'::regclass);

ALTER TABLE ONLY jira_connect_installations ALTER COLUMN id SET DEFAULT nextval('jira_connect_installations_id_seq'::regclass);

ALTER TABLE ONLY jira_connect_subscriptions ALTER COLUMN id SET DEFAULT nextval('jira_connect_subscriptions_id_seq'::regclass);

ALTER TABLE ONLY jira_imports ALTER COLUMN id SET DEFAULT nextval('jira_imports_id_seq'::regclass);

ALTER TABLE ONLY jira_tracker_data ALTER COLUMN id SET DEFAULT nextval('jira_tracker_data_id_seq'::regclass);

ALTER TABLE ONLY keys ALTER COLUMN id SET DEFAULT nextval('keys_id_seq'::regclass);

ALTER TABLE ONLY label_links ALTER COLUMN id SET DEFAULT nextval('label_links_id_seq'::regclass);

ALTER TABLE ONLY label_priorities ALTER COLUMN id SET DEFAULT nextval('label_priorities_id_seq'::regclass);

ALTER TABLE ONLY labels ALTER COLUMN id SET DEFAULT nextval('labels_id_seq'::regclass);

ALTER TABLE ONLY ldap_group_links ALTER COLUMN id SET DEFAULT nextval('ldap_group_links_id_seq'::regclass);

ALTER TABLE ONLY lfs_file_locks ALTER COLUMN id SET DEFAULT nextval('lfs_file_locks_id_seq'::regclass);

ALTER TABLE ONLY lfs_object_states ALTER COLUMN lfs_object_id SET DEFAULT nextval('lfs_object_states_lfs_object_id_seq'::regclass);

ALTER TABLE ONLY lfs_objects ALTER COLUMN id SET DEFAULT nextval('lfs_objects_id_seq'::regclass);

ALTER TABLE ONLY lfs_objects_projects ALTER COLUMN id SET DEFAULT nextval('lfs_objects_projects_id_seq'::regclass);

ALTER TABLE ONLY licenses ALTER COLUMN id SET DEFAULT nextval('licenses_id_seq'::regclass);

ALTER TABLE ONLY list_user_preferences ALTER COLUMN id SET DEFAULT nextval('list_user_preferences_id_seq'::regclass);

ALTER TABLE ONLY lists ALTER COLUMN id SET DEFAULT nextval('lists_id_seq'::regclass);

ALTER TABLE ONLY loose_foreign_keys_deleted_records ALTER COLUMN id SET DEFAULT nextval('loose_foreign_keys_deleted_records_id_seq'::regclass);

ALTER TABLE ONLY member_tasks ALTER COLUMN id SET DEFAULT nextval('member_tasks_id_seq'::regclass);

ALTER TABLE ONLY members ALTER COLUMN id SET DEFAULT nextval('members_id_seq'::regclass);

ALTER TABLE ONLY merge_request_assignees ALTER COLUMN id SET DEFAULT nextval('merge_request_assignees_id_seq'::regclass);

ALTER TABLE ONLY merge_request_blocks ALTER COLUMN id SET DEFAULT nextval('merge_request_blocks_id_seq'::regclass);

ALTER TABLE ONLY merge_request_cleanup_schedules ALTER COLUMN merge_request_id SET DEFAULT nextval('merge_request_cleanup_schedules_merge_request_id_seq'::regclass);

ALTER TABLE ONLY merge_request_context_commits ALTER COLUMN id SET DEFAULT nextval('merge_request_context_commits_id_seq'::regclass);

ALTER TABLE ONLY merge_request_diff_commit_users ALTER COLUMN id SET DEFAULT nextval('merge_request_diff_commit_users_id_seq'::regclass);

ALTER TABLE ONLY merge_request_diff_details ALTER COLUMN merge_request_diff_id SET DEFAULT nextval('merge_request_diff_details_merge_request_diff_id_seq'::regclass);

ALTER TABLE ONLY merge_request_diffs ALTER COLUMN id SET DEFAULT nextval('merge_request_diffs_id_seq'::regclass);

ALTER TABLE ONLY merge_request_metrics ALTER COLUMN id SET DEFAULT nextval('merge_request_metrics_id_seq'::regclass);

ALTER TABLE ONLY merge_request_reviewers ALTER COLUMN id SET DEFAULT nextval('merge_request_reviewers_id_seq'::regclass);

ALTER TABLE ONLY merge_request_user_mentions ALTER COLUMN id SET DEFAULT nextval('merge_request_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY merge_requests ALTER COLUMN id SET DEFAULT nextval('merge_requests_id_seq'::regclass);

ALTER TABLE ONLY merge_requests_closing_issues ALTER COLUMN id SET DEFAULT nextval('merge_requests_closing_issues_id_seq'::regclass);

ALTER TABLE ONLY merge_requests_compliance_violations ALTER COLUMN id SET DEFAULT nextval('merge_requests_compliance_violations_id_seq'::regclass);

ALTER TABLE ONLY merge_trains ALTER COLUMN id SET DEFAULT nextval('merge_trains_id_seq'::regclass);

ALTER TABLE ONLY metrics_dashboard_annotations ALTER COLUMN id SET DEFAULT nextval('metrics_dashboard_annotations_id_seq'::regclass);

ALTER TABLE ONLY metrics_users_starred_dashboards ALTER COLUMN id SET DEFAULT nextval('metrics_users_starred_dashboards_id_seq'::regclass);

ALTER TABLE ONLY milestones ALTER COLUMN id SET DEFAULT nextval('milestones_id_seq'::regclass);

ALTER TABLE ONLY namespace_admin_notes ALTER COLUMN id SET DEFAULT nextval('namespace_admin_notes_id_seq'::regclass);

ALTER TABLE ONLY namespace_aggregation_schedules ALTER COLUMN namespace_id SET DEFAULT nextval('namespace_aggregation_schedules_namespace_id_seq'::regclass);

ALTER TABLE ONLY namespace_root_storage_statistics ALTER COLUMN namespace_id SET DEFAULT nextval('namespace_root_storage_statistics_namespace_id_seq'::regclass);

ALTER TABLE ONLY namespace_statistics ALTER COLUMN id SET DEFAULT nextval('namespace_statistics_id_seq'::regclass);

ALTER TABLE ONLY namespaces ALTER COLUMN id SET DEFAULT nextval('namespaces_id_seq'::regclass);

ALTER TABLE ONLY namespaces_sync_events ALTER COLUMN id SET DEFAULT nextval('namespaces_sync_events_id_seq'::regclass);

ALTER TABLE ONLY note_diff_files ALTER COLUMN id SET DEFAULT nextval('note_diff_files_id_seq'::regclass);

ALTER TABLE ONLY notes ALTER COLUMN id SET DEFAULT nextval('notes_id_seq'::regclass);

ALTER TABLE ONLY notification_settings ALTER COLUMN id SET DEFAULT nextval('notification_settings_id_seq'::regclass);

ALTER TABLE ONLY oauth_access_grants ALTER COLUMN id SET DEFAULT nextval('oauth_access_grants_id_seq'::regclass);

ALTER TABLE ONLY oauth_access_tokens ALTER COLUMN id SET DEFAULT nextval('oauth_access_tokens_id_seq'::regclass);

ALTER TABLE ONLY oauth_applications ALTER COLUMN id SET DEFAULT nextval('oauth_applications_id_seq'::regclass);

ALTER TABLE ONLY oauth_openid_requests ALTER COLUMN id SET DEFAULT nextval('oauth_openid_requests_id_seq'::regclass);

ALTER TABLE ONLY onboarding_progresses ALTER COLUMN id SET DEFAULT nextval('onboarding_progresses_id_seq'::regclass);

ALTER TABLE ONLY operations_feature_flag_scopes ALTER COLUMN id SET DEFAULT nextval('operations_feature_flag_scopes_id_seq'::regclass);

ALTER TABLE ONLY operations_feature_flags ALTER COLUMN id SET DEFAULT nextval('operations_feature_flags_id_seq'::regclass);

ALTER TABLE ONLY operations_feature_flags_clients ALTER COLUMN id SET DEFAULT nextval('operations_feature_flags_clients_id_seq'::regclass);

ALTER TABLE ONLY operations_feature_flags_issues ALTER COLUMN id SET DEFAULT nextval('operations_feature_flags_issues_id_seq'::regclass);

ALTER TABLE ONLY operations_scopes ALTER COLUMN id SET DEFAULT nextval('operations_scopes_id_seq'::regclass);

ALTER TABLE ONLY operations_strategies ALTER COLUMN id SET DEFAULT nextval('operations_strategies_id_seq'::regclass);

ALTER TABLE ONLY operations_strategies_user_lists ALTER COLUMN id SET DEFAULT nextval('operations_strategies_user_lists_id_seq'::regclass);

ALTER TABLE ONLY operations_user_lists ALTER COLUMN id SET DEFAULT nextval('operations_user_lists_id_seq'::regclass);

ALTER TABLE ONLY packages_build_infos ALTER COLUMN id SET DEFAULT nextval('packages_build_infos_id_seq'::regclass);

ALTER TABLE ONLY packages_composer_cache_files ALTER COLUMN id SET DEFAULT nextval('packages_composer_cache_files_id_seq'::regclass);

ALTER TABLE ONLY packages_conan_file_metadata ALTER COLUMN id SET DEFAULT nextval('packages_conan_file_metadata_id_seq'::regclass);

ALTER TABLE ONLY packages_conan_metadata ALTER COLUMN id SET DEFAULT nextval('packages_conan_metadata_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_group_architectures ALTER COLUMN id SET DEFAULT nextval('packages_debian_group_architectures_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_group_component_files ALTER COLUMN id SET DEFAULT nextval('packages_debian_group_component_files_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_group_components ALTER COLUMN id SET DEFAULT nextval('packages_debian_group_components_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_group_distribution_keys ALTER COLUMN id SET DEFAULT nextval('packages_debian_group_distribution_keys_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_group_distributions ALTER COLUMN id SET DEFAULT nextval('packages_debian_group_distributions_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_project_architectures ALTER COLUMN id SET DEFAULT nextval('packages_debian_project_architectures_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_project_component_files ALTER COLUMN id SET DEFAULT nextval('packages_debian_project_component_files_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_project_components ALTER COLUMN id SET DEFAULT nextval('packages_debian_project_components_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_project_distribution_keys ALTER COLUMN id SET DEFAULT nextval('packages_debian_project_distribution_keys_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_project_distributions ALTER COLUMN id SET DEFAULT nextval('packages_debian_project_distributions_id_seq'::regclass);

ALTER TABLE ONLY packages_debian_publications ALTER COLUMN id SET DEFAULT nextval('packages_debian_publications_id_seq'::regclass);

ALTER TABLE ONLY packages_dependencies ALTER COLUMN id SET DEFAULT nextval('packages_dependencies_id_seq'::regclass);

ALTER TABLE ONLY packages_dependency_links ALTER COLUMN id SET DEFAULT nextval('packages_dependency_links_id_seq'::regclass);

ALTER TABLE ONLY packages_events ALTER COLUMN id SET DEFAULT nextval('packages_events_id_seq'::regclass);

ALTER TABLE ONLY packages_maven_metadata ALTER COLUMN id SET DEFAULT nextval('packages_maven_metadata_id_seq'::regclass);

ALTER TABLE ONLY packages_package_file_build_infos ALTER COLUMN id SET DEFAULT nextval('packages_package_file_build_infos_id_seq'::regclass);

ALTER TABLE ONLY packages_package_files ALTER COLUMN id SET DEFAULT nextval('packages_package_files_id_seq'::regclass);

ALTER TABLE ONLY packages_packages ALTER COLUMN id SET DEFAULT nextval('packages_packages_id_seq'::regclass);

ALTER TABLE ONLY packages_tags ALTER COLUMN id SET DEFAULT nextval('packages_tags_id_seq'::regclass);

ALTER TABLE ONLY pages_deployment_states ALTER COLUMN pages_deployment_id SET DEFAULT nextval('pages_deployment_states_pages_deployment_id_seq'::regclass);

ALTER TABLE ONLY pages_deployments ALTER COLUMN id SET DEFAULT nextval('pages_deployments_id_seq'::regclass);

ALTER TABLE ONLY pages_domain_acme_orders ALTER COLUMN id SET DEFAULT nextval('pages_domain_acme_orders_id_seq'::regclass);

ALTER TABLE ONLY pages_domains ALTER COLUMN id SET DEFAULT nextval('pages_domains_id_seq'::regclass);

ALTER TABLE ONLY partitioned_foreign_keys ALTER COLUMN id SET DEFAULT nextval('partitioned_foreign_keys_id_seq'::regclass);

ALTER TABLE ONLY path_locks ALTER COLUMN id SET DEFAULT nextval('path_locks_id_seq'::regclass);

ALTER TABLE ONLY personal_access_tokens ALTER COLUMN id SET DEFAULT nextval('personal_access_tokens_id_seq'::regclass);

ALTER TABLE ONLY plan_limits ALTER COLUMN id SET DEFAULT nextval('plan_limits_id_seq'::regclass);

ALTER TABLE ONLY plans ALTER COLUMN id SET DEFAULT nextval('plans_id_seq'::regclass);

ALTER TABLE ONLY pool_repositories ALTER COLUMN id SET DEFAULT nextval('pool_repositories_id_seq'::regclass);

ALTER TABLE ONLY postgres_async_indexes ALTER COLUMN id SET DEFAULT nextval('postgres_async_indexes_id_seq'::regclass);

ALTER TABLE ONLY postgres_reindex_actions ALTER COLUMN id SET DEFAULT nextval('postgres_reindex_actions_id_seq'::regclass);

ALTER TABLE ONLY postgres_reindex_queued_actions ALTER COLUMN id SET DEFAULT nextval('postgres_reindex_queued_actions_id_seq'::regclass);

ALTER TABLE ONLY product_analytics_events_experimental ALTER COLUMN id SET DEFAULT nextval('product_analytics_events_experimental_id_seq'::regclass);

ALTER TABLE ONLY programming_languages ALTER COLUMN id SET DEFAULT nextval('programming_languages_id_seq'::regclass);

ALTER TABLE ONLY project_aliases ALTER COLUMN id SET DEFAULT nextval('project_aliases_id_seq'::regclass);

ALTER TABLE ONLY project_auto_devops ALTER COLUMN id SET DEFAULT nextval('project_auto_devops_id_seq'::regclass);

ALTER TABLE ONLY project_ci_cd_settings ALTER COLUMN id SET DEFAULT nextval('project_ci_cd_settings_id_seq'::regclass);

ALTER TABLE ONLY project_ci_feature_usages ALTER COLUMN id SET DEFAULT nextval('project_ci_feature_usages_id_seq'::regclass);

ALTER TABLE ONLY project_compliance_framework_settings ALTER COLUMN project_id SET DEFAULT nextval('project_compliance_framework_settings_project_id_seq'::regclass);

ALTER TABLE ONLY project_custom_attributes ALTER COLUMN id SET DEFAULT nextval('project_custom_attributes_id_seq'::regclass);

ALTER TABLE ONLY project_daily_statistics ALTER COLUMN id SET DEFAULT nextval('project_daily_statistics_id_seq'::regclass);

ALTER TABLE ONLY project_deploy_tokens ALTER COLUMN id SET DEFAULT nextval('project_deploy_tokens_id_seq'::regclass);

ALTER TABLE ONLY project_export_jobs ALTER COLUMN id SET DEFAULT nextval('project_export_jobs_id_seq'::regclass);

ALTER TABLE ONLY project_features ALTER COLUMN id SET DEFAULT nextval('project_features_id_seq'::regclass);

ALTER TABLE ONLY project_group_links ALTER COLUMN id SET DEFAULT nextval('project_group_links_id_seq'::regclass);

ALTER TABLE ONLY project_import_data ALTER COLUMN id SET DEFAULT nextval('project_import_data_id_seq'::regclass);

ALTER TABLE ONLY project_mirror_data ALTER COLUMN id SET DEFAULT nextval('project_mirror_data_id_seq'::regclass);

ALTER TABLE ONLY project_repositories ALTER COLUMN id SET DEFAULT nextval('project_repositories_id_seq'::regclass);

ALTER TABLE ONLY project_repository_states ALTER COLUMN id SET DEFAULT nextval('project_repository_states_id_seq'::regclass);

ALTER TABLE ONLY project_repository_storage_moves ALTER COLUMN id SET DEFAULT nextval('project_repository_storage_moves_id_seq'::regclass);

ALTER TABLE ONLY project_security_settings ALTER COLUMN project_id SET DEFAULT nextval('project_security_settings_project_id_seq'::regclass);

ALTER TABLE ONLY project_statistics ALTER COLUMN id SET DEFAULT nextval('project_statistics_id_seq'::regclass);

ALTER TABLE ONLY project_topics ALTER COLUMN id SET DEFAULT nextval('project_topics_id_seq'::regclass);

ALTER TABLE ONLY project_tracing_settings ALTER COLUMN id SET DEFAULT nextval('project_tracing_settings_id_seq'::regclass);

ALTER TABLE ONLY projects ALTER COLUMN id SET DEFAULT nextval('projects_id_seq'::regclass);

ALTER TABLE ONLY projects_sync_events ALTER COLUMN id SET DEFAULT nextval('projects_sync_events_id_seq'::regclass);

ALTER TABLE ONLY prometheus_alert_events ALTER COLUMN id SET DEFAULT nextval('prometheus_alert_events_id_seq'::regclass);

ALTER TABLE ONLY prometheus_alerts ALTER COLUMN id SET DEFAULT nextval('prometheus_alerts_id_seq'::regclass);

ALTER TABLE ONLY prometheus_metrics ALTER COLUMN id SET DEFAULT nextval('prometheus_metrics_id_seq'::regclass);

ALTER TABLE ONLY protected_branch_merge_access_levels ALTER COLUMN id SET DEFAULT nextval('protected_branch_merge_access_levels_id_seq'::regclass);

ALTER TABLE ONLY protected_branch_push_access_levels ALTER COLUMN id SET DEFAULT nextval('protected_branch_push_access_levels_id_seq'::regclass);

ALTER TABLE ONLY protected_branch_unprotect_access_levels ALTER COLUMN id SET DEFAULT nextval('protected_branch_unprotect_access_levels_id_seq'::regclass);

ALTER TABLE ONLY protected_branches ALTER COLUMN id SET DEFAULT nextval('protected_branches_id_seq'::regclass);

ALTER TABLE ONLY protected_environment_deploy_access_levels ALTER COLUMN id SET DEFAULT nextval('protected_environment_deploy_access_levels_id_seq'::regclass);

ALTER TABLE ONLY protected_environments ALTER COLUMN id SET DEFAULT nextval('protected_environments_id_seq'::regclass);

ALTER TABLE ONLY protected_tag_create_access_levels ALTER COLUMN id SET DEFAULT nextval('protected_tag_create_access_levels_id_seq'::regclass);

ALTER TABLE ONLY protected_tags ALTER COLUMN id SET DEFAULT nextval('protected_tags_id_seq'::regclass);

ALTER TABLE ONLY push_rules ALTER COLUMN id SET DEFAULT nextval('push_rules_id_seq'::regclass);

ALTER TABLE ONLY raw_usage_data ALTER COLUMN id SET DEFAULT nextval('raw_usage_data_id_seq'::regclass);

ALTER TABLE ONLY redirect_routes ALTER COLUMN id SET DEFAULT nextval('redirect_routes_id_seq'::regclass);

ALTER TABLE ONLY release_links ALTER COLUMN id SET DEFAULT nextval('release_links_id_seq'::regclass);

ALTER TABLE ONLY releases ALTER COLUMN id SET DEFAULT nextval('releases_id_seq'::regclass);

ALTER TABLE ONLY remote_mirrors ALTER COLUMN id SET DEFAULT nextval('remote_mirrors_id_seq'::regclass);

ALTER TABLE ONLY required_code_owners_sections ALTER COLUMN id SET DEFAULT nextval('required_code_owners_sections_id_seq'::regclass);

ALTER TABLE ONLY requirements ALTER COLUMN id SET DEFAULT nextval('requirements_id_seq'::regclass);

ALTER TABLE ONLY requirements_management_test_reports ALTER COLUMN id SET DEFAULT nextval('requirements_management_test_reports_id_seq'::regclass);

ALTER TABLE ONLY resource_iteration_events ALTER COLUMN id SET DEFAULT nextval('resource_iteration_events_id_seq'::regclass);

ALTER TABLE ONLY resource_label_events ALTER COLUMN id SET DEFAULT nextval('resource_label_events_id_seq'::regclass);

ALTER TABLE ONLY resource_milestone_events ALTER COLUMN id SET DEFAULT nextval('resource_milestone_events_id_seq'::regclass);

ALTER TABLE ONLY resource_state_events ALTER COLUMN id SET DEFAULT nextval('resource_state_events_id_seq'::regclass);

ALTER TABLE ONLY resource_weight_events ALTER COLUMN id SET DEFAULT nextval('resource_weight_events_id_seq'::regclass);

ALTER TABLE ONLY reviews ALTER COLUMN id SET DEFAULT nextval('reviews_id_seq'::regclass);

ALTER TABLE ONLY routes ALTER COLUMN id SET DEFAULT nextval('routes_id_seq'::regclass);

ALTER TABLE ONLY saml_group_links ALTER COLUMN id SET DEFAULT nextval('saml_group_links_id_seq'::regclass);

ALTER TABLE ONLY saml_providers ALTER COLUMN id SET DEFAULT nextval('saml_providers_id_seq'::regclass);

ALTER TABLE ONLY scim_identities ALTER COLUMN id SET DEFAULT nextval('scim_identities_id_seq'::regclass);

ALTER TABLE ONLY scim_oauth_access_tokens ALTER COLUMN id SET DEFAULT nextval('scim_oauth_access_tokens_id_seq'::regclass);

ALTER TABLE ONLY security_findings ALTER COLUMN id SET DEFAULT nextval('security_findings_id_seq'::regclass);

ALTER TABLE ONLY security_orchestration_policy_configurations ALTER COLUMN id SET DEFAULT nextval('security_orchestration_policy_configurations_id_seq'::regclass);

ALTER TABLE ONLY security_orchestration_policy_rule_schedules ALTER COLUMN id SET DEFAULT nextval('security_orchestration_policy_rule_schedules_id_seq'::regclass);

ALTER TABLE ONLY security_scans ALTER COLUMN id SET DEFAULT nextval('security_scans_id_seq'::regclass);

ALTER TABLE ONLY self_managed_prometheus_alert_events ALTER COLUMN id SET DEFAULT nextval('self_managed_prometheus_alert_events_id_seq'::regclass);

ALTER TABLE ONLY sent_notifications ALTER COLUMN id SET DEFAULT nextval('sent_notifications_id_seq'::regclass);

ALTER TABLE ONLY sentry_issues ALTER COLUMN id SET DEFAULT nextval('sentry_issues_id_seq'::regclass);

ALTER TABLE ONLY shards ALTER COLUMN id SET DEFAULT nextval('shards_id_seq'::regclass);

ALTER TABLE ONLY slack_integrations ALTER COLUMN id SET DEFAULT nextval('slack_integrations_id_seq'::regclass);

ALTER TABLE ONLY smartcard_identities ALTER COLUMN id SET DEFAULT nextval('smartcard_identities_id_seq'::regclass);

ALTER TABLE ONLY snippet_repository_storage_moves ALTER COLUMN id SET DEFAULT nextval('snippet_repository_storage_moves_id_seq'::regclass);

ALTER TABLE ONLY snippet_user_mentions ALTER COLUMN id SET DEFAULT nextval('snippet_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY snippets ALTER COLUMN id SET DEFAULT nextval('snippets_id_seq'::regclass);

ALTER TABLE ONLY software_license_policies ALTER COLUMN id SET DEFAULT nextval('software_license_policies_id_seq'::regclass);

ALTER TABLE ONLY software_licenses ALTER COLUMN id SET DEFAULT nextval('software_licenses_id_seq'::regclass);

ALTER TABLE ONLY spam_logs ALTER COLUMN id SET DEFAULT nextval('spam_logs_id_seq'::regclass);

ALTER TABLE ONLY sprints ALTER COLUMN id SET DEFAULT nextval('sprints_id_seq'::regclass);

ALTER TABLE ONLY status_check_responses ALTER COLUMN id SET DEFAULT nextval('status_check_responses_id_seq'::regclass);

ALTER TABLE ONLY status_page_published_incidents ALTER COLUMN id SET DEFAULT nextval('status_page_published_incidents_id_seq'::regclass);

ALTER TABLE ONLY status_page_settings ALTER COLUMN project_id SET DEFAULT nextval('status_page_settings_project_id_seq'::regclass);

ALTER TABLE ONLY subscriptions ALTER COLUMN id SET DEFAULT nextval('subscriptions_id_seq'::regclass);

ALTER TABLE ONLY suggestions ALTER COLUMN id SET DEFAULT nextval('suggestions_id_seq'::regclass);

ALTER TABLE ONLY system_note_metadata ALTER COLUMN id SET DEFAULT nextval('system_note_metadata_id_seq'::regclass);

ALTER TABLE ONLY taggings ALTER COLUMN id SET DEFAULT nextval('taggings_id_seq'::regclass);

ALTER TABLE ONLY tags ALTER COLUMN id SET DEFAULT nextval('tags_id_seq'::regclass);

ALTER TABLE ONLY term_agreements ALTER COLUMN id SET DEFAULT nextval('term_agreements_id_seq'::regclass);

ALTER TABLE ONLY terraform_state_versions ALTER COLUMN id SET DEFAULT nextval('terraform_state_versions_id_seq'::regclass);

ALTER TABLE ONLY terraform_states ALTER COLUMN id SET DEFAULT nextval('terraform_states_id_seq'::regclass);

ALTER TABLE ONLY timelogs ALTER COLUMN id SET DEFAULT nextval('timelogs_id_seq'::regclass);

ALTER TABLE ONLY todos ALTER COLUMN id SET DEFAULT nextval('todos_id_seq'::regclass);

ALTER TABLE ONLY token_with_ivs ALTER COLUMN id SET DEFAULT nextval('token_with_ivs_id_seq'::regclass);

ALTER TABLE ONLY topics ALTER COLUMN id SET DEFAULT nextval('topics_id_seq'::regclass);

ALTER TABLE ONLY trending_projects ALTER COLUMN id SET DEFAULT nextval('trending_projects_id_seq'::regclass);

ALTER TABLE ONLY u2f_registrations ALTER COLUMN id SET DEFAULT nextval('u2f_registrations_id_seq'::regclass);

ALTER TABLE ONLY upcoming_reconciliations ALTER COLUMN id SET DEFAULT nextval('upcoming_reconciliations_id_seq'::regclass);

ALTER TABLE ONLY upload_states ALTER COLUMN upload_id SET DEFAULT nextval('upload_states_upload_id_seq'::regclass);

ALTER TABLE ONLY uploads ALTER COLUMN id SET DEFAULT nextval('uploads_id_seq'::regclass);

ALTER TABLE ONLY user_agent_details ALTER COLUMN id SET DEFAULT nextval('user_agent_details_id_seq'::regclass);

ALTER TABLE ONLY user_callouts ALTER COLUMN id SET DEFAULT nextval('user_callouts_id_seq'::regclass);

ALTER TABLE ONLY user_canonical_emails ALTER COLUMN id SET DEFAULT nextval('user_canonical_emails_id_seq'::regclass);

ALTER TABLE ONLY user_custom_attributes ALTER COLUMN id SET DEFAULT nextval('user_custom_attributes_id_seq'::regclass);

ALTER TABLE ONLY user_details ALTER COLUMN user_id SET DEFAULT nextval('user_details_user_id_seq'::regclass);

ALTER TABLE ONLY user_group_callouts ALTER COLUMN id SET DEFAULT nextval('user_group_callouts_id_seq'::regclass);

ALTER TABLE ONLY user_permission_export_uploads ALTER COLUMN id SET DEFAULT nextval('user_permission_export_uploads_id_seq'::regclass);

ALTER TABLE ONLY user_preferences ALTER COLUMN id SET DEFAULT nextval('user_preferences_id_seq'::regclass);

ALTER TABLE ONLY user_statuses ALTER COLUMN user_id SET DEFAULT nextval('user_statuses_user_id_seq'::regclass);

ALTER TABLE ONLY user_synced_attributes_metadata ALTER COLUMN id SET DEFAULT nextval('user_synced_attributes_metadata_id_seq'::regclass);

ALTER TABLE ONLY users ALTER COLUMN id SET DEFAULT nextval('users_id_seq'::regclass);

ALTER TABLE ONLY users_ops_dashboard_projects ALTER COLUMN id SET DEFAULT nextval('users_ops_dashboard_projects_id_seq'::regclass);

ALTER TABLE ONLY users_star_projects ALTER COLUMN id SET DEFAULT nextval('users_star_projects_id_seq'::regclass);

ALTER TABLE ONLY users_statistics ALTER COLUMN id SET DEFAULT nextval('users_statistics_id_seq'::regclass);

ALTER TABLE ONLY vulnerabilities ALTER COLUMN id SET DEFAULT nextval('vulnerabilities_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_exports ALTER COLUMN id SET DEFAULT nextval('vulnerability_exports_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_external_issue_links ALTER COLUMN id SET DEFAULT nextval('vulnerability_external_issue_links_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_feedback ALTER COLUMN id SET DEFAULT nextval('vulnerability_feedback_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_assets ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_assets_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_headers ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_headers_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_requests ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_requests_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_responses ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_responses_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_sources ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_sources_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidence_supporting_messages ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidence_supporting_messages_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_evidences ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_evidences_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_links ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_links_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_finding_signatures ALTER COLUMN id SET DEFAULT nextval('vulnerability_finding_signatures_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_findings_remediations ALTER COLUMN id SET DEFAULT nextval('vulnerability_findings_remediations_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_flags ALTER COLUMN id SET DEFAULT nextval('vulnerability_flags_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_historical_statistics ALTER COLUMN id SET DEFAULT nextval('vulnerability_historical_statistics_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_identifiers ALTER COLUMN id SET DEFAULT nextval('vulnerability_identifiers_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_issue_links ALTER COLUMN id SET DEFAULT nextval('vulnerability_issue_links_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_occurrence_identifiers ALTER COLUMN id SET DEFAULT nextval('vulnerability_occurrence_identifiers_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_occurrence_pipelines ALTER COLUMN id SET DEFAULT nextval('vulnerability_occurrence_pipelines_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_occurrences ALTER COLUMN id SET DEFAULT nextval('vulnerability_occurrences_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_reads ALTER COLUMN id SET DEFAULT nextval('vulnerability_reads_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_remediations ALTER COLUMN id SET DEFAULT nextval('vulnerability_remediations_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_scanners ALTER COLUMN id SET DEFAULT nextval('vulnerability_scanners_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_statistics ALTER COLUMN id SET DEFAULT nextval('vulnerability_statistics_id_seq'::regclass);

ALTER TABLE ONLY vulnerability_user_mentions ALTER COLUMN id SET DEFAULT nextval('vulnerability_user_mentions_id_seq'::regclass);

ALTER TABLE ONLY web_hook_logs ALTER COLUMN id SET DEFAULT nextval('web_hook_logs_id_seq'::regclass);

ALTER TABLE ONLY web_hooks ALTER COLUMN id SET DEFAULT nextval('web_hooks_id_seq'::regclass);

ALTER TABLE ONLY webauthn_registrations ALTER COLUMN id SET DEFAULT nextval('webauthn_registrations_id_seq'::regclass);

ALTER TABLE ONLY wiki_page_meta ALTER COLUMN id SET DEFAULT nextval('wiki_page_meta_id_seq'::regclass);

ALTER TABLE ONLY wiki_page_slugs ALTER COLUMN id SET DEFAULT nextval('wiki_page_slugs_id_seq'::regclass);

ALTER TABLE ONLY work_item_types ALTER COLUMN id SET DEFAULT nextval('work_item_types_id_seq'::regclass);

ALTER TABLE ONLY x509_certificates ALTER COLUMN id SET DEFAULT nextval('x509_certificates_id_seq'::regclass);

ALTER TABLE ONLY x509_commit_signatures ALTER COLUMN id SET DEFAULT nextval('x509_commit_signatures_id_seq'::regclass);

ALTER TABLE ONLY x509_issuers ALTER COLUMN id SET DEFAULT nextval('x509_issuers_id_seq'::regclass);

ALTER TABLE ONLY zentao_tracker_data ALTER COLUMN id SET DEFAULT nextval('zentao_tracker_data_id_seq'::regclass);

ALTER TABLE ONLY zoom_meetings ALTER COLUMN id SET DEFAULT nextval('zoom_meetings_id_seq'::regclass);

ALTER TABLE ONLY audit_events
    ADD CONSTRAINT audit_events_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_000000
    ADD CONSTRAINT audit_events_000000_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202212
    ADD CONSTRAINT audit_events_202212_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202301
    ADD CONSTRAINT audit_events_202301_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202302
    ADD CONSTRAINT audit_events_202302_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202303
    ADD CONSTRAINT audit_events_202303_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202304
    ADD CONSTRAINT audit_events_202304_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202305
    ADD CONSTRAINT audit_events_202305_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202306
    ADD CONSTRAINT audit_events_202306_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202307
    ADD CONSTRAINT audit_events_202307_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202308
    ADD CONSTRAINT audit_events_202308_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.audit_events_202309
    ADD CONSTRAINT audit_events_202309_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY batched_background_migration_job_transition_logs
    ADD CONSTRAINT batched_background_migration_job_transition_logs_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_000000
    ADD CONSTRAINT batched_background_migration_job_transition_logs_000000_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202303
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202303_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202304
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202304_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202305
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202305_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202306
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202306_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202307
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202307_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202308
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202308_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202309
    ADD CONSTRAINT batched_background_migration_job_transition_logs_202309_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202211
    ADD CONSTRAINT incident_management_pending_alert_escalations_202211_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202212
    ADD CONSTRAINT incident_management_pending_alert_escalations_202212_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY incident_management_pending_alert_escalations
    ADD CONSTRAINT incident_management_pending_alert_escalations_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202301
    ADD CONSTRAINT incident_management_pending_alert_escalations_202301_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202302
    ADD CONSTRAINT incident_management_pending_alert_escalations_202302_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202303
    ADD CONSTRAINT incident_management_pending_alert_escalations_202303_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202304
    ADD CONSTRAINT incident_management_pending_alert_escalations_202304_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202305
    ADD CONSTRAINT incident_management_pending_alert_escalations_202305_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202306
    ADD CONSTRAINT incident_management_pending_alert_escalations_202306_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202307
    ADD CONSTRAINT incident_management_pending_alert_escalations_202307_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202308
    ADD CONSTRAINT incident_management_pending_alert_escalations_202308_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202309
    ADD CONSTRAINT incident_management_pending_alert_escalations_202309_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202211
    ADD CONSTRAINT incident_management_pending_issue_escalations_202211_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202212
    ADD CONSTRAINT incident_management_pending_issue_escalations_202212_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY incident_management_pending_issue_escalations
    ADD CONSTRAINT incident_management_pending_issue_escalations_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202301
    ADD CONSTRAINT incident_management_pending_issue_escalations_202301_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202302
    ADD CONSTRAINT incident_management_pending_issue_escalations_202302_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202303
    ADD CONSTRAINT incident_management_pending_issue_escalations_202303_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202304
    ADD CONSTRAINT incident_management_pending_issue_escalations_202304_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202305
    ADD CONSTRAINT incident_management_pending_issue_escalations_202305_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202306
    ADD CONSTRAINT incident_management_pending_issue_escalations_202306_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202307
    ADD CONSTRAINT incident_management_pending_issue_escalations_202307_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202308
    ADD CONSTRAINT incident_management_pending_issue_escalations_202308_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202309
    ADD CONSTRAINT incident_management_pending_issue_escalations_202309_pkey PRIMARY KEY (id, process_at);

ALTER TABLE ONLY verification_codes
    ADD CONSTRAINT verification_codes_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_000000
    ADD CONSTRAINT verification_codes_000000_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202302
    ADD CONSTRAINT verification_codes_202302_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202303
    ADD CONSTRAINT verification_codes_202303_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202304
    ADD CONSTRAINT verification_codes_202304_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202305
    ADD CONSTRAINT verification_codes_202305_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202306
    ADD CONSTRAINT verification_codes_202306_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202307
    ADD CONSTRAINT verification_codes_202307_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202308
    ADD CONSTRAINT verification_codes_202308_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.verification_codes_202309
    ADD CONSTRAINT verification_codes_202309_pkey PRIMARY KEY (created_at, visitor_id_code, code, phone);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_000000
    ADD CONSTRAINT web_hook_logs_000000_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY web_hook_logs
    ADD CONSTRAINT web_hook_logs_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202212
    ADD CONSTRAINT web_hook_logs_202212_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202301
    ADD CONSTRAINT web_hook_logs_202301_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202302
    ADD CONSTRAINT web_hook_logs_202302_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202303
    ADD CONSTRAINT web_hook_logs_202303_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202304
    ADD CONSTRAINT web_hook_logs_202304_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202305
    ADD CONSTRAINT web_hook_logs_202305_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202306
    ADD CONSTRAINT web_hook_logs_202306_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202307
    ADD CONSTRAINT web_hook_logs_202307_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202308
    ADD CONSTRAINT web_hook_logs_202308_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY gitlab_partitions_dynamic.web_hook_logs_202309
    ADD CONSTRAINT web_hook_logs_202309_pkey PRIMARY KEY (id, created_at);

ALTER TABLE ONLY analytics_cycle_analytics_issue_stage_events
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_00
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_00_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_01
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_01_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_02
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_02_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_03
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_03_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_04
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_04_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_05
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_05_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_06
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_06_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_07
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_07_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_08
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_08_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_09
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_09_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_10
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_10_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_11
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_11_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_12
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_12_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_13
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_13_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_14
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_14_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_15
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_15_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_16
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_16_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_17
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_17_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_18
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_18_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_19
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_19_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_20
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_20_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_21
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_21_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_22
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_22_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_23
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_23_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_24
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_24_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_25
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_25_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_26
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_26_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_27
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_27_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_28
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_28_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_29
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_29_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_30
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_30_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_issue_stage_events_31
    ADD CONSTRAINT analytics_cycle_analytics_issue_stage_events_31_pkey PRIMARY KEY (stage_event_hash_id, issue_id);

ALTER TABLE ONLY analytics_cycle_analytics_merge_request_stage_events
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_00
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_00_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_01
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_01_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_02
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_02_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_03
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_03_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_04
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_04_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_05
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_05_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_06
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_06_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_07
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_07_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_08
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_08_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_09
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_09_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_10
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_10_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_11
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_11_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_12
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_12_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_13
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_13_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_14
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_14_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_15
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_15_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_16
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_16_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_17
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_17_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_18
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_18_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_19
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_19_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_20
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_20_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_21
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_21_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_22
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_22_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_23
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_23_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_24
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_24_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_25
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_25_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_26
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_26_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_27
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_27_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_28
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_28_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_29
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_29_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_30
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_30_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY gitlab_partitions_static.analytics_cycle_analytics_merge_request_stage_events_31
    ADD CONSTRAINT analytics_cycle_analytics_merge_request_stage_events_31_pkey PRIMARY KEY (stage_event_hash_id, merge_request_id);

ALTER TABLE ONLY issue_search_data
    ADD CONSTRAINT issue_search_data_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_00
    ADD CONSTRAINT issue_search_data_00_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_01
    ADD CONSTRAINT issue_search_data_01_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_02
    ADD CONSTRAINT issue_search_data_02_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_03
    ADD CONSTRAINT issue_search_data_03_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_04
    ADD CONSTRAINT issue_search_data_04_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_05
    ADD CONSTRAINT issue_search_data_05_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_06
    ADD CONSTRAINT issue_search_data_06_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_07
    ADD CONSTRAINT issue_search_data_07_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_08
    ADD CONSTRAINT issue_search_data_08_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_09
    ADD CONSTRAINT issue_search_data_09_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_10
    ADD CONSTRAINT issue_search_data_10_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_11
    ADD CONSTRAINT issue_search_data_11_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_12
    ADD CONSTRAINT issue_search_data_12_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_13
    ADD CONSTRAINT issue_search_data_13_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_14
    ADD CONSTRAINT issue_search_data_14_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_15
    ADD CONSTRAINT issue_search_data_15_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_16
    ADD CONSTRAINT issue_search_data_16_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_17
    ADD CONSTRAINT issue_search_data_17_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_18
    ADD CONSTRAINT issue_search_data_18_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_19
    ADD CONSTRAINT issue_search_data_19_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_20
    ADD CONSTRAINT issue_search_data_20_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_21
    ADD CONSTRAINT issue_search_data_21_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_22
    ADD CONSTRAINT issue_search_data_22_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_23
    ADD CONSTRAINT issue_search_data_23_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_24
    ADD CONSTRAINT issue_search_data_24_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_25
    ADD CONSTRAINT issue_search_data_25_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_26
    ADD CONSTRAINT issue_search_data_26_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_27
    ADD CONSTRAINT issue_search_data_27_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_28
    ADD CONSTRAINT issue_search_data_28_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_29
    ADD CONSTRAINT issue_search_data_29_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_30
    ADD CONSTRAINT issue_search_data_30_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_31
    ADD CONSTRAINT issue_search_data_31_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_32
    ADD CONSTRAINT issue_search_data_32_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_33
    ADD CONSTRAINT issue_search_data_33_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_34
    ADD CONSTRAINT issue_search_data_34_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_35
    ADD CONSTRAINT issue_search_data_35_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_36
    ADD CONSTRAINT issue_search_data_36_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_37
    ADD CONSTRAINT issue_search_data_37_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_38
    ADD CONSTRAINT issue_search_data_38_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_39
    ADD CONSTRAINT issue_search_data_39_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_40
    ADD CONSTRAINT issue_search_data_40_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_41
    ADD CONSTRAINT issue_search_data_41_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_42
    ADD CONSTRAINT issue_search_data_42_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_43
    ADD CONSTRAINT issue_search_data_43_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_44
    ADD CONSTRAINT issue_search_data_44_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_45
    ADD CONSTRAINT issue_search_data_45_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_46
    ADD CONSTRAINT issue_search_data_46_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_47
    ADD CONSTRAINT issue_search_data_47_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_48
    ADD CONSTRAINT issue_search_data_48_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_49
    ADD CONSTRAINT issue_search_data_49_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_50
    ADD CONSTRAINT issue_search_data_50_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_51
    ADD CONSTRAINT issue_search_data_51_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_52
    ADD CONSTRAINT issue_search_data_52_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_53
    ADD CONSTRAINT issue_search_data_53_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_54
    ADD CONSTRAINT issue_search_data_54_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_55
    ADD CONSTRAINT issue_search_data_55_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_56
    ADD CONSTRAINT issue_search_data_56_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_57
    ADD CONSTRAINT issue_search_data_57_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_58
    ADD CONSTRAINT issue_search_data_58_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_59
    ADD CONSTRAINT issue_search_data_59_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_60
    ADD CONSTRAINT issue_search_data_60_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_61
    ADD CONSTRAINT issue_search_data_61_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_62
    ADD CONSTRAINT issue_search_data_62_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY gitlab_partitions_static.issue_search_data_63
    ADD CONSTRAINT issue_search_data_63_pkey PRIMARY KEY (project_id, issue_id);

ALTER TABLE ONLY loose_foreign_keys_deleted_records
    ADD CONSTRAINT loose_foreign_keys_deleted_records_pkey PRIMARY KEY (partition, id);

ALTER TABLE ONLY gitlab_partitions_static.loose_foreign_keys_deleted_records_1
    ADD CONSTRAINT loose_foreign_keys_deleted_records_1_pkey PRIMARY KEY (partition, id);

ALTER TABLE ONLY product_analytics_events_experimental
    ADD CONSTRAINT product_analytics_events_experimental_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_00
    ADD CONSTRAINT product_analytics_events_experimental_00_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_01
    ADD CONSTRAINT product_analytics_events_experimental_01_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_02
    ADD CONSTRAINT product_analytics_events_experimental_02_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_03
    ADD CONSTRAINT product_analytics_events_experimental_03_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_04
    ADD CONSTRAINT product_analytics_events_experimental_04_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_05
    ADD CONSTRAINT product_analytics_events_experimental_05_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_06
    ADD CONSTRAINT product_analytics_events_experimental_06_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_07
    ADD CONSTRAINT product_analytics_events_experimental_07_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_08
    ADD CONSTRAINT product_analytics_events_experimental_08_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_09
    ADD CONSTRAINT product_analytics_events_experimental_09_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_10
    ADD CONSTRAINT product_analytics_events_experimental_10_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_11
    ADD CONSTRAINT product_analytics_events_experimental_11_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_12
    ADD CONSTRAINT product_analytics_events_experimental_12_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_13
    ADD CONSTRAINT product_analytics_events_experimental_13_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_14
    ADD CONSTRAINT product_analytics_events_experimental_14_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_15
    ADD CONSTRAINT product_analytics_events_experimental_15_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_16
    ADD CONSTRAINT product_analytics_events_experimental_16_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_17
    ADD CONSTRAINT product_analytics_events_experimental_17_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_18
    ADD CONSTRAINT product_analytics_events_experimental_18_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_19
    ADD CONSTRAINT product_analytics_events_experimental_19_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_20
    ADD CONSTRAINT product_analytics_events_experimental_20_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_21
    ADD CONSTRAINT product_analytics_events_experimental_21_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_22
    ADD CONSTRAINT product_analytics_events_experimental_22_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_23
    ADD CONSTRAINT product_analytics_events_experimental_23_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_24
    ADD CONSTRAINT product_analytics_events_experimental_24_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_25
    ADD CONSTRAINT product_analytics_events_experimental_25_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_26
    ADD CONSTRAINT product_analytics_events_experimental_26_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_27
    ADD CONSTRAINT product_analytics_events_experimental_27_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_28
    ADD CONSTRAINT product_analytics_events_experimental_28_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_29
    ADD CONSTRAINT product_analytics_events_experimental_29_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_30
    ADD CONSTRAINT product_analytics_events_experimental_30_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_31
    ADD CONSTRAINT product_analytics_events_experimental_31_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_32
    ADD CONSTRAINT product_analytics_events_experimental_32_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_33
    ADD CONSTRAINT product_analytics_events_experimental_33_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_34
    ADD CONSTRAINT product_analytics_events_experimental_34_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_35
    ADD CONSTRAINT product_analytics_events_experimental_35_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_36
    ADD CONSTRAINT product_analytics_events_experimental_36_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_37
    ADD CONSTRAINT product_analytics_events_experimental_37_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_38
    ADD CONSTRAINT product_analytics_events_experimental_38_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_39
    ADD CONSTRAINT product_analytics_events_experimental_39_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_40
    ADD CONSTRAINT product_analytics_events_experimental_40_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_41
    ADD CONSTRAINT product_analytics_events_experimental_41_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_42
    ADD CONSTRAINT product_analytics_events_experimental_42_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_43
    ADD CONSTRAINT product_analytics_events_experimental_43_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_44
    ADD CONSTRAINT product_analytics_events_experimental_44_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_45
    ADD CONSTRAINT product_analytics_events_experimental_45_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_46
    ADD CONSTRAINT product_analytics_events_experimental_46_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_47
    ADD CONSTRAINT product_analytics_events_experimental_47_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_48
    ADD CONSTRAINT product_analytics_events_experimental_48_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_49
    ADD CONSTRAINT product_analytics_events_experimental_49_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_50
    ADD CONSTRAINT product_analytics_events_experimental_50_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_51
    ADD CONSTRAINT product_analytics_events_experimental_51_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_52
    ADD CONSTRAINT product_analytics_events_experimental_52_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_53
    ADD CONSTRAINT product_analytics_events_experimental_53_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_54
    ADD CONSTRAINT product_analytics_events_experimental_54_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_55
    ADD CONSTRAINT product_analytics_events_experimental_55_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_56
    ADD CONSTRAINT product_analytics_events_experimental_56_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_57
    ADD CONSTRAINT product_analytics_events_experimental_57_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_58
    ADD CONSTRAINT product_analytics_events_experimental_58_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_59
    ADD CONSTRAINT product_analytics_events_experimental_59_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_60
    ADD CONSTRAINT product_analytics_events_experimental_60_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_61
    ADD CONSTRAINT product_analytics_events_experimental_61_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_62
    ADD CONSTRAINT product_analytics_events_experimental_62_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY gitlab_partitions_static.product_analytics_events_experimental_63
    ADD CONSTRAINT product_analytics_events_experimental_63_pkey PRIMARY KEY (id, project_id);

ALTER TABLE ONLY abuse_reports
    ADD CONSTRAINT abuse_reports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY agent_activity_events
    ADD CONSTRAINT agent_activity_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY agent_group_authorizations
    ADD CONSTRAINT agent_group_authorizations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY agent_project_authorizations
    ADD CONSTRAINT agent_project_authorizations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY alert_management_alert_assignees
    ADD CONSTRAINT alert_management_alert_assignees_pkey PRIMARY KEY (id);

ALTER TABLE ONLY alert_management_alert_user_mentions
    ADD CONSTRAINT alert_management_alert_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY alert_management_alerts
    ADD CONSTRAINT alert_management_alerts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY alert_management_http_integrations
    ADD CONSTRAINT alert_management_http_integrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY allowed_email_domains
    ADD CONSTRAINT allowed_email_domains_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_cycle_analytics_group_stages
    ADD CONSTRAINT analytics_cycle_analytics_group_stages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_cycle_analytics_group_value_streams
    ADD CONSTRAINT analytics_cycle_analytics_group_value_streams_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_cycle_analytics_project_stages
    ADD CONSTRAINT analytics_cycle_analytics_project_stages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_cycle_analytics_project_value_streams
    ADD CONSTRAINT analytics_cycle_analytics_project_value_streams_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_cycle_analytics_stage_event_hashes
    ADD CONSTRAINT analytics_cycle_analytics_stage_event_hashes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_devops_adoption_segments
    ADD CONSTRAINT analytics_devops_adoption_segments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_devops_adoption_snapshots
    ADD CONSTRAINT analytics_devops_adoption_snapshots_pkey PRIMARY KEY (id);

ALTER TABLE ONLY analytics_language_trend_repository_languages
    ADD CONSTRAINT analytics_language_trend_repository_languages_pkey PRIMARY KEY (programming_language_id, project_id, snapshot_date);

ALTER TABLE ONLY analytics_usage_trends_measurements
    ADD CONSTRAINT analytics_usage_trends_measurements_pkey PRIMARY KEY (id);

ALTER TABLE ONLY appearances
    ADD CONSTRAINT appearances_pkey PRIMARY KEY (id);

ALTER TABLE ONLY application_setting_terms
    ADD CONSTRAINT application_setting_terms_pkey PRIMARY KEY (id);

ALTER TABLE ONLY application_settings
    ADD CONSTRAINT application_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_merge_request_rule_sources
    ADD CONSTRAINT approval_merge_request_rule_sources_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_merge_request_rules_approved_approvers
    ADD CONSTRAINT approval_merge_request_rules_approved_approvers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_merge_request_rules_groups
    ADD CONSTRAINT approval_merge_request_rules_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_merge_request_rules
    ADD CONSTRAINT approval_merge_request_rules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_merge_request_rules_users
    ADD CONSTRAINT approval_merge_request_rules_users_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_project_rules_groups
    ADD CONSTRAINT approval_project_rules_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_project_rules
    ADD CONSTRAINT approval_project_rules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approval_project_rules_protected_branches
    ADD CONSTRAINT approval_project_rules_protected_branches_pkey PRIMARY KEY (approval_project_rule_id, protected_branch_id);

ALTER TABLE ONLY approval_project_rules_users
    ADD CONSTRAINT approval_project_rules_users_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approvals
    ADD CONSTRAINT approvals_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approver_groups
    ADD CONSTRAINT approver_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY approvers
    ADD CONSTRAINT approvers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY atlassian_identities
    ADD CONSTRAINT atlassian_identities_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY audit_events_external_audit_event_destinations
    ADD CONSTRAINT audit_events_external_audit_event_destinations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY authentication_events
    ADD CONSTRAINT authentication_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY award_emoji
    ADD CONSTRAINT award_emoji_pkey PRIMARY KEY (id);

ALTER TABLE ONLY aws_roles
    ADD CONSTRAINT aws_roles_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY background_migration_jobs
    ADD CONSTRAINT background_migration_jobs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY badges
    ADD CONSTRAINT badges_pkey PRIMARY KEY (id);

ALTER TABLE ONLY banned_users
    ADD CONSTRAINT banned_users_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY batched_background_migration_jobs
    ADD CONSTRAINT batched_background_migration_jobs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY batched_background_migrations
    ADD CONSTRAINT batched_background_migrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY board_assignees
    ADD CONSTRAINT board_assignees_pkey PRIMARY KEY (id);

ALTER TABLE ONLY board_group_recent_visits
    ADD CONSTRAINT board_group_recent_visits_pkey PRIMARY KEY (id);

ALTER TABLE ONLY board_labels
    ADD CONSTRAINT board_labels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY board_project_recent_visits
    ADD CONSTRAINT board_project_recent_visits_pkey PRIMARY KEY (id);

ALTER TABLE ONLY board_user_preferences
    ADD CONSTRAINT board_user_preferences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_board_labels
    ADD CONSTRAINT boards_epic_board_labels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_board_positions
    ADD CONSTRAINT boards_epic_board_positions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_board_recent_visits
    ADD CONSTRAINT boards_epic_board_recent_visits_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_boards
    ADD CONSTRAINT boards_epic_boards_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_list_user_preferences
    ADD CONSTRAINT boards_epic_list_user_preferences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_lists
    ADD CONSTRAINT boards_epic_lists_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards_epic_user_preferences
    ADD CONSTRAINT boards_epic_user_preferences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY boards
    ADD CONSTRAINT boards_pkey PRIMARY KEY (id);

ALTER TABLE ONLY broadcast_messages
    ADD CONSTRAINT broadcast_messages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_configurations
    ADD CONSTRAINT bulk_import_configurations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_entities
    ADD CONSTRAINT bulk_import_entities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_export_uploads
    ADD CONSTRAINT bulk_import_export_uploads_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_exports
    ADD CONSTRAINT bulk_import_exports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_failures
    ADD CONSTRAINT bulk_import_failures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_import_trackers
    ADD CONSTRAINT bulk_import_trackers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY bulk_imports
    ADD CONSTRAINT bulk_imports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY chat_names
    ADD CONSTRAINT chat_names_pkey PRIMARY KEY (id);

ALTER TABLE ONLY chat_teams
    ADD CONSTRAINT chat_teams_pkey PRIMARY KEY (id);

ALTER TABLE vulnerability_scanners
    ADD CONSTRAINT check_37608c9db5 CHECK ((char_length(vendor) <= 255)) NOT VALID;

ALTER TABLE security_findings
    ADD CONSTRAINT check_6c2851a8c9 CHECK ((uuid IS NOT NULL)) NOT VALID;

ALTER TABLE sprints
    ADD CONSTRAINT check_ccd8a1eae0 CHECK ((start_date IS NOT NULL)) NOT VALID;

ALTER TABLE group_import_states
    ADD CONSTRAINT check_cda75c7c3f CHECK ((user_id IS NOT NULL)) NOT VALID;

ALTER TABLE sprints
    ADD CONSTRAINT check_df3816aed7 CHECK ((due_date IS NOT NULL)) NOT VALID;

ALTER TABLE ONLY ci_build_needs
    ADD CONSTRAINT ci_build_needs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_build_pending_states
    ADD CONSTRAINT ci_build_pending_states_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_build_report_results
    ADD CONSTRAINT ci_build_report_results_pkey PRIMARY KEY (build_id);

ALTER TABLE ONLY ci_build_trace_chunks
    ADD CONSTRAINT ci_build_trace_chunks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_build_trace_metadata
    ADD CONSTRAINT ci_build_trace_metadata_pkey PRIMARY KEY (build_id);

ALTER TABLE ONLY ci_builds_metadata
    ADD CONSTRAINT ci_builds_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_builds
    ADD CONSTRAINT ci_builds_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_builds_runner_session
    ADD CONSTRAINT ci_builds_runner_session_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_daily_build_group_report_results
    ADD CONSTRAINT ci_daily_build_group_report_results_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_deleted_objects
    ADD CONSTRAINT ci_deleted_objects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_freeze_periods
    ADD CONSTRAINT ci_freeze_periods_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_group_variables
    ADD CONSTRAINT ci_group_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_instance_variables
    ADD CONSTRAINT ci_instance_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_job_artifact_states
    ADD CONSTRAINT ci_job_artifact_states_pkey PRIMARY KEY (job_artifact_id);

ALTER TABLE ONLY ci_job_artifacts
    ADD CONSTRAINT ci_job_artifacts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_job_token_project_scope_links
    ADD CONSTRAINT ci_job_token_project_scope_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_job_variables
    ADD CONSTRAINT ci_job_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_minutes_additional_packs
    ADD CONSTRAINT ci_minutes_additional_packs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_namespace_mirrors
    ADD CONSTRAINT ci_namespace_mirrors_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_namespace_monthly_usages
    ADD CONSTRAINT ci_namespace_monthly_usages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pending_builds
    ADD CONSTRAINT ci_pending_builds_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_artifacts
    ADD CONSTRAINT ci_pipeline_artifacts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_chat_data
    ADD CONSTRAINT ci_pipeline_chat_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_messages
    ADD CONSTRAINT ci_pipeline_messages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_schedule_variables
    ADD CONSTRAINT ci_pipeline_schedule_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_schedules
    ADD CONSTRAINT ci_pipeline_schedules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipeline_variables
    ADD CONSTRAINT ci_pipeline_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_pipelines_config
    ADD CONSTRAINT ci_pipelines_config_pkey PRIMARY KEY (pipeline_id);

ALTER TABLE ONLY ci_pipelines
    ADD CONSTRAINT ci_pipelines_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_platform_metrics
    ADD CONSTRAINT ci_platform_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_project_mirrors
    ADD CONSTRAINT ci_project_mirrors_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_project_monthly_usages
    ADD CONSTRAINT ci_project_monthly_usages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_refs
    ADD CONSTRAINT ci_refs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_resource_groups
    ADD CONSTRAINT ci_resource_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_resources
    ADD CONSTRAINT ci_resources_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_runner_namespaces
    ADD CONSTRAINT ci_runner_namespaces_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_runner_projects
    ADD CONSTRAINT ci_runner_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_runners
    ADD CONSTRAINT ci_runners_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_running_builds
    ADD CONSTRAINT ci_running_builds_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_sources_pipelines
    ADD CONSTRAINT ci_sources_pipelines_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_sources_projects
    ADD CONSTRAINT ci_sources_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_stages
    ADD CONSTRAINT ci_stages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_subscriptions_projects
    ADD CONSTRAINT ci_subscriptions_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_trigger_requests
    ADD CONSTRAINT ci_trigger_requests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_triggers
    ADD CONSTRAINT ci_triggers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_unit_test_failures
    ADD CONSTRAINT ci_unit_test_failures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_unit_tests
    ADD CONSTRAINT ci_unit_tests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ci_variables
    ADD CONSTRAINT ci_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_agent_tokens
    ADD CONSTRAINT cluster_agent_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_agents
    ADD CONSTRAINT cluster_agents_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_groups
    ADD CONSTRAINT cluster_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_platforms_kubernetes
    ADD CONSTRAINT cluster_platforms_kubernetes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_projects
    ADD CONSTRAINT cluster_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_providers_aws
    ADD CONSTRAINT cluster_providers_aws_pkey PRIMARY KEY (id);

ALTER TABLE ONLY cluster_providers_gcp
    ADD CONSTRAINT cluster_providers_gcp_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_cert_managers
    ADD CONSTRAINT clusters_applications_cert_managers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_cilium
    ADD CONSTRAINT clusters_applications_cilium_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_crossplane
    ADD CONSTRAINT clusters_applications_crossplane_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_elastic_stacks
    ADD CONSTRAINT clusters_applications_elastic_stacks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_helm
    ADD CONSTRAINT clusters_applications_helm_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_ingress
    ADD CONSTRAINT clusters_applications_ingress_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_jupyter
    ADD CONSTRAINT clusters_applications_jupyter_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_knative
    ADD CONSTRAINT clusters_applications_knative_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_prometheus
    ADD CONSTRAINT clusters_applications_prometheus_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_applications_runners
    ADD CONSTRAINT clusters_applications_runners_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters_integration_elasticstack
    ADD CONSTRAINT clusters_integration_elasticstack_pkey PRIMARY KEY (cluster_id);

ALTER TABLE ONLY clusters_integration_prometheus
    ADD CONSTRAINT clusters_integration_prometheus_pkey PRIMARY KEY (cluster_id);

ALTER TABLE ONLY clusters_kubernetes_namespaces
    ADD CONSTRAINT clusters_kubernetes_namespaces_pkey PRIMARY KEY (id);

ALTER TABLE ONLY clusters
    ADD CONSTRAINT clusters_pkey PRIMARY KEY (id);

ALTER TABLE ONLY commit_user_mentions
    ADD CONSTRAINT commit_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY compliance_management_frameworks
    ADD CONSTRAINT compliance_management_frameworks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY container_expiration_policies
    ADD CONSTRAINT container_expiration_policies_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY container_repositories
    ADD CONSTRAINT container_repositories_pkey PRIMARY KEY (id);

ALTER TABLE ONLY content_blocked_states
    ADD CONSTRAINT content_blocked_states_pkey PRIMARY KEY (id);

ALTER TABLE ONLY conversational_development_index_metrics
    ADD CONSTRAINT conversational_development_index_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY coverage_fuzzing_corpuses
    ADD CONSTRAINT coverage_fuzzing_corpuses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY csv_issue_imports
    ADD CONSTRAINT csv_issue_imports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY custom_emoji
    ADD CONSTRAINT custom_emoji_pkey PRIMARY KEY (id);

ALTER TABLE ONLY customer_relations_contacts
    ADD CONSTRAINT customer_relations_contacts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY customer_relations_organizations
    ADD CONSTRAINT customer_relations_organizations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_profile_schedules
    ADD CONSTRAINT dast_profile_schedules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_profiles_pipelines
    ADD CONSTRAINT dast_profiles_pipelines_pkey PRIMARY KEY (dast_profile_id, ci_pipeline_id);

ALTER TABLE ONLY dast_profiles
    ADD CONSTRAINT dast_profiles_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_scanner_profiles_builds
    ADD CONSTRAINT dast_scanner_profiles_builds_pkey PRIMARY KEY (dast_scanner_profile_id, ci_build_id);

ALTER TABLE ONLY dast_scanner_profiles
    ADD CONSTRAINT dast_scanner_profiles_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_site_profile_secret_variables
    ADD CONSTRAINT dast_site_profile_secret_variables_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_site_profiles_builds
    ADD CONSTRAINT dast_site_profiles_builds_pkey PRIMARY KEY (dast_site_profile_id, ci_build_id);

ALTER TABLE ONLY dast_site_profiles_pipelines
    ADD CONSTRAINT dast_site_profiles_pipelines_pkey PRIMARY KEY (dast_site_profile_id, ci_pipeline_id);

ALTER TABLE ONLY dast_site_profiles
    ADD CONSTRAINT dast_site_profiles_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_site_tokens
    ADD CONSTRAINT dast_site_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_site_validations
    ADD CONSTRAINT dast_site_validations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dast_sites
    ADD CONSTRAINT dast_sites_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dependency_proxy_blobs
    ADD CONSTRAINT dependency_proxy_blobs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dependency_proxy_group_settings
    ADD CONSTRAINT dependency_proxy_group_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dependency_proxy_image_ttl_group_policies
    ADD CONSTRAINT dependency_proxy_image_ttl_group_policies_pkey PRIMARY KEY (group_id);

ALTER TABLE ONLY dependency_proxy_manifests
    ADD CONSTRAINT dependency_proxy_manifests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY deploy_keys_projects
    ADD CONSTRAINT deploy_keys_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY deploy_tokens
    ADD CONSTRAINT deploy_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY deployment_approvals
    ADD CONSTRAINT deployment_approvals_pkey PRIMARY KEY (id);

ALTER TABLE ONLY deployment_clusters
    ADD CONSTRAINT deployment_clusters_pkey PRIMARY KEY (deployment_id);

ALTER TABLE ONLY deployment_merge_requests
    ADD CONSTRAINT deployment_merge_requests_pkey PRIMARY KEY (deployment_id, merge_request_id);

ALTER TABLE ONLY deployments
    ADD CONSTRAINT deployments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY description_versions
    ADD CONSTRAINT description_versions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY design_management_designs
    ADD CONSTRAINT design_management_designs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY design_management_designs_versions
    ADD CONSTRAINT design_management_designs_versions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY design_management_versions
    ADD CONSTRAINT design_management_versions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY design_user_mentions
    ADD CONSTRAINT design_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY detached_partitions
    ADD CONSTRAINT detached_partitions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY diff_note_positions
    ADD CONSTRAINT diff_note_positions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY dora_daily_metrics
    ADD CONSTRAINT dora_daily_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY draft_notes
    ADD CONSTRAINT draft_notes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY elastic_index_settings
    ADD CONSTRAINT elastic_index_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY elastic_reindexing_slices
    ADD CONSTRAINT elastic_reindexing_slices_pkey PRIMARY KEY (id);

ALTER TABLE ONLY elastic_reindexing_subtasks
    ADD CONSTRAINT elastic_reindexing_subtasks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY elastic_reindexing_tasks
    ADD CONSTRAINT elastic_reindexing_tasks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY elasticsearch_indexed_namespaces
    ADD CONSTRAINT elasticsearch_indexed_namespaces_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY elasticsearch_indexed_projects
    ADD CONSTRAINT elasticsearch_indexed_projects_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY emails
    ADD CONSTRAINT emails_pkey PRIMARY KEY (id);

ALTER TABLE ONLY environments
    ADD CONSTRAINT environments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY epic_issues
    ADD CONSTRAINT epic_issues_pkey PRIMARY KEY (id);

ALTER TABLE ONLY epic_metrics
    ADD CONSTRAINT epic_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY epic_user_mentions
    ADD CONSTRAINT epic_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY epics
    ADD CONSTRAINT epics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY error_tracking_client_keys
    ADD CONSTRAINT error_tracking_client_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY error_tracking_error_events
    ADD CONSTRAINT error_tracking_error_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY error_tracking_errors
    ADD CONSTRAINT error_tracking_errors_pkey PRIMARY KEY (id);

ALTER TABLE ONLY events
    ADD CONSTRAINT events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY evidences
    ADD CONSTRAINT evidences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY experiment_subjects
    ADD CONSTRAINT experiment_subjects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY experiment_users
    ADD CONSTRAINT experiment_users_pkey PRIMARY KEY (id);

ALTER TABLE ONLY experiments
    ADD CONSTRAINT experiments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY external_approval_rules
    ADD CONSTRAINT external_approval_rules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY external_approval_rules_protected_branches
    ADD CONSTRAINT external_approval_rules_protected_branches_pkey PRIMARY KEY (id);

ALTER TABLE ONLY external_pull_requests
    ADD CONSTRAINT external_pull_requests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY external_status_checks
    ADD CONSTRAINT external_status_checks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY external_status_checks_protected_branches
    ADD CONSTRAINT external_status_checks_protected_branches_pkey PRIMARY KEY (id);

ALTER TABLE ONLY feature_gates
    ADD CONSTRAINT feature_gates_pkey PRIMARY KEY (id);

ALTER TABLE ONLY features
    ADD CONSTRAINT features_pkey PRIMARY KEY (id);

ALTER TABLE ONLY fork_network_members
    ADD CONSTRAINT fork_network_members_pkey PRIMARY KEY (id);

ALTER TABLE ONLY fork_networks
    ADD CONSTRAINT fork_networks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_cache_invalidation_events
    ADD CONSTRAINT geo_cache_invalidation_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_container_repository_updated_events
    ADD CONSTRAINT geo_container_repository_updated_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_event_log
    ADD CONSTRAINT geo_event_log_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_events
    ADD CONSTRAINT geo_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_hashed_storage_attachments_events
    ADD CONSTRAINT geo_hashed_storage_attachments_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_hashed_storage_migrated_events
    ADD CONSTRAINT geo_hashed_storage_migrated_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_job_artifact_deleted_events
    ADD CONSTRAINT geo_job_artifact_deleted_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_lfs_object_deleted_events
    ADD CONSTRAINT geo_lfs_object_deleted_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_node_namespace_links
    ADD CONSTRAINT geo_node_namespace_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_node_statuses
    ADD CONSTRAINT geo_node_statuses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_nodes
    ADD CONSTRAINT geo_nodes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_repositories_changed_events
    ADD CONSTRAINT geo_repositories_changed_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_repository_created_events
    ADD CONSTRAINT geo_repository_created_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_repository_deleted_events
    ADD CONSTRAINT geo_repository_deleted_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_repository_renamed_events
    ADD CONSTRAINT geo_repository_renamed_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_repository_updated_events
    ADD CONSTRAINT geo_repository_updated_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY geo_reset_checksum_events
    ADD CONSTRAINT geo_reset_checksum_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY gitlab_subscription_histories
    ADD CONSTRAINT gitlab_subscription_histories_pkey PRIMARY KEY (id);

ALTER TABLE ONLY gitlab_subscriptions
    ADD CONSTRAINT gitlab_subscriptions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY gpg_key_subkeys
    ADD CONSTRAINT gpg_key_subkeys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY gpg_keys
    ADD CONSTRAINT gpg_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY gpg_signatures
    ADD CONSTRAINT gpg_signatures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY grafana_integrations
    ADD CONSTRAINT grafana_integrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_custom_attributes
    ADD CONSTRAINT group_custom_attributes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_deletion_schedules
    ADD CONSTRAINT group_deletion_schedules_pkey PRIMARY KEY (group_id);

ALTER TABLE ONLY group_deploy_keys_groups
    ADD CONSTRAINT group_deploy_keys_groups_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_deploy_keys
    ADD CONSTRAINT group_deploy_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_deploy_tokens
    ADD CONSTRAINT group_deploy_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_group_links
    ADD CONSTRAINT group_group_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_import_states
    ADD CONSTRAINT group_import_states_pkey PRIMARY KEY (group_id);

ALTER TABLE ONLY group_merge_request_approval_settings
    ADD CONSTRAINT group_merge_request_approval_settings_pkey PRIMARY KEY (group_id);

ALTER TABLE ONLY group_repository_storage_moves
    ADD CONSTRAINT group_repository_storage_moves_pkey PRIMARY KEY (id);

ALTER TABLE ONLY group_wiki_repositories
    ADD CONSTRAINT group_wiki_repositories_pkey PRIMARY KEY (group_id);

ALTER TABLE ONLY historical_data
    ADD CONSTRAINT historical_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY identities
    ADD CONSTRAINT identities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY import_export_uploads
    ADD CONSTRAINT import_export_uploads_pkey PRIMARY KEY (id);

ALTER TABLE ONLY import_failures
    ADD CONSTRAINT import_failures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY in_product_marketing_emails
    ADD CONSTRAINT in_product_marketing_emails_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_oncall_shifts
    ADD CONSTRAINT inc_mgmnt_no_overlapping_oncall_shifts EXCLUDE USING gist (rotation_id WITH =, tstzrange(starts_at, ends_at, '[)'::text) WITH &&);

ALTER TABLE ONLY incident_management_escalation_policies
    ADD CONSTRAINT incident_management_escalation_policies_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_escalation_rules
    ADD CONSTRAINT incident_management_escalation_rules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_issuable_escalation_statuses
    ADD CONSTRAINT incident_management_issuable_escalation_statuses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_oncall_participants
    ADD CONSTRAINT incident_management_oncall_participants_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_oncall_rotations
    ADD CONSTRAINT incident_management_oncall_rotations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_oncall_schedules
    ADD CONSTRAINT incident_management_oncall_schedules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_oncall_shifts
    ADD CONSTRAINT incident_management_oncall_shifts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY incident_management_timeline_events
    ADD CONSTRAINT incident_management_timeline_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY index_statuses
    ADD CONSTRAINT index_statuses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY insights
    ADD CONSTRAINT insights_pkey PRIMARY KEY (id);

ALTER TABLE ONLY integrations
    ADD CONSTRAINT integrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY internal_ids
    ADD CONSTRAINT internal_ids_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ip_restrictions
    ADD CONSTRAINT ip_restrictions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issuable_metric_images
    ADD CONSTRAINT issuable_metric_images_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issuable_severities
    ADD CONSTRAINT issuable_severities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issuable_slas
    ADD CONSTRAINT issuable_slas_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_assignees
    ADD CONSTRAINT issue_assignees_pkey PRIMARY KEY (issue_id, user_id);

ALTER TABLE ONLY issue_customer_relations_contacts
    ADD CONSTRAINT issue_customer_relations_contacts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_email_participants
    ADD CONSTRAINT issue_email_participants_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_emails
    ADD CONSTRAINT issue_emails_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_links
    ADD CONSTRAINT issue_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_metrics
    ADD CONSTRAINT issue_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_tracker_data
    ADD CONSTRAINT issue_tracker_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issue_user_mentions
    ADD CONSTRAINT issue_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issues
    ADD CONSTRAINT issues_pkey PRIMARY KEY (id);

ALTER TABLE ONLY issues_prometheus_alert_events
    ADD CONSTRAINT issues_prometheus_alert_events_pkey PRIMARY KEY (issue_id, prometheus_alert_event_id);

ALTER TABLE ONLY issues_self_managed_prometheus_alert_events
    ADD CONSTRAINT issues_self_managed_prometheus_alert_events_pkey PRIMARY KEY (issue_id, self_managed_prometheus_alert_event_id);

ALTER TABLE ONLY sprints
    ADD CONSTRAINT iteration_start_and_due_date_iterations_cadence_id_constraint EXCLUDE USING gist (iterations_cadence_id WITH =, daterange(start_date, due_date, '[]'::text) WITH &&) WHERE ((group_id IS NOT NULL));

ALTER TABLE ONLY sprints
    ADD CONSTRAINT iteration_start_and_due_daterange_project_id_constraint EXCLUDE USING gist (project_id WITH =, daterange(start_date, due_date, '[]'::text) WITH &&) WHERE ((project_id IS NOT NULL));

ALTER TABLE ONLY iterations_cadences
    ADD CONSTRAINT iterations_cadences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY jira_connect_installations
    ADD CONSTRAINT jira_connect_installations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY jira_connect_subscriptions
    ADD CONSTRAINT jira_connect_subscriptions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY jira_imports
    ADD CONSTRAINT jira_imports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY jira_tracker_data
    ADD CONSTRAINT jira_tracker_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY keys
    ADD CONSTRAINT keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY label_links
    ADD CONSTRAINT label_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY label_priorities
    ADD CONSTRAINT label_priorities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY labels
    ADD CONSTRAINT labels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY ldap_group_links
    ADD CONSTRAINT ldap_group_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY lfs_file_locks
    ADD CONSTRAINT lfs_file_locks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY lfs_object_states
    ADD CONSTRAINT lfs_object_states_pkey PRIMARY KEY (lfs_object_id);

ALTER TABLE ONLY lfs_objects
    ADD CONSTRAINT lfs_objects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY lfs_objects_projects
    ADD CONSTRAINT lfs_objects_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY licenses
    ADD CONSTRAINT licenses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY list_user_preferences
    ADD CONSTRAINT list_user_preferences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY lists
    ADD CONSTRAINT lists_pkey PRIMARY KEY (id);

ALTER TABLE ONLY member_tasks
    ADD CONSTRAINT member_tasks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY members
    ADD CONSTRAINT members_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_assignees
    ADD CONSTRAINT merge_request_assignees_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_blocks
    ADD CONSTRAINT merge_request_blocks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_cleanup_schedules
    ADD CONSTRAINT merge_request_cleanup_schedules_pkey PRIMARY KEY (merge_request_id);

ALTER TABLE ONLY merge_request_context_commit_diff_files
    ADD CONSTRAINT merge_request_context_commit_diff_files_pkey PRIMARY KEY (merge_request_context_commit_id, relative_order);

ALTER TABLE ONLY merge_request_context_commits
    ADD CONSTRAINT merge_request_context_commits_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_diff_commit_users
    ADD CONSTRAINT merge_request_diff_commit_users_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_diff_commits
    ADD CONSTRAINT merge_request_diff_commits_pkey PRIMARY KEY (merge_request_diff_id, relative_order);

ALTER TABLE ONLY merge_request_diff_details
    ADD CONSTRAINT merge_request_diff_details_pkey PRIMARY KEY (merge_request_diff_id);

ALTER TABLE ONLY merge_request_diff_files
    ADD CONSTRAINT merge_request_diff_files_pkey PRIMARY KEY (merge_request_diff_id, relative_order);

ALTER TABLE ONLY merge_request_diffs
    ADD CONSTRAINT merge_request_diffs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_metrics
    ADD CONSTRAINT merge_request_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_reviewers
    ADD CONSTRAINT merge_request_reviewers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_request_user_mentions
    ADD CONSTRAINT merge_request_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_requests_closing_issues
    ADD CONSTRAINT merge_requests_closing_issues_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_requests_compliance_violations
    ADD CONSTRAINT merge_requests_compliance_violations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_requests
    ADD CONSTRAINT merge_requests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY merge_trains
    ADD CONSTRAINT merge_trains_pkey PRIMARY KEY (id);

ALTER TABLE ONLY metrics_dashboard_annotations
    ADD CONSTRAINT metrics_dashboard_annotations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY metrics_users_starred_dashboards
    ADD CONSTRAINT metrics_users_starred_dashboards_pkey PRIMARY KEY (id);

ALTER TABLE ONLY milestone_releases
    ADD CONSTRAINT milestone_releases_pkey PRIMARY KEY (milestone_id, release_id);

ALTER TABLE ONLY milestones
    ADD CONSTRAINT milestones_pkey PRIMARY KEY (id);

ALTER TABLE ONLY namespace_admin_notes
    ADD CONSTRAINT namespace_admin_notes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY namespace_aggregation_schedules
    ADD CONSTRAINT namespace_aggregation_schedules_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY namespace_limits
    ADD CONSTRAINT namespace_limits_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY namespace_package_settings
    ADD CONSTRAINT namespace_package_settings_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY namespace_root_storage_statistics
    ADD CONSTRAINT namespace_root_storage_statistics_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY namespace_settings
    ADD CONSTRAINT namespace_settings_pkey PRIMARY KEY (namespace_id);

ALTER TABLE ONLY namespace_statistics
    ADD CONSTRAINT namespace_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY namespaces
    ADD CONSTRAINT namespaces_pkey PRIMARY KEY (id);

ALTER TABLE ONLY namespaces_sync_events
    ADD CONSTRAINT namespaces_sync_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY note_diff_files
    ADD CONSTRAINT note_diff_files_pkey PRIMARY KEY (id);

ALTER TABLE ONLY notes
    ADD CONSTRAINT notes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY notification_settings
    ADD CONSTRAINT notification_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY oauth_access_grants
    ADD CONSTRAINT oauth_access_grants_pkey PRIMARY KEY (id);

ALTER TABLE ONLY oauth_access_tokens
    ADD CONSTRAINT oauth_access_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY oauth_applications
    ADD CONSTRAINT oauth_applications_pkey PRIMARY KEY (id);

ALTER TABLE ONLY oauth_openid_requests
    ADD CONSTRAINT oauth_openid_requests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY onboarding_progresses
    ADD CONSTRAINT onboarding_progresses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_feature_flag_scopes
    ADD CONSTRAINT operations_feature_flag_scopes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_feature_flags_clients
    ADD CONSTRAINT operations_feature_flags_clients_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_feature_flags_issues
    ADD CONSTRAINT operations_feature_flags_issues_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_feature_flags
    ADD CONSTRAINT operations_feature_flags_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_scopes
    ADD CONSTRAINT operations_scopes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_strategies
    ADD CONSTRAINT operations_strategies_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_strategies_user_lists
    ADD CONSTRAINT operations_strategies_user_lists_pkey PRIMARY KEY (id);

ALTER TABLE ONLY operations_user_lists
    ADD CONSTRAINT operations_user_lists_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_build_infos
    ADD CONSTRAINT packages_build_infos_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_composer_cache_files
    ADD CONSTRAINT packages_composer_cache_files_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_composer_metadata
    ADD CONSTRAINT packages_composer_metadata_pkey PRIMARY KEY (package_id);

ALTER TABLE ONLY packages_conan_file_metadata
    ADD CONSTRAINT packages_conan_file_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_conan_metadata
    ADD CONSTRAINT packages_conan_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_file_metadata
    ADD CONSTRAINT packages_debian_file_metadata_pkey PRIMARY KEY (package_file_id);

ALTER TABLE ONLY packages_debian_group_architectures
    ADD CONSTRAINT packages_debian_group_architectures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_group_component_files
    ADD CONSTRAINT packages_debian_group_component_files_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_group_components
    ADD CONSTRAINT packages_debian_group_components_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_group_distribution_keys
    ADD CONSTRAINT packages_debian_group_distribution_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_group_distributions
    ADD CONSTRAINT packages_debian_group_distributions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_project_architectures
    ADD CONSTRAINT packages_debian_project_architectures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_project_component_files
    ADD CONSTRAINT packages_debian_project_component_files_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_project_components
    ADD CONSTRAINT packages_debian_project_components_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_project_distribution_keys
    ADD CONSTRAINT packages_debian_project_distribution_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_project_distributions
    ADD CONSTRAINT packages_debian_project_distributions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_debian_publications
    ADD CONSTRAINT packages_debian_publications_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_dependencies
    ADD CONSTRAINT packages_dependencies_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_dependency_links
    ADD CONSTRAINT packages_dependency_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_events
    ADD CONSTRAINT packages_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_helm_file_metadata
    ADD CONSTRAINT packages_helm_file_metadata_pkey PRIMARY KEY (package_file_id);

ALTER TABLE ONLY packages_maven_metadata
    ADD CONSTRAINT packages_maven_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_npm_metadata
    ADD CONSTRAINT packages_npm_metadata_pkey PRIMARY KEY (package_id);

ALTER TABLE ONLY packages_nuget_dependency_link_metadata
    ADD CONSTRAINT packages_nuget_dependency_link_metadata_pkey PRIMARY KEY (dependency_link_id);

ALTER TABLE ONLY packages_nuget_metadata
    ADD CONSTRAINT packages_nuget_metadata_pkey PRIMARY KEY (package_id);

ALTER TABLE ONLY packages_package_file_build_infos
    ADD CONSTRAINT packages_package_file_build_infos_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_package_files
    ADD CONSTRAINT packages_package_files_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_packages
    ADD CONSTRAINT packages_packages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY packages_pypi_metadata
    ADD CONSTRAINT packages_pypi_metadata_pkey PRIMARY KEY (package_id);

ALTER TABLE ONLY packages_rubygems_metadata
    ADD CONSTRAINT packages_rubygems_metadata_pkey PRIMARY KEY (package_id);

ALTER TABLE ONLY packages_tags
    ADD CONSTRAINT packages_tags_pkey PRIMARY KEY (id);

ALTER TABLE ONLY pages_deployment_states
    ADD CONSTRAINT pages_deployment_states_pkey PRIMARY KEY (pages_deployment_id);

ALTER TABLE ONLY pages_deployments
    ADD CONSTRAINT pages_deployments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY pages_domain_acme_orders
    ADD CONSTRAINT pages_domain_acme_orders_pkey PRIMARY KEY (id);

ALTER TABLE ONLY pages_domains
    ADD CONSTRAINT pages_domains_pkey PRIMARY KEY (id);

ALTER TABLE ONLY partitioned_foreign_keys
    ADD CONSTRAINT partitioned_foreign_keys_pkey PRIMARY KEY (id);

ALTER TABLE ONLY path_locks
    ADD CONSTRAINT path_locks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY personal_access_tokens
    ADD CONSTRAINT personal_access_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY plan_limits
    ADD CONSTRAINT plan_limits_pkey PRIMARY KEY (id);

ALTER TABLE ONLY plans
    ADD CONSTRAINT plans_pkey PRIMARY KEY (id);

ALTER TABLE ONLY pool_repositories
    ADD CONSTRAINT pool_repositories_pkey PRIMARY KEY (id);

ALTER TABLE ONLY postgres_async_indexes
    ADD CONSTRAINT postgres_async_indexes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY postgres_reindex_actions
    ADD CONSTRAINT postgres_reindex_actions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY postgres_reindex_queued_actions
    ADD CONSTRAINT postgres_reindex_queued_actions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY programming_languages
    ADD CONSTRAINT programming_languages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_access_tokens
    ADD CONSTRAINT project_access_tokens_pkey PRIMARY KEY (personal_access_token_id, project_id);

ALTER TABLE ONLY project_alerting_settings
    ADD CONSTRAINT project_alerting_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_aliases
    ADD CONSTRAINT project_aliases_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_authorizations
    ADD CONSTRAINT project_authorizations_pkey PRIMARY KEY (user_id, project_id, access_level);

ALTER TABLE ONLY project_auto_devops
    ADD CONSTRAINT project_auto_devops_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_ci_cd_settings
    ADD CONSTRAINT project_ci_cd_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_ci_feature_usages
    ADD CONSTRAINT project_ci_feature_usages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_compliance_framework_settings
    ADD CONSTRAINT project_compliance_framework_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_custom_attributes
    ADD CONSTRAINT project_custom_attributes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_daily_statistics
    ADD CONSTRAINT project_daily_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_deploy_tokens
    ADD CONSTRAINT project_deploy_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_error_tracking_settings
    ADD CONSTRAINT project_error_tracking_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_export_jobs
    ADD CONSTRAINT project_export_jobs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_feature_usages
    ADD CONSTRAINT project_feature_usages_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_features
    ADD CONSTRAINT project_features_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_group_links
    ADD CONSTRAINT project_group_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_import_data
    ADD CONSTRAINT project_import_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_incident_management_settings
    ADD CONSTRAINT project_incident_management_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_metrics_settings
    ADD CONSTRAINT project_metrics_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_mirror_data
    ADD CONSTRAINT project_mirror_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_pages_metadata
    ADD CONSTRAINT project_pages_metadata_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_repositories
    ADD CONSTRAINT project_repositories_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_repository_states
    ADD CONSTRAINT project_repository_states_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_repository_storage_moves
    ADD CONSTRAINT project_repository_storage_moves_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_security_settings
    ADD CONSTRAINT project_security_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_settings
    ADD CONSTRAINT project_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY project_statistics
    ADD CONSTRAINT project_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_topics
    ADD CONSTRAINT project_topics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY project_tracing_settings
    ADD CONSTRAINT project_tracing_settings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY projects
    ADD CONSTRAINT projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY projects_sync_events
    ADD CONSTRAINT projects_sync_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY prometheus_alert_events
    ADD CONSTRAINT prometheus_alert_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY prometheus_alerts
    ADD CONSTRAINT prometheus_alerts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY prometheus_metrics
    ADD CONSTRAINT prometheus_metrics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_branch_merge_access_levels
    ADD CONSTRAINT protected_branch_merge_access_levels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_branch_push_access_levels
    ADD CONSTRAINT protected_branch_push_access_levels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_branch_unprotect_access_levels
    ADD CONSTRAINT protected_branch_unprotect_access_levels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_branches
    ADD CONSTRAINT protected_branches_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_environment_deploy_access_levels
    ADD CONSTRAINT protected_environment_deploy_access_levels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_environments
    ADD CONSTRAINT protected_environments_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_tag_create_access_levels
    ADD CONSTRAINT protected_tag_create_access_levels_pkey PRIMARY KEY (id);

ALTER TABLE ONLY protected_tags
    ADD CONSTRAINT protected_tags_pkey PRIMARY KEY (id);

ALTER TABLE ONLY push_event_payloads
    ADD CONSTRAINT push_event_payloads_pkey PRIMARY KEY (event_id);

ALTER TABLE ONLY push_rules
    ADD CONSTRAINT push_rules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY raw_usage_data
    ADD CONSTRAINT raw_usage_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY redirect_routes
    ADD CONSTRAINT redirect_routes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY release_links
    ADD CONSTRAINT release_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY releases
    ADD CONSTRAINT releases_pkey PRIMARY KEY (id);

ALTER TABLE ONLY remote_mirrors
    ADD CONSTRAINT remote_mirrors_pkey PRIMARY KEY (id);

ALTER TABLE ONLY repository_languages
    ADD CONSTRAINT repository_languages_pkey PRIMARY KEY (project_id, programming_language_id);

ALTER TABLE ONLY required_code_owners_sections
    ADD CONSTRAINT required_code_owners_sections_pkey PRIMARY KEY (id);

ALTER TABLE ONLY requirements_management_test_reports
    ADD CONSTRAINT requirements_management_test_reports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY requirements
    ADD CONSTRAINT requirements_pkey PRIMARY KEY (id);

ALTER TABLE ONLY resource_iteration_events
    ADD CONSTRAINT resource_iteration_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY resource_label_events
    ADD CONSTRAINT resource_label_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY resource_milestone_events
    ADD CONSTRAINT resource_milestone_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY resource_state_events
    ADD CONSTRAINT resource_state_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY resource_weight_events
    ADD CONSTRAINT resource_weight_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY reviews
    ADD CONSTRAINT reviews_pkey PRIMARY KEY (id);

ALTER TABLE ONLY routes
    ADD CONSTRAINT routes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY saml_group_links
    ADD CONSTRAINT saml_group_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY saml_providers
    ADD CONSTRAINT saml_providers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY scim_identities
    ADD CONSTRAINT scim_identities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY scim_oauth_access_tokens
    ADD CONSTRAINT scim_oauth_access_tokens_pkey PRIMARY KEY (id);

ALTER TABLE ONLY security_findings
    ADD CONSTRAINT security_findings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY security_orchestration_policy_configurations
    ADD CONSTRAINT security_orchestration_policy_configurations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY security_orchestration_policy_rule_schedules
    ADD CONSTRAINT security_orchestration_policy_rule_schedules_pkey PRIMARY KEY (id);

ALTER TABLE ONLY security_scans
    ADD CONSTRAINT security_scans_pkey PRIMARY KEY (id);

ALTER TABLE ONLY self_managed_prometheus_alert_events
    ADD CONSTRAINT self_managed_prometheus_alert_events_pkey PRIMARY KEY (id);

ALTER TABLE ONLY sent_notifications
    ADD CONSTRAINT sent_notifications_pkey PRIMARY KEY (id);

ALTER TABLE ONLY sentry_issues
    ADD CONSTRAINT sentry_issues_pkey PRIMARY KEY (id);

ALTER TABLE ONLY sprints
    ADD CONSTRAINT sequence_is_unique_per_iterations_cadence_id UNIQUE (iterations_cadence_id, sequence) DEFERRABLE INITIALLY DEFERRED;

ALTER TABLE ONLY serverless_domain_cluster
    ADD CONSTRAINT serverless_domain_cluster_pkey PRIMARY KEY (uuid);

ALTER TABLE ONLY service_desk_settings
    ADD CONSTRAINT service_desk_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY shards
    ADD CONSTRAINT shards_pkey PRIMARY KEY (id);

ALTER TABLE ONLY slack_integrations
    ADD CONSTRAINT slack_integrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY smartcard_identities
    ADD CONSTRAINT smartcard_identities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY snippet_repositories
    ADD CONSTRAINT snippet_repositories_pkey PRIMARY KEY (snippet_id);

ALTER TABLE ONLY snippet_repository_storage_moves
    ADD CONSTRAINT snippet_repository_storage_moves_pkey PRIMARY KEY (id);

ALTER TABLE ONLY snippet_statistics
    ADD CONSTRAINT snippet_statistics_pkey PRIMARY KEY (snippet_id);

ALTER TABLE ONLY snippet_user_mentions
    ADD CONSTRAINT snippet_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY snippets
    ADD CONSTRAINT snippets_pkey PRIMARY KEY (id);

ALTER TABLE ONLY software_license_policies
    ADD CONSTRAINT software_license_policies_pkey PRIMARY KEY (id);

ALTER TABLE ONLY software_licenses
    ADD CONSTRAINT software_licenses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY spam_logs
    ADD CONSTRAINT spam_logs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY sprints
    ADD CONSTRAINT sprints_pkey PRIMARY KEY (id);

ALTER TABLE ONLY status_check_responses
    ADD CONSTRAINT status_check_responses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY status_page_published_incidents
    ADD CONSTRAINT status_page_published_incidents_pkey PRIMARY KEY (id);

ALTER TABLE ONLY status_page_settings
    ADD CONSTRAINT status_page_settings_pkey PRIMARY KEY (project_id);

ALTER TABLE ONLY subscriptions
    ADD CONSTRAINT subscriptions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY suggestions
    ADD CONSTRAINT suggestions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY system_note_metadata
    ADD CONSTRAINT system_note_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY taggings
    ADD CONSTRAINT taggings_pkey PRIMARY KEY (id);

ALTER TABLE ONLY tags
    ADD CONSTRAINT tags_pkey PRIMARY KEY (id);

ALTER TABLE ONLY term_agreements
    ADD CONSTRAINT term_agreements_pkey PRIMARY KEY (id);

ALTER TABLE ONLY terraform_state_versions
    ADD CONSTRAINT terraform_state_versions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY terraform_states
    ADD CONSTRAINT terraform_states_pkey PRIMARY KEY (id);

ALTER TABLE ONLY timelogs
    ADD CONSTRAINT timelogs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY todos
    ADD CONSTRAINT todos_pkey PRIMARY KEY (id);

ALTER TABLE ONLY token_with_ivs
    ADD CONSTRAINT token_with_ivs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY topics
    ADD CONSTRAINT topics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY trending_projects
    ADD CONSTRAINT trending_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY u2f_registrations
    ADD CONSTRAINT u2f_registrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY upcoming_reconciliations
    ADD CONSTRAINT upcoming_reconciliations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY upload_states
    ADD CONSTRAINT upload_states_pkey PRIMARY KEY (upload_id);

ALTER TABLE ONLY uploads
    ADD CONSTRAINT uploads_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_agent_details
    ADD CONSTRAINT user_agent_details_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_callouts
    ADD CONSTRAINT user_callouts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_canonical_emails
    ADD CONSTRAINT user_canonical_emails_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_credit_card_validations
    ADD CONSTRAINT user_credit_card_validations_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY user_custom_attributes
    ADD CONSTRAINT user_custom_attributes_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_details
    ADD CONSTRAINT user_details_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY user_follow_users
    ADD CONSTRAINT user_follow_users_pkey PRIMARY KEY (follower_id, followee_id);

ALTER TABLE ONLY user_group_callouts
    ADD CONSTRAINT user_group_callouts_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_highest_roles
    ADD CONSTRAINT user_highest_roles_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY user_interacted_projects
    ADD CONSTRAINT user_interacted_projects_pkey PRIMARY KEY (project_id, user_id);

ALTER TABLE ONLY user_permission_export_uploads
    ADD CONSTRAINT user_permission_export_uploads_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_preferences
    ADD CONSTRAINT user_preferences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY user_statuses
    ADD CONSTRAINT user_statuses_pkey PRIMARY KEY (user_id);

ALTER TABLE ONLY user_synced_attributes_metadata
    ADD CONSTRAINT user_synced_attributes_metadata_pkey PRIMARY KEY (id);

ALTER TABLE ONLY users_ops_dashboard_projects
    ADD CONSTRAINT users_ops_dashboard_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY users
    ADD CONSTRAINT users_pkey PRIMARY KEY (id);

ALTER TABLE ONLY users_security_dashboard_projects
    ADD CONSTRAINT users_security_dashboard_projects_pkey PRIMARY KEY (project_id, user_id);

ALTER TABLE ONLY users_star_projects
    ADD CONSTRAINT users_star_projects_pkey PRIMARY KEY (id);

ALTER TABLE ONLY users_statistics
    ADD CONSTRAINT users_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerabilities
    ADD CONSTRAINT vulnerabilities_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_exports
    ADD CONSTRAINT vulnerability_exports_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_external_issue_links
    ADD CONSTRAINT vulnerability_external_issue_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_feedback
    ADD CONSTRAINT vulnerability_feedback_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_assets
    ADD CONSTRAINT vulnerability_finding_evidence_assets_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_headers
    ADD CONSTRAINT vulnerability_finding_evidence_headers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_requests
    ADD CONSTRAINT vulnerability_finding_evidence_requests_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_responses
    ADD CONSTRAINT vulnerability_finding_evidence_responses_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_sources
    ADD CONSTRAINT vulnerability_finding_evidence_sources_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidence_supporting_messages
    ADD CONSTRAINT vulnerability_finding_evidence_supporting_messages_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_evidences
    ADD CONSTRAINT vulnerability_finding_evidences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_links
    ADD CONSTRAINT vulnerability_finding_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_finding_signatures
    ADD CONSTRAINT vulnerability_finding_signatures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_findings_remediations
    ADD CONSTRAINT vulnerability_findings_remediations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_flags
    ADD CONSTRAINT vulnerability_flags_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_historical_statistics
    ADD CONSTRAINT vulnerability_historical_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_identifiers
    ADD CONSTRAINT vulnerability_identifiers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_issue_links
    ADD CONSTRAINT vulnerability_issue_links_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_occurrence_identifiers
    ADD CONSTRAINT vulnerability_occurrence_identifiers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_occurrence_pipelines
    ADD CONSTRAINT vulnerability_occurrence_pipelines_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_occurrences
    ADD CONSTRAINT vulnerability_occurrences_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_reads
    ADD CONSTRAINT vulnerability_reads_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_remediations
    ADD CONSTRAINT vulnerability_remediations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_scanners
    ADD CONSTRAINT vulnerability_scanners_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_statistics
    ADD CONSTRAINT vulnerability_statistics_pkey PRIMARY KEY (id);

ALTER TABLE ONLY vulnerability_user_mentions
    ADD CONSTRAINT vulnerability_user_mentions_pkey PRIMARY KEY (id);

ALTER TABLE ONLY web_hooks
    ADD CONSTRAINT web_hooks_pkey PRIMARY KEY (id);

ALTER TABLE ONLY webauthn_registrations
    ADD CONSTRAINT webauthn_registrations_pkey PRIMARY KEY (id);

ALTER TABLE ONLY wiki_page_meta
    ADD CONSTRAINT wiki_page_meta_pkey PRIMARY KEY (id);

ALTER TABLE ONLY wiki_page_slugs
    ADD CONSTRAINT wiki_page_slugs_pkey PRIMARY KEY (id);

ALTER TABLE ONLY work_item_types
    ADD CONSTRAINT work_item_types_pkey PRIMARY KEY (id);

ALTER TABLE ONLY x509_certificates
    ADD CONSTRAINT x509_certificates_pkey PRIMARY KEY (id);

ALTER TABLE ONLY x509_commit_signatures
    ADD CONSTRAINT x509_commit_signatures_pkey PRIMARY KEY (id);

ALTER TABLE ONLY x509_issuers
    ADD CONSTRAINT x509_issuers_pkey PRIMARY KEY (id);

ALTER TABLE ONLY zentao_tracker_data
    ADD CONSTRAINT zentao_tracker_data_pkey PRIMARY KEY (id);

ALTER TABLE ONLY zoom_meetings
    ADD CONSTRAINT zoom_meetings_pkey PRIMARY KEY (id);

CREATE INDEX analytics_index_audit_events_part_on_created_at_and_author_id ON ONLY audit_events USING btree (created_at, author_id);

CREATE INDEX audit_events_000000_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_000000 USING btree (created_at, author_id);

CREATE INDEX idx_audit_events_part_on_entity_id_desc_author_id_created_at ON ONLY audit_events USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_000000_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_000000 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202212_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202212 USING btree (created_at, author_id);

CREATE INDEX audit_events_202212_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202212 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202301_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202301 USING btree (created_at, author_id);

CREATE INDEX audit_events_202301_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202301 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202302_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202302 USING btree (created_at, author_id);

CREATE INDEX audit_events_202302_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202302 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202303_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202303 USING btree (created_at, author_id);

CREATE INDEX audit_events_202303_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202303 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202304_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202304 USING btree (created_at, author_id);

CREATE INDEX audit_events_202304_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202304 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202305_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202305 USING btree (created_at, author_id);

CREATE INDEX audit_events_202305_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202305 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202306_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202306 USING btree (created_at, author_id);

CREATE INDEX audit_events_202306_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202306 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202307_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202307 USING btree (created_at, author_id);

CREATE INDEX audit_events_202307_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202307 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202308_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202308 USING btree (created_at, author_id);

CREATE INDEX audit_events_202308_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202308 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX audit_events_202309_created_at_author_id_idx ON gitlab_partitions_dynamic.audit_events_202309 USING btree (created_at, author_id);

CREATE INDEX audit_events_202309_entity_id_entity_type_id_author_id_crea_idx ON gitlab_partitions_dynamic.audit_events_202309 USING btree (entity_id, entity_type, id DESC, author_id, created_at);

CREATE INDEX i_batched_background_migration_job_transition_logs_on_job_id ON ONLY batched_background_migration_job_transition_logs USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration__idx ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_000000 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx1 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202303 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx2 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202304 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx3 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202305 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx4 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202306 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx5 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202307 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx6 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202308 USING btree (batched_background_migration_job_id);

CREATE INDEX batched_background_migration__batched_background_migration_idx7 ON gitlab_partitions_dynamic.batched_background_migration_job_transition_logs_202309 USING btree (batched_background_migration_job_id);

CREATE INDEX incident_management_pending_alert_escalations_20221_rule_id_idx ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202211 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2022_alert_id_idx ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202211 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_2022_rule_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202212 USING btree (rule_id);

CREATE INDEX index_incident_management_pending_alert_escalations_on_rule_id ON ONLY incident_management_pending_alert_escalations USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_20230_rule_id_idx ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202301 USING btree (rule_id);

CREATE INDEX index_incident_management_pending_alert_escalations_on_alert_id ON ONLY incident_management_pending_alert_escalations USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_alert_id_idx ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202301 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202302 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx2 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202303 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx3 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202304 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx4 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202305 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx5 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202306 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx6 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202307 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx7 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202308 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_2023_rule_id_idx8 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202309 USING btree (rule_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202212 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx2 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202302 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx3 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202303 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx4 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202304 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx5 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202305 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx6 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202306 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx7 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202307 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx8 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202308 USING btree (alert_id);

CREATE INDEX incident_management_pending_alert_escalations_202_alert_id_idx9 ON gitlab_partitions_dynamic.incident_management_pending_alert_escalations_202309 USING btree (alert_id);

CREATE INDEX incident_management_pending_issue_escalations_20221_rule_id_idx ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202211 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2022_issue_id_idx ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202211 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_2022_rule_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202212 USING btree (rule_id);

CREATE INDEX index_incident_management_pending_issue_escalations_on_rule_id ON ONLY incident_management_pending_issue_escalations USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_20230_rule_id_idx ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202301 USING btree (rule_id);

CREATE INDEX index_incident_management_pending_issue_escalations_on_issue_id ON ONLY incident_management_pending_issue_escalations USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_issue_id_idx ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202301 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202302 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx2 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202303 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx3 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202304 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx4 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202305 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx5 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202306 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx6 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202307 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx7 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202308 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_2023_rule_id_idx8 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202309 USING btree (rule_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx1 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202212 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx2 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202302 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx3 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202303 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx4 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202304 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx5 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202305 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx6 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202306 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx7 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202307 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx8 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202308 USING btree (issue_id);

CREATE INDEX incident_management_pending_issue_escalations_202_issue_id_idx9 ON gitlab_partitions_dynamic.incident_management_pending_issue_escalations_202309 USING btree (issue_id);

CREATE INDEX index_web_hook_logs_part_on_web_hook_id ON ONLY web_hook_logs USING btree (web_hook_id);

CREATE INDEX index_1554dc6f11 ON gitlab_partitions_dynamic.web_hook_logs_202301 USING btree (web_hook_id);

CREATE INDEX index_719c6187cb ON gitlab_partitions_dynamic.web_hook_logs_202212 USING btree (web_hook_id);

CREATE INDEX index_80bf138a51 ON gitlab_partitions_dynamic.web_hook_logs_000000 USING btree (web_hook_id);

CREATE INDEX index_web_hook_logs_part_on_created_at_and_web_hook_id ON ONLY web_hook_logs USING btree (created_at, web_hook_id);

CREATE INDEX index_962081978f ON gitlab_partitions_dynamic.web_hook_logs_202212 USING btree (created_at, web_hook_id);

CREATE INDEX index_eecfac613f ON gitlab_partitions_dynamic.web_hook_logs_000000 USING btree (created_at, web_hook_id);

CREATE INDEX index_fdb8d5eeea ON gitlab_partitions_dynamic.web_hook_logs_202301 USING btree (created_at, web_hook_id);

CREATE UNIQUE INDEX index_verification_codes_on_phone_and_visitor_id_code ON ONLY verification_codes USING btree (visitor_id_code, phone, created_at);

COMMENT ON INDEX index_verification_codes_on_phone_and_visitor_id_code IS 'JiHu-specific index';

CREATE UNIQUE INDEX verification_codes_000000_visitor_id_code_phone_created_at_idx ON gitlab_partitions_dynamic.verification_codes_000000 USING btree (visitor_id_code, phone, created_at);

CREATE UNIQUE INDEX verification_codes_202302_visitor_id_code_phone_created_at_idx ON gitlab_partitions_dynamic.verification_codes_202302 USING btree (visitor_id_code, phone, created_at);

CREATE UNIQUE INDEX verification_codes_202303_visitor_id_code_phone_created_at_idx ON gitlab_partitions_dynamic.verification_codes_202303 USING btree (visitor_id_code, phone, created_at);

CREATE UNIQUE INDEX verification_codes_202304_visitor_id_code_phone_created_at_idx ON gitlab_partitions_dynamic.verification_codes_202304 USING btree (visitor_id_code, phone, created_at);

CREATE UNIQUE INDEX verification_codes_202305_visitor_i