/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/

import { Disposable } from '../../../../../base/common/lifecycle.js';
import type { OperatingSystem } from '../../../../../base/common/platform.js';
import { regExpLeadsToEndlessLoop } from '../../../../../base/common/strings.js';
import { IConfigurationService } from '../../../../../platform/configuration/common/configuration.js';
import { TerminalChatAgentToolsSettingId } from '../common/terminalChatAgentToolsConfiguration.js';
import { isPowerShell } from './runInTerminalHelpers.js';

interface IAutoApproveRule {
	regex: RegExp;
	regexCaseInsensitive: RegExp;
	sourceText: string;
}

export type ICommandApprovalResult = 'approved' | 'denied' | 'noMatch';

const neverMatchRegex = /(?!.*)/;

export class CommandLineAutoApprover extends Disposable {
	private _denyListRules: IAutoApproveRule[] = [];
	private _allowListRules: IAutoApproveRule[] = [];
	private _allowListCommandLineRules: IAutoApproveRule[] = [];
	private _denyListCommandLineRules: IAutoApproveRule[] = [];

	constructor(
		@IConfigurationService private readonly _configurationService: IConfigurationService,
	) {
		super();
		this.updateConfiguration();
		this._register(this._configurationService.onDidChangeConfiguration(e => {
			if (
				e.affectsConfiguration(TerminalChatAgentToolsSettingId.AutoApprove) ||
				e.affectsConfiguration(TerminalChatAgentToolsSettingId.DeprecatedAutoApproveCompatible)
			) {
				this.updateConfiguration();
			}
		}));
	}

	updateConfiguration() {
		let configValue = this._configurationService.getValue(TerminalChatAgentToolsSettingId.AutoApprove);
		const deprecatedValue = this._configurationService.getValue(TerminalChatAgentToolsSettingId.DeprecatedAutoApproveCompatible);
		if (deprecatedValue && typeof deprecatedValue === 'object' && configValue && typeof configValue === 'object') {
			configValue = {
				...configValue,
				...deprecatedValue
			};
		}

		const {
			denyListRules,
			allowListRules,
			allowListCommandLineRules,
			denyListCommandLineRules
		} = this._mapAutoApproveConfigToRules(configValue);
		this._allowListRules = allowListRules;
		this._denyListRules = denyListRules;
		this._allowListCommandLineRules = allowListCommandLineRules;
		this._denyListCommandLineRules = denyListCommandLineRules;
	}

	isCommandAutoApproved(command: string, shell: string, os: OperatingSystem): { result: ICommandApprovalResult; reason: string } {
		// Check the deny list to see if this command requires explicit approval
		for (const rule of this._denyListRules) {
			if (this._commandMatchesRule(rule, command, shell, os)) {
				return { result: 'denied', reason: `Command '${command}' is denied by deny list rule: ${rule.sourceText}` };
			}
		}

		// Check the allow list to see if the command is allowed to run without explicit approval
		for (const rule of this._allowListRules) {
			if (this._commandMatchesRule(rule, command, shell, os)) {
				return { result: 'approved', reason: `Command '${command}' is approved by allow list rule: ${rule.sourceText}` };
			}
		}

		// TODO: LLM-based auto-approval https://github.com/microsoft/vscode/issues/253267

		// Fallback is always to require approval
		return { result: 'noMatch', reason: `Command '${command}' has no matching auto approve entries` };
	}

	isCommandLineAutoApproved(commandLine: string): { result: ICommandApprovalResult; reason: string } {
		// Check the deny list first to see if this command line requires explicit approval
		for (const rule of this._denyListCommandLineRules) {
			if (rule.regex.test(commandLine)) {
				return { result: 'denied', reason: `Command line '${commandLine}' is denied by deny list rule: ${rule.sourceText}` };
			}
		}

		// Check if the full command line matches any of the allow list command line regexes
		for (const rule of this._allowListCommandLineRules) {
			if (rule.regex.test(commandLine)) {
				return { result: 'approved', reason: `Command line '${commandLine}' is approved by allow list rule: ${rule.sourceText}` };
			}
		}
		return { result: 'noMatch', reason: `Command line '${commandLine}' has no matching auto approve entries` };
	}

	private _removeEnvAssignments(command: string, shell: string, os: OperatingSystem): string {
		const trimmedCommand = command.trimStart();

		// PowerShell environment variable syntax is `$env:VAR='value';` and treated as a different
		// command
		if (isPowerShell(shell, os)) {
			return trimmedCommand;
		}

		// For bash/sh/bourne shell and unknown shells (fallback to bourne shell syntax)
		// Handle environment variable assignments like: VAR=value VAR2=value command
		// This regex matches one or more environment variable assignments at the start
		const envVarPattern = /^(\s*[A-Za-z_][A-Za-z0-9_]*=(?:[^\s'"]|'[^']*'|"[^"]*")*\s+)+/;
		const match = trimmedCommand.match(envVarPattern);

		if (match) {
			const actualCommand = trimmedCommand.slice(match[0].length).trimStart();
			return actualCommand || trimmedCommand; // Fallback to original if nothing left
		}

		return trimmedCommand;
	}

	private _commandMatchesRule(rule: IAutoApproveRule, command: string, shell: string, os: OperatingSystem): boolean {
		const actualCommand = this._removeEnvAssignments(command, shell, os);
		const isPwsh = isPowerShell(shell, os);

		// PowerShell is case insensitive regardless of platform
		if ((isPwsh ? rule.regexCaseInsensitive : rule.regex).test(actualCommand)) {
			return true;
		} else if (isPwsh && actualCommand.startsWith('(')) {
			// Allow ignoring of the leading ( for PowerShell commands as it's a command pattern to
			// operate on the output of a command. For example `(Get-Content README.md) ...`
			if (rule.regexCaseInsensitive.test(actualCommand.slice(1))) {
				return true;
			}
		}
		return false;
	}

	private _mapAutoApproveConfigToRules(config: unknown): {
		denyListRules: IAutoApproveRule[];
		allowListRules: IAutoApproveRule[];
		allowListCommandLineRules: IAutoApproveRule[];
		denyListCommandLineRules: IAutoApproveRule[];
	} {
		if (!config || typeof config !== 'object') {
			return {
				denyListRules: [],
				allowListRules: [],
				allowListCommandLineRules: [],
				denyListCommandLineRules: []
			};
		}

		const denyListRules: IAutoApproveRule[] = [];
		const allowListRules: IAutoApproveRule[] = [];
		const allowListCommandLineRules: IAutoApproveRule[] = [];
		const denyListCommandLineRules: IAutoApproveRule[] = [];

		Object.entries(config).forEach(([key, value]) => {
			if (typeof value === 'boolean') {
				const { regex, regexCaseInsensitive } = this._convertAutoApproveEntryToRegex(key);
				// IMPORTANT: Only true and false are used, null entries need to be ignored
				if (value === true) {
					allowListRules.push({ regex, regexCaseInsensitive, sourceText: key });
				} else if (value === false) {
					denyListRules.push({ regex, regexCaseInsensitive, sourceText: key });
				}
			} else if (typeof value === 'object' && value !== null) {
				// Handle object format like { approve: true/false, matchCommandLine: true/false }
				const objectValue = value as { approve?: boolean; matchCommandLine?: boolean };
				if (typeof objectValue.approve === 'boolean') {
					const { regex, regexCaseInsensitive } = this._convertAutoApproveEntryToRegex(key);
					if (objectValue.approve === true) {
						if (objectValue.matchCommandLine === true) {
							allowListCommandLineRules.push({ regex, regexCaseInsensitive, sourceText: key });
						} else {
							allowListRules.push({ regex, regexCaseInsensitive, sourceText: key });
						}
					} else if (objectValue.approve === false) {
						if (objectValue.matchCommandLine === true) {
							denyListCommandLineRules.push({ regex, regexCaseInsensitive, sourceText: key });
						} else {
							denyListRules.push({ regex, regexCaseInsensitive, sourceText: key });
						}
					}
				}
			}
		});

		return {
			denyListRules,
			allowListRules,
			allowListCommandLineRules,
			denyListCommandLineRules
		};
	}

	private _convertAutoApproveEntryToRegex(value: string): { regex: RegExp; regexCaseInsensitive: RegExp } {
		const regex = this._doConvertAutoApproveEntryToRegex(value);
		if (regex.flags.includes('i')) {
			return { regex, regexCaseInsensitive: regex };
		}
		return { regex, regexCaseInsensitive: new RegExp(regex.source, regex.flags + 'i') };
	}

	private _doConvertAutoApproveEntryToRegex(value: string): RegExp {
		// If it's wrapped in `/`, it's in regex format and should be converted directly
		// Support all standard JavaScript regex flags: d, g, i, m, s, u, v, y
		const regexMatch = value.match(/^\/(?<pattern>.+)\/(?<flags>[dgimsuvy]*)$/);
		const regexPattern = regexMatch?.groups?.pattern;
		if (regexPattern) {
			let flags = regexMatch.groups?.flags;
			// Remove global flag as it changes how the regex state works which we need to handle
			// internally
			if (flags) {
				flags = flags.replaceAll('g', '');
			}

			// Allow .* as users expect this would match everything
			if (regexPattern === '.*') {
				return new RegExp(regexPattern);

			}

			try {
				const regex = new RegExp(regexPattern, flags || undefined);
				if (regExpLeadsToEndlessLoop(regex)) {
					return neverMatchRegex;
				}

				return regex;
			} catch (error) {
				return neverMatchRegex;
			}
		}

		// The empty string should be ignored, rather than approve everything
		if (value === '') {
			return neverMatchRegex;
		}

		// Escape regex special characters
		const sanitizedValue = value.replace(/[\\^$.*+?()[\]{}|]/g, '\\$&');

		// Regular strings should match the start of the command line and be a word boundary
		return new RegExp(`^${sanitizedValue}\\b`);
	}
}
