/*---------------------------------------------------------------------------------------------
 *  Copyright (c) Microsoft Corporation. All rights reserved.
 *  Licensed under the MIT License. See License.txt in the project root for license information.
 *--------------------------------------------------------------------------------------------*/

import type { IConfigurationService } from '../../../../../../platform/configuration/common/configuration.js';
import { ContextKeyExpr } from '../../../../../../platform/contextkey/common/contextkey.js';
import { PromptsType } from '../promptTypes.js';
import { INSTRUCTIONS_DEFAULT_SOURCE_FOLDER, PROMPT_DEFAULT_SOURCE_FOLDER, getPromptFileDefaultLocation } from './promptFileLocations.js';

/**
 * Configuration helper for the `reusable prompts` feature.
 * @see {@link PromptsConfig.KEY}, {@link PromptsConfig.PROMPT_LOCATIONS_KEY}, {@link PromptsConfig.INSTRUCTIONS_LOCATION_KEY} or {@link PromptsConfig.MODE_LOCATION_KEY}.
 *
 * ### Functions
 *
 * - {@link enabled} allows to check if the feature is enabled
 * - {@link getLocationsValue} allows to current read configuration value
 * - {@link promptSourceFolders} gets list of source folders for prompt files
 *
 * ### File Paths Resolution
 *
 * We resolve only `*.prompt.md` files inside the resulting source folders. Relative paths are resolved
 * relative to:
 *
 * - the current workspace `root`, if applicable, in other words one of the workspace folders
 *   can be used as a prompt files source folder
 * - root of each top-level folder in the workspace (if there are multiple workspace folders)
 * - current root folder (if a single folder is open)
 */
export namespace PromptsConfig {
	/**
	 * Configuration key for the `reusable prompts` feature
	 * (also known as `prompt files`, `prompt instructions`, etc.).
	 */
	export const KEY = 'chat.promptFiles';

	/**
	 * Configuration key for the locations of reusable prompt files.
	 */
	export const PROMPT_LOCATIONS_KEY = 'chat.promptFilesLocations';

	/**
	 * Configuration key for the locations of instructions files.
	 */
	export const INSTRUCTIONS_LOCATION_KEY = 'chat.instructionsFilesLocations';
	/**
	 * Configuration key for the locations of mode files.
	 */
	export const MODE_LOCATION_KEY = 'chat.modeFilesLocations';

	/**
	 * Configuration key for use of the copilot instructions file.
	 */
	export const USE_COPILOT_INSTRUCTION_FILES = 'github.copilot.chat.codeGeneration.useInstructionFiles';

	/**
	 * Configuration key for the copilot instruction setting.
	 */
	export const COPILOT_INSTRUCTIONS = 'github.copilot.chat.codeGeneration.instructions';

	/**
	 * Checks if the feature is enabled.
	 * @see {@link PromptsConfig.KEY}.
	 */
	export function enabled(configService: IConfigurationService): boolean {
		const enabledValue = configService.getValue(PromptsConfig.KEY);

		return asBoolean(enabledValue) ?? false;
	}

	/**
	 * Context key expression for the `reusable prompts` feature `enabled` status.
	 */
	export const enabledCtx = ContextKeyExpr.equals(`config.${PromptsConfig.KEY}`, true);

	/**
	 * Get value of the `reusable prompt locations` configuration setting.
	 * @see {@link PROMPT_LOCATIONS_CONFIG_KEY}, {@link INSTRUCTIONS_LOCATIONS_CONFIG_KEY}, {@link MODE_LOCATIONS_CONFIG_KEY}.
	 */
	export function getLocationsValue(configService: IConfigurationService, type: PromptsType): Record<string, boolean> | undefined {
		const key = getPromptFileLocationsConfigKey(type);
		const configValue = configService.getValue(key);

		if (configValue === undefined || configValue === null || Array.isArray(configValue)) {
			return undefined;
		}

		// note! this would be also true for `null` and `array`,
		// 		 but those cases are already handled above
		if (typeof configValue === 'object') {
			const paths: Record<string, boolean> = {};

			for (const [path, value] of Object.entries(configValue)) {
				const cleanPath = path.trim();
				const booleanValue = asBoolean(value);

				// if value can be mapped to a boolean, and the clean
				// path is not empty, add it to the map
				if ((booleanValue !== undefined) && cleanPath) {
					paths[cleanPath] = booleanValue;
				}
			}

			return paths;
		}

		return undefined;
	}

	/**
	 * Gets list of source folders for prompt files.
	 * Defaults to {@link PROMPT_DEFAULT_SOURCE_FOLDER}, {@link INSTRUCTIONS_DEFAULT_SOURCE_FOLDER} or {@link MODE_DEFAULT_SOURCE_FOLDER}.
	 */
	export function promptSourceFolders(configService: IConfigurationService, type: PromptsType): string[] {
		const value = getLocationsValue(configService, type);
		const defaultSourceFolder = getPromptFileDefaultLocation(type);

		// note! the `value &&` part handles the `undefined`, `null`, and `false` cases
		if (value && (typeof value === 'object')) {
			const paths: string[] = [];

			// if the default source folder is not explicitly disabled, add it
			if (value[defaultSourceFolder] !== false) {
				paths.push(defaultSourceFolder);
			}

			// copy all the enabled paths to the result list
			for (const [path, enabledValue] of Object.entries(value)) {
				// we already added the default source folder, so skip it
				if ((enabledValue === false) || (path === defaultSourceFolder)) {
					continue;
				}

				paths.push(path);
			}

			return paths;
		}

		// `undefined`, `null`, and `false` cases
		return [];
	}

}

export function getPromptFileLocationsConfigKey(type: PromptsType): string {
	switch (type) {
		case PromptsType.instructions:
			return PromptsConfig.INSTRUCTIONS_LOCATION_KEY;
		case PromptsType.prompt:
			return PromptsConfig.PROMPT_LOCATIONS_KEY;
		case PromptsType.mode:
			return PromptsConfig.MODE_LOCATION_KEY;
		default:
			throw new Error('Unknown prompt type');
	}
}

/**
 * Helper to parse an input value of `any` type into a boolean.
 *
 * @param value - input value to parse
 * @returns `true` if the value is the boolean `true` value or a string that can
 * 			be clearly mapped to a boolean (e.g., `"true"`, `"TRUE"`, `"FaLSe"`, etc.),
 * 			`undefined` for rest of the values
 */
export function asBoolean(value: unknown): boolean | undefined {
	if (typeof value === 'boolean') {
		return value;
	}

	if (typeof value === 'string') {
		const cleanValue = value.trim().toLowerCase();
		if (cleanValue === 'true') {
			return true;
		}

		if (cleanValue === 'false') {
			return false;
		}

		return undefined;
	}

	return undefined;
}
