# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

from configuration import Configuration
from generators.filegenerator import FileGenerator

class MainPageFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = 'qml/pages/MainPage.qml'

    def generate(self) -> str:
        content =\
"""import QtQuick 2.6
import Sailfish.Silica 1.0
import ru.auroraos.WebView 1.0
import Sailfish.WebEngine 1.0

Page {
    id: page

    property bool viewSuspended: false
    allowedOrientations: Orientation.All

    function suspend() {
        if (!page.viewSuspended) {
            page.viewSuspended = true;
            webView.stopRender();
        }
    }

    function resume() {
        if (viewSuspended) {
            page.viewSuspended = false;
            webView.startRender();
        } else {
            backgroundTimer.stop();
        }
    }

    backNavigation: false

    WebView {
        id: webView
        objectName: "webView"
"""
        if self.config.user_agent:
            content +=\
fr"""
        userAgent: "{self.config.user_agent}"
"""
        content += \
"""
        url: targetUrl
        anchors {
            top: parent.top
            left: parent.left
            right: parent.right
            bottom: pageFooter.top
        }

        LoadRequestExtension{
            enabled: true

            function beforeUrlLoad(url, userGesture, isRedirect){
"""
        if self.config.frame_urls:
            content += \
f"""
                var iframePattern = new RegExp("{'|'.join(self.config.frame_urls)}");
                if (iframePattern.test(url["url"])) {{
                    return true;
                }}
"""
        if self.config.external_urls:
            content += \
f"""
                var externalPattern = new RegExp("{'|'.join(self.config.external_urls)}");
                if (externalPattern.test(url["url"])) {{
                    Qt.openUrlExternally(url["url"]);
                    return false;
                }}
"""
        if self.config.push_id_injector:
            for replacement in self.config.push_id_injector:
                content += \
fr"""
                if (new RegExp("{replacement['trigger_on']}").test(url["url"])) {{
                    webView.url = "{replacement['replace_with']}".arg(AppCtrl.registrationId);
                    return true;
                }}
"""
        content +=\
"""
                return true;
            }
        }

        TouchInput {
            enabled: true
        }

        KeyboardInput {
            enabled: true
        }
"""
        if self.config.push_app_id:
            content +=\
"""
        property var data
        property bool callHandler: false

        onDataChanged: {
            var url = data["url"];
            if (url) {
                callHandler = true;
                webView.url = url;
            }
        }
"""
            if self.config.push_handler_name:
                content +=\
f"""
        onLoadFinished: {{
            callHandler = false;
            var dataArray = [];
            for (var prop in data) {{
                if (prop === "url") {{
                    continue;
                }}
                dataArray.push("'%1':'%2'".arg(prop).arg(data[prop]));
            }}
            runJavaScript("{self.config.push_handler_name}({{%1}})".arg(dataArray.join(",")));
        }}
"""
        content +=\
"""    }

    Item {
        id: pageFooter

        objectName: "pageFooter"
        anchors {
            left: parent.left
            right: parent.right
            bottom: parent.bottom
        }
        height: Math.max(aboutButton.height, urlField.height) + Theme.paddingMedium * 2

        IconButton {
            id: aboutButton

            objectName: "aboutButton"
            anchors {
                left: parent.left
                leftMargin: Theme.paddingMedium
                verticalCenter: parent.verticalCenter
            }
            icon {
                source: "image://theme/icon-m-about"
                sourceSize {
                    width: Theme.iconSizeMedium
                    height: Theme.iconSizeMedium
                }
            }

            onClicked: pageStack.push(Qt.resolvedUrl("AboutPage.qml"))
        }

        TextField {
            id: urlField

            objectName: "urlField"
            anchors {
                left: aboutButton.right
                right: parent.right
                verticalCenter: parent.verticalCenter
            }
            inputMethodHints: Qt.ImhNoPredictiveText | Qt.ImhUrlCharactersOnly
            focusOutBehavior: FocusBehavior.ClearPageFocus
            labelVisible: false
            placeholderText: qsTr("URL")
            textLeftPadding: 0
            textLeftMargin: Theme.paddingMedium
            font {
                pixelSize: Theme.fontSizeLarge
                family: Theme.fontFamilyHeading
            }
            EnterKey.iconSource: "image://theme/icon-m-search"

            EnterKey.onClicked: {
                webView.url = text
                webView.focus = true
            }
            Component.onCompleted: urlField.text = targetUrl
        }
    }
"""
        if not self.config.background_activity:
            content +=\
"""
    Connections {
        target: Qt.application
        onStateChanged: {
            if (Qt.application.state === Qt.ApplicationActive) {
                resume();
            } else {
                backgroundTimer.restart();
            }
        }
    }
    Timer {
        id: backgroundTimer

        interval: 60 * 1000
        running: false
        repeat: false
        onTriggered: {
            suspend()
        }
    }
"""
        if self.config.push_handler_name:
            content +=\
"""
    Connections {
        target: AppCtrl

        onDataReceived: webView.data = data
    }
"""
        if self.config.web_permissions:
            content +=\
f"""
    Component.onCompleted: {{
        {self._permission_list(self.config.web_permissions)}
    }}
"""
        content += "}\n"
        return content

    def _permission_list(self, list):
        policies = []
        for permission in self.config.web_permissions:
            if permission.lower() == 'geolocation':
                policies.append('WebEngineSettings.geolocationRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'microphone':
                policies.append('WebEngineSettings.microphoneRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'camera':
                policies.append('WebEngineSettings.cameraRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'cookie':
                policies.append('WebEngineSettings.cookieBehavior = WebEngineSettings.AcceptAll;')
            elif permission.lower() == 'popup':
                policies.append('WebEngineSettings.popupEnabled = true;')
        return '\n        '.join(policies)
