# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

from configuration import Configuration
from generators.filegenerator import FileGenerator

class ProFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = f"{self.config.full_name}.pro"

    def generate(self) -> str:
        pcg_config = []
        headers = []
        configs = ['c++17', 'auroraapp', 'auroraapp_i18n']
        defines = [fr"TARGET_URL=\\\"{self.config.url}\\\""]
        distfiles = [
            f"rpm/{self.config.full_name}.spec",
            f"{self.config.full_name}.dekstop",
            f"qml/{self.config.full_name}.qml",
        ]

        if self.config.webview_version == "Sailfish.WebView":
            headers.append('src/connectionchecker.h')
            distfiles.append('qml/components/ConnectionHeader.qml')
            distfiles.append('qml/components/ConnmanManager.qml')
            distfiles.append('qml/pages/WebViewPage.qml')
        elif self.config.webview_version == "ru.auroraos.WebView":
            pcg_config.append('aurorawebview')
            distfiles.append('qml/pages/MainPage.qml',)
            distfiles.append('qml/pages/AboutPage.qml',)
        
        sources = [f"src/{self.config.application_name.lower()}.cpp"]
        if self.config.push_app_id:
            configs.insert(0, 'link_pkgconfig')
            pcg_config.append('pushclient')
            pcg_config.append('nemonotifications-qt5')
            defines.extend([
                fr"DBUS_SERVICE=\\\"{self.config.full_name}\\\"",
                fr"DBUS_PATH=\\\"/{self.config.full_name.replace('.', '/')}\\\"",
                fr"DBUS_INTERFACE=\\\"{self.config.full_name}\\\"",
                fr"APP_ID_FILE_PATH=\\\"$$APP_ID_PATH/$$APP_ID_FILE\\\"",
            ])
            for classname in ['applicationservice', 'applicationcontroller']:
                    sources.append(f"src/{classname}.cpp")
                    headers.append(f"src/{classname}.h")
        if not self.config.background_activity:
            distfiles.append('qml/cover/DefaultCoverPage.qml')
        content =\
fr"""TARGET = {self.config.full_name}

CONFIG += \
    {self._multiline(configs)}
"""
        if self.config.webview_version == "ru.auroraos.WebView":
            content +=\
fr"""
QMAKE_LFLAGS += -Wl,--no-as-needed
QMAKE_LFLAGS += '-Wl,-rpath,\'/usr/\$$LIB/cef\''
"""
        if len(pcg_config)>0:
            content +=\
fr"""
PKGCONFIG += \
    {self._multiline(pcg_config)}
"""
        content +=\
fr"""
DEFINES += \
    {self._multiline(defines)}

"""
        if self.config.push_app_id:
            content +=\
fr"""QT += dbus

APP_ID = {self.config.push_app_id}
APP_ID_PATH = /usr/share/$$TARGET
APP_ID_FILE = applicationid

applicationid_install.extra = echo $$APP_ID > $$OUT_PWD/$$APP_ID_FILE
applicationid_install.files = $$OUT_PWD/$$APP_ID_FILE
applicationid_install.path = $$APP_ID_PATH
applicationid_install.CONFIG = no_check_exist

QMAKE_CLEAN += \
    $${{applicationid_install.files}}

QMAKE_EXTRA_TARGETS += \
    applicationid_install

INSTALLS += \
    applicationid_install

"""
        content +=\
fr"""SOURCES += \
    {self._multiline(sources)}
"""
        if len(headers)>0:
            content +=\
fr"""
HEADERS += \
    {self._multiline(headers)}
"""
        content +=\
fr"""
DISTFILES += \
    {self._multiline(distfiles)}
    
AURORAAPP_ICONS = 86x86 108x108 128x128 172x172

"""
        return content

    def _multiline(self, list):
        return ' \\\n    '.join(list)
