# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

import os
from configuration import Configuration
from generators.filegenerator import FileGenerator

class ApplicationServiceFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = 'src/applicationservice'

    def generate_header(self) -> str:
        content =\
"""\
#ifndef APPLICATIONSERVICE_H
#define APPLICATIONSERVICE_H

#include <QtDBus/QDBusAbstractAdaptor>

class ApplicationService : public QDBusAbstractAdaptor
{
    Q_OBJECT

    Q_CLASSINFO("D-Bus Interface", DBUS_INTERFACE)

public:
    explicit ApplicationService(QObject *parent = nullptr);
    ~ApplicationService() override;

    static bool isRegistered();

    static QString notifyDBusService();
    static QString notifyDBusPath();
    static QString notifyDBusIface();
    static QString notifyDBusMethod();

    static qint32 updateApplicationArgs(const QStringList &arguments);

signals:
    void showGui(const QVariantMap &data = QVariantMap());

public slots:
    void handleApplicationArgs(const QStringList &arguments);
    void handleApplicationWakeUp(const QVariantMap &data);
};

#endif // APPLICATIONSERVICE_H
"""
        return content

    def generate_source(self) -> str:
        content =\
"""\
#include <QtGui/QGuiApplication>
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusConnectionInterface>
#include <QtDBus/QDBusMessage>

#include "applicationservice.h"

namespace {
const QString dbusServiceStr = QStringLiteral(DBUS_SERVICE);
const QString dbusPathStr = QStringLiteral(DBUS_PATH);
const QString dbusIfaceStr = QStringLiteral(DBUS_INTERFACE);
const QString dbusMethodAppArgs = QStringLiteral("handleApplicationArgs");
const QString dbusMethodAppWakeUp = QStringLiteral("handleApplicationWakeUp");
}

ApplicationService::ApplicationService(QObject *parent)
    : QDBusAbstractAdaptor(parent) {
    setAutoRelaySignals(true);

    auto dbus = QDBusConnection::sessionBus();
    dbus.registerObject(dbusPathStr, this, QDBusConnection::ExportAllSlots);
    if (!isRegistered()) {
        bool success = dbus.registerService(dbusServiceStr);
        if (!success)
            qApp->quit();
    }
}

ApplicationService::~ApplicationService() {
    auto dbus = QDBusConnection::sessionBus();
    dbus.unregisterObject(dbusPathStr);
    dbus.unregisterService(dbusServiceStr);
}

bool ApplicationService::isRegistered() {
    return QDBusConnection::sessionBus().interface()->isServiceRegistered(dbusServiceStr);
}

QString ApplicationService::notifyDBusService()
{
    return dbusServiceStr;
}

QString ApplicationService::notifyDBusPath()
{
    return dbusPathStr;
}

QString ApplicationService::notifyDBusIface()
{
    return dbusIfaceStr;
}

QString ApplicationService::notifyDBusMethod()
{
    return dbusMethodAppWakeUp;
}

qint32 ApplicationService::updateApplicationArgs(const QStringList &arguments) {
    auto message = QDBusMessage::createMethodCall(dbusServiceStr, dbusPathStr, dbusIfaceStr, dbusMethodAppArgs);
    message.setArguments(QList<QVariant>() << arguments);
    auto reply = QDBusConnection::sessionBus().call(message);

    return 0;
}

void ApplicationService::handleApplicationArgs(const QStringList &arguments) {
    if (arguments.indexOf(QStringLiteral("/no-gui")) != -1)
        return;

    emit showGui();
}

void ApplicationService::handleApplicationWakeUp(const QVariantMap &data) {
    emit showGui(data);
}
"""
        return content

    def save(self) -> None:
        filename = self.project_dir + self.path
        os.makedirs(os.path.dirname(filename), exist_ok=True)
        with open(filename + '.h', mode='w', encoding='utf-8') as f:
            f.write(self.generate_header())
        
        with open(filename + '.cpp', mode='w', encoding='utf-8') as f:
            f.write(self.generate_source())
