# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

import os
from configuration import Configuration
from generators.filegenerator import FileGenerator

class ApplicationControllerFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = 'src/applicationcontroller'

    def generate_header(self) -> str:
        content =\
"""\
#ifndef APPLICATIONCONTROLLER_H
#define APPLICATIONCONTROLLER_H

#include <QtCore/QObject>
#include <QtCore/QVariantMap>

class QQuickView;
namespace Aurora {
namespace PushNotifications {
class Client;
}
}

class ApplicationController : public QObject
{
    Q_OBJECT
    Q_PROPERTY(QString applicationId READ applicationId NOTIFY applicationIdChanged)
    Q_PROPERTY(QString registrationId READ registrationId NOTIFY registrationIdChanged)
public:
    explicit ApplicationController(QObject *parent = nullptr);
    ApplicationController(const QStringList &arguments, QObject *parent = nullptr);

    QString applicationId() const;
    QString registrationId() const;

signals:
    void applicationIdChanged(const QString &applicationId);
    void registrationIdChanged(const QString &registrationId);
    void dataReceived(const QVariantMap &data);

public slots:
    void showGui(const QVariantMap &data = QVariantMap());

private:
    QString getApplicationId() const;
    void setApplicationId(const QString &applicationId);
    void setRegistrationId(const QString &registrationId);

private:
    Aurora::PushNotifications::Client *m_notificationsClient;
    QQuickView *m_view;
    QString m_registrationId;
};

#endif // APPLICATIONCONTROLLER_H

"""
        return content

    def generate_source(self) -> str:
        content =\
"""\
#include <QtCore/QFile>
#include <QtCore/QDebug>
#include <QtCore/QJsonDocument>
#include <QtCore/QJsonObject>
#include <QtGui/QGuiApplication>
#include <QtQml/QQmlContext>
#include <QtQuick/QQuickView>
#include <auroraapp.h>
"""
        if self.config.webview_version == "ru.auroraos.WebView":
             content +=\
"""
#include <webenginecontext.h>
"""
        content +=\
"""
#include <push_client.h>
#include <nemonotifications-qt5/notification.h>

#include "applicationcontroller.h"
#include "applicationservice.h"

namespace {
const QString applicationIdFilePath = QStringLiteral(APP_ID_FILE_PATH);
}

using namespace Aurora::PushNotifications;

ApplicationController::ApplicationController(QObject *parent)
    : ApplicationController({  }, parent) {
}

ApplicationController::ApplicationController(const QStringList &arguments, QObject *parent)
    : QObject(parent)
    , m_notificationsClient(new Client(this))
    , m_view(nullptr) {
    setApplicationId(getApplicationId());

    if (arguments.indexOf(QStringLiteral("/no-gui")) == -1)
        showGui();

    connect(m_notificationsClient, &Client::clientInactive, [this]() {
        if (m_view == nullptr)
            qApp->quit();
    });
    connect(m_notificationsClient, &Client::pushSystemReadinessChanged, [](bool status) {
        qWarning() << "Push system is" << (status ? "available" : "not available");
    });
    connect(m_notificationsClient, &Client::registrationId, [this](const QString &registrationId) {
        setRegistrationId(registrationId);
    });
    connect(m_notificationsClient, &Client::registrationError, []() {
        qWarning() << "Push system have problems with registrationId";
    });
    connect(m_notificationsClient, &Client::notifications, [this](const PushList &pushList) {
        for (const auto &push : pushList) {
            auto jsonDcoument = QJsonDocument::fromJson(push.data.toUtf8());

            QVariant defaultAction = Notification::remoteAction(QStringLiteral("default"), tr("Open app"),
                                                                ApplicationService::notifyDBusService(),
                                                                ApplicationService::notifyDBusPath(),
                                                                ApplicationService::notifyDBusIface(),
                                                                ApplicationService::notifyDBusMethod(),
                                                                QVariantList() << jsonDcoument.object().toVariantMap());

            Notification notification;
            notification.setAppName(tr("Web Messenger"));
            notification.setSummary(push.title);
            notification.setBody(push.message);
            notification.setIsTransient(false);
            notification.setItemCount(1);
            notification.setRemoteAction(defaultAction);
            notification.publish();
        }
    });
}

QString ApplicationController::applicationId() const
{
    return m_notificationsClient->applicationId();
}

QString ApplicationController::registrationId() const
{
    return m_registrationId;
}

void ApplicationController::showGui(const QVariantMap &data) {
    if (m_view != nullptr) {
        m_view->raise();
        m_view->showFullScreen();
    } else {
        m_view = Aurora::Application::createView();
"""
        if self.config.webview_version == "ru.auroraos.WebView":
                content +=\
"""
        Aurora::WebView::WebEngineContext::InitBrowser();
"""
        content +=\
"""
        m_view->rootContext()->setContextProperty("AppCtrl", this);
        m_view->rootContext()->setContextProperty("targetUrl", TARGET_URL);
        m_view->setSource(Aurora::Application::pathToMainQml());
        m_view->show();
    }

    emit dataReceived(data);
}

QString ApplicationController::getApplicationId() const {
    QFile applicationIdFile(applicationIdFilePath);
    if (applicationIdFile.exists() && applicationIdFile.open(QIODevice::ReadOnly)) {
        auto data = applicationIdFile.readAll().trimmed();
        applicationIdFile.close();
        return data;
    } else {
        return QStringLiteral("applicationId not set");
    }
}

void ApplicationController::setApplicationId(const QString &applicationId) {
    if (m_notificationsClient->applicationId() == applicationId)
        return;

    m_notificationsClient->setApplicationId(applicationId);
    m_notificationsClient->registrate();

    emit applicationIdChanged(applicationId);
}

void ApplicationController::setRegistrationId(const QString &registrationId) {
    if (registrationId == m_registrationId)
        return;

    m_registrationId = registrationId;

    emit registrationIdChanged(registrationId);
}
"""
        return content

    def save(self) -> None:
        filename = self.project_dir + self.path
        os.makedirs(os.path.dirname(filename), exist_ok=True)
        with open(filename + '.h', mode='w', encoding='utf-8') as f:
            f.write(self.generate_header())
        
        with open(filename + '.cpp', mode='w', encoding='utf-8') as f:
            f.write(self.generate_source())
