# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

from configuration import Configuration
from generators.filegenerator import FileGenerator

class WebViewFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = 'qml/pages/WebViewPage.qml'

    def generate(self) -> str:
        content =\
f"""import QtQuick 2.0
import Sailfish.Silica 1.0
import Sailfish.WebView 1.0
import Sailfish.WebEngine 1.0
import {self.config.organization_name} 1.0
import "../components/"

Page {{
    id: page

    property bool viewSuspended: false

    function suspend() {{
        if (!page.viewSuspended) {{
            page.viewSuspended = true;
            webView.suspendView();
        }}
    }}

    function resume() {{
        if (viewSuspended) {{
            page.viewSuspended = false;
            webView.resumeView();
        }} else {{
            backgroundTimer.stop();
        }}
    }}

    property real maxHeight: {{
        switch (orientation) {{
        case Orientation.Portrait:
            return Screen.height;
        case Orientation.Landscape:
            return Screen.width;
        case Orientation.PortraitInverted:
            return Screen.height;
        case Orientation.LandscapeInverted:
            return Screen.width;
        }}
    }}

    property real cutoutOffset: {{
        switch (orientation) {{
        case Orientation.Portrait:
            return Math.max(SafeZoneRect.insets.top, SafeZoneRect.appInsets.top)
        case Orientation.Landscape:
            return Math.max(SafeZoneRect.insets.left,
                            SafeZoneRect.appInsets.top)
        case Orientation.PortraitInverted:
            return Math.max(SafeZoneRect.insets.bottom,
                            SafeZoneRect.appInsets.top)
        case Orientation.LandscapeInverted:
            return Math.max(SafeZoneRect.insets.right,
                            SafeZoneRect.appInsets.top)
        }}
    }}


    backNavigation: false

    ConnectionHeader {{
        id: connectionHeader

        connected: network.online && connectionChecker.connected

        IconButton {{
            icon.source: "image://theme/icon-m-reload"
            visible: !connectionChecker.running
            anchors {{
                right: parent.right
                verticalCenter: parent.verticalCenter
            }}
            onClicked: {{
                webView.reload();
                connectionChecker.ping();
            }}
        }}

        BusyIndicator {{
            visible: connectionChecker.running
            running: visible
            size: BusyIndicatorSize.Small
            anchors {{
                right: parent.right
                verticalCenter: parent.verticalCenter
                rightMargin: Theme.paddingSmall
            }}
        }}
    }}

    WebView {{
        id: webView
"""
        if self.config.background_activity:
            content +=\
"""
        active: true
"""
        if self.config.user_agent:
            content +=\
f"""
        httpUserAgent: "{self.config.user_agent}"
"""
        content += \
"""
        url: targetUrl
        anchors {
            left: parent.left
            right: parent.right
            top: connectionHeader.bottom
        }
        height: (maxHeight - connectionHeader.height - cutoutOffset)
        footerMargin: keyboardOpened ? cutoutOffset : 0

        urlLoadingOverride: function (data) {
"""
        if self.config.frame_urls:
            content += \
f"""
            var iframePattern = new RegExp("{'|'.join(self.config.frame_urls)}");
            if (iframePattern.test(data.url)) {{
                return true;
            }}
"""
        if self.config.external_urls:
            content += \
f"""
            var externalPattern = new RegExp("{'|'.join(self.config.external_urls)}");
            if (externalPattern.test(data.url)) {{
                Qt.openUrlExternally(data.url);
                return false;
            }}
"""
        if self.config.push_id_injector:
            for replacement in self.config.push_id_injector:
                content += \
f"""
            if (new RegExp("{replacement['trigger_on']}").test(data.url)) {{
                webView.url = "{replacement['replace_with']}".arg(AppCtrl.registrationId);
                return true;
            }}
"""
        content +=\
"""            connectionChecker.ping();
            return true;
        }
"""
        if self.config.push_app_id:
            content +=\
"""
        property var data
        property bool callHandler: false

        onDataChanged: {
            var url = data["url"];
            if (url) {
                callHandler = true;
                webView.url = url;
            }
        }
"""
            if self.config.push_handler_name:
                content +=\
f"""
        onLoadProgressChanged: {{
            if (callHandler && loadProgress == 100) {{
                callHandler = false;
                var dataArray = [];
                for (var prop in data) {{
                    if (prop === "url") {{
                        continue;
                    }}
                    dataArray.push("'%1':'%2'".arg(prop).arg(data[prop]));
                }}
                runJavaScript("{self.config.push_handler_name}({{%1}})".arg(dataArray.join(",")));
            }}
        }}
"""
        content +=\
"""    }

    ConnectionChecker {
        id: connectionChecker
    }

    ConnmanManager {
        id: network
    }
"""
        if not self.config.background_activity:
            content +=\
"""
    Connections {
        target: Qt.application
        onStateChanged: {
            if (Qt.application.state === Qt.ApplicationActive) {
                resume();
            } else {
                backgroundTimer.restart();
            }
        }
    }

    Timer {
        id: backgroundTimer

        interval: 60 * 1000
        running: false
        repeat: false
        onTriggered: {
            suspend()
        }
    }
"""
        if self.config.push_handler_name:
            content +=\
"""
    Connections {
        target: AppCtrl

        onDataReceived: webView.data = data
    }
"""
        if self.config.web_permissions:
            content +=\
f"""
    Component.onCompleted: {{
        {self._permission_list(self.config.web_permissions)}
    }}
"""
        content += "}\n"
        return content

    def _permission_list(self, list):
        policies = []
        for permission in self.config.web_permissions:
            if permission.lower() == 'geolocation':
                policies.append('WebEngineSettings.geolocationRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'microphone':
                policies.append('WebEngineSettings.microphoneRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'camera':
                policies.append('WebEngineSettings.cameraRequestPolicy = WebEngineSettings.Allow;')
            elif permission.lower() == 'cookie':
                policies.append('WebEngineSettings.cookieBehavior = WebEngineSettings.AcceptAll;')
            elif permission.lower() == 'popup':
                policies.append('WebEngineSettings.popupEnabled = true;')
        return '\n        '.join(policies)
