# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

from configuration import Configuration
from generators.filegenerator import FileGenerator

class MainFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = f"src/{self.config.application_name.lower()}.cpp"

    def generate(self) -> str:
        content =\
"""\
#include <QGuiApplication>
#include <QQmlContext>
#include <QQuickView>
#include <QScopedPointer>
#include <auroraapp.h>

#include "connectionchecker.h"
"""
        if self.config.push_app_id:
            content +=\
"""
#include "applicationcontroller.h"
#include "applicationservice.h"
"""
        content +=\
f"""
int main(int argc, char *argv[])
{{
    QScopedPointer<QGuiApplication> application(Aurora::Application::application(argc, argv));
    application->setOrganizationName(QStringLiteral("{self.config.organization_name}"));
    application->setApplicationName(QStringLiteral("{self.config.application_name}"));

    qmlRegisterType<ConnectionChecker>("{self.config.organization_name}", 1, 0, "ConnectionChecker");
"""
        if self.config.push_app_id:
            content +=\
"""
    auto applicationArgs = application->arguments();
    applicationArgs.removeFirst();

    if (ApplicationService::isRegistered()) {
        return ApplicationService::updateApplicationArgs(applicationArgs);
    } else {
        auto applicationService = new ApplicationService(application.data());
        auto applicationController = new ApplicationController(applicationArgs, application.data());

        QObject::connect(applicationService, &ApplicationService::showGui,
                         applicationController, &ApplicationController::showGui);

        return application->exec();
    }
}
"""
        else:
            content +=\
f"""
    QScopedPointer<QQuickView> view(Aurora::Application::createView());
    view->rootContext()->setContextProperty("targetUrl", TARGET_URL);
    view->setSource(
            Aurora::Application::pathTo(QStringLiteral("qml/{self.config.full_name}.qml")));
    view->show();

    return application->exec();
}}
"""
        return content
