# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

from configuration import Configuration
from generators.filegenerator import FileGenerator

class ConnectionCheckerFileGenerator(FileGenerator):
    def __init__(self, config: Configuration, project_dir: str = './') -> None:
        super().__init__(config, project_dir)
        self.path = 'src/connectionchecker.h'

    def generate(self) -> str:
        content =\
r"""#ifndef CONNECTIONCHECKER_H
#define CONNECTIONCHECKER_H

#include <QNetworkAccessManager>
#include <QNetworkConfigurationManager>
#include <QNetworkReply>
#include <QUrl>
#include <QObject>

class ConnectionChecker : public QObject
{
    Q_OBJECT
    Q_PROPERTY(bool connected READ connected NOTIFY connectedChanged)
    Q_PROPERTY(bool running READ running NOTIFY runningChanged)
public:
    /*!
     * \brief Add handler for network manager's replies.
     * Send initial request to verify connection to the host.
     */
    ConnectionChecker(QObject *parent = nullptr)
        : QObject(parent), _connected(true), _running(false)
    {
        QObject::connect(&nam, SIGNAL(finished(QNetworkReply *)), this,
                         SLOT(handleReply(QNetworkReply *)));
        ping();
    }

    /*!
     * \brief Whether the last HTTP request to the host was successful.
     */
    bool connected() { return _connected; }

    /*!
     * \brief Whether there is a running HTTP request.
     */
    bool running() { return _running; }

    /*!
     * \brief Send a GET request with the 'Host' header set to target URL
     * so that we can verify the connection between client and the host.
     */
    Q_INVOKABLE void ping()
    {
        if (ncm.isOnline()) {
            setRunning(true);
            QUrl url(TARGET_URL);
            QNetworkRequest req(url);
            req.setRawHeader(QByteArray("Host"), url.host(QUrl::FullyEncoded).toUtf8());
            nam.get(req);
        } else {
            setConnected(false);
        }
    }

signals:
    void connectedChanged(bool connected);
    void runningChanged(bool running);

private slots:
    /*!
     * \brief Handle network manager's reply to a previously sent request.
     * The lack of reply's content means the host is unreachable.
     * \param reply HTTP reply to a request.
     */
    void handleReply(QNetworkReply* reply) {
        setRunning(false);
        setConnected(reply->error() == QNetworkReply::NoError);
        reply->deleteLater();
    }

private:
    void setConnected(bool connected)
    {
        if (connected != _connected) {
            _connected = connected;
            emit connectedChanged(connected);
        }
    }
    void setRunning(bool running)
    {
        if (running != _running) {
            _running = running;
            emit runningChanged(running);
        }
    }

    QNetworkConfigurationManager ncm;
    QNetworkAccessManager nam;
    bool _connected;
    bool _running;
};

#endif // CONNECTIONCHECKER_H
"""
        return content
