# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

import argparse
from ast import arg
import json
import os
import pathlib
import shutil
from configuration import Configuration
from generators.Sailfish_WebView import main_sailfish, webview, connectionchecker, connectionheader, connmanmanager
from generators.ru_auroraos_WebView import aboutpage, main_auroraos, mainpage
from generators import desktop, spec, pro, cover, applicationwindow, applicationcontroller, applicationservice

def generate(file: str, project_dir: str) -> int:
    if not os.path.exists(file):
        raise FileNotFoundError(f"{file} does not exist")

    with open(file, encoding="utf-8") as f:
        config = Configuration(json.load(f))

    desktop.DesktopFileGenerator(config=config, project_dir=project_dir).save()
    spec.SpecFileGenerator(config=config, project_dir=project_dir).save()
    pro.ProFileGenerator(config=config, project_dir=project_dir).save()

    if config.webview_version == "Sailfish.WebView":
        main_sailfish.MainFileGenerator(config=config, project_dir=project_dir).save()
        connectionchecker.ConnectionCheckerFileGenerator(config=config, project_dir=project_dir).save()
        connectionheader.ConnectionHeaderFileGenerator(config=config, project_dir=project_dir).save()
        connmanmanager.ConnmanManagerFileGenerator(config=config, project_dir=project_dir).save()
        webview.WebViewFileGenerator(config=config, project_dir=project_dir).save()
    elif config.webview_version == "ru.auroraos.WebView":
        main_auroraos.MainFileGenerator(config=config, project_dir=project_dir).save()
        mainpage.MainPageFileGenerator(config=config, project_dir=project_dir).save()
        aboutpage.AboutPageFileGenerator(config=config, project_dir=project_dir).save()
    
    if config.push_app_id:
        applicationcontroller.ApplicationControllerFileGenerator(config=config, project_dir=project_dir).save()
        applicationservice.ApplicationServiceFileGenerator(config=config, project_dir=project_dir).save()
    if not config.background_activity:
        cover.CoverFileGenerator(config=config, project_dir=project_dir).save()

        os.makedirs(f"{config.application_name}/qml/icons/", exist_ok=True)
        shutil.copy('./default_icons/defaultcovericon.svg',
                    f"{config.application_name}/qml/icons/{config.application_name}.svg")
    applicationwindow.ApplicationWindowFileGenerator(config=config, project_dir=project_dir).save()
    if config.icon:
        if not list(pathlib.Path(config.icon).glob('**/*.png')):
            raise FileNotFoundError(f"Directory {config.icon} does not contains icons")

        for icon in pathlib.Path(config.icon).glob('**/*.png'):
            dest_dir = f"{project_dir}/{config.application_name}/icons/{icon.parent.name}/"
            os.makedirs(os.path.dirname(dest_dir), exist_ok=True)
            shutil.copy(icon, f"{dest_dir}{config.full_name}.png")
    else:
        for resolution in [86, 108, 128, 172]:
            os.makedirs(f"{project_dir}/{config.application_name}/icons/{resolution}x{resolution}", exist_ok=True)
            shutil.copy(f"./default_icons/{resolution}x{resolution}/defaulticon.png",
                        f"{project_dir}/{config.application_name}/icons/{resolution}x{resolution}/{config.full_name}.png")


def parse_arguments() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description='Generate WebView-based Aurora OS application from JSON configuration.'
    )
    parser.add_argument(
        dest='file',
        type=str,
        help='Path to the JSON configuration file.'
    )
    parser.add_argument(
        '-p', '--project-dir',
        help='Path to the generated project location.',
        type=str,
        default="./"
    )

    return parser.parse_args()

if __name__ == '__main__':
    args = parse_arguments()
    generate(args.file, args.project_dir)
