# SPDX-FileCopyrightText: 2024 Open Mobile Platform LLC <community@omp.ru>
# SPDX-License-Identifier: BSD-3-Clause

import re

class ConfigurationError(Exception):
    def __init__(self, *args: object) -> None:
        super().__init__(*args)

class Configuration:
    PERMISSIONS = frozenset(['AccessSecurityLog', 'LogSecurityEvents', 'Sensors', 'SecureStorage', 'PushNotifications',
                             'DeviceInfo', 'Audio', 'Bluetooth', 'Camera', 'Internet', 'Location', 'MediaIndexing',
                             'Microphone', 'NFC', 'RemovableMedia', 'UserDirs', 'WebView', 'Documents', 'Downloads',
                             'Music', 'Pictures', 'PublicDir', 'Videos'])
    VERSIONS = frozenset(['Sailfish.WebView', 'ru.auroraos.WebView'])
    WEB_PERMISSIONS = frozenset(['geolocation', 'microphone', 'camera', 'cookie', 'popup'])
    REQUIRED_KEYS = frozenset(['application_name', 'organization_name', 'summary', 'version',
                               'application_permission', 'background_activity', 'url', 'webview_version'])
    OPTIONAL_KEYS = frozenset(['desktop_name', 'description', 'icon', 'external_urls', 'frame_urls', 'web_permissions',
                               'user_agent', 'push_app_id', 'push_handler_name', 'push_id_injector'])
    KEYS = REQUIRED_KEYS.union(OPTIONAL_KEYS)

    def __init__(self, json: dict) -> None:
        keys_input = set()
        for key, value in json.items():
            if key not in self.KEYS:
                raise ConfigurationError(f"Unknown key {key}")
            
            if key == 'webview_version':
                if value not in self.VERSIONS:
                    raise ConfigurationError(f"Unknown value {value} for {key}")
            
            if key == 'application_permission' or key == 'external_urls' or key == 'frame_urls' or key == 'web_permissions' or key == 'push_id_injector':
                if type(value) != list:
                    raise ConfigurationError(f"{key} must be a list")

            if key == 'application_permission':
                unknown_permissions = set(value).difference(self.PERMISSIONS)
                if unknown_permissions:
                    raise ConfigurationError(f"Unknown application permission {', '.join(unknown_permissions)}")
            elif key == "web_permissions":
                unknown_permissions = set([permission.lower() for permission in value]).difference(self.WEB_PERMISSIONS)
                if unknown_permissions:
                    raise ConfigurationError(f"Unknown Web permission {', '.join(unknown_permissions)}")
            elif key == 'organization_name':
                value = '.'.join(value.split('.')[::-1])
            
            keys_input.add(key)

            setattr(self, key, value)
        
        missing_keys = self.REQUIRED_KEYS.difference(set(json.keys()))
        if missing_keys:
            raise ConfigurationError(f"{', '.join(missing_keys)} must be set")

        if not locals().get('icon', None):
            locals()['icon'] = './default_icons'

        self.full_name = self.organization_name + '.' + self.application_name

        if not hasattr(self, 'desktop_name'):
            setattr(self, 'desktop_name', self.application_name)

        for key in self.OPTIONAL_KEYS:
            if not hasattr(self, key):
                setattr(self, key, None)
