// SPDX-FileCopyrightText: 2022 - 2023 Open Mobile Platform LLC <community@omp.ru>
// SPDX-License-Identifier: BSD-3-Clause

#include "basedocument.h"

#include "documentmapper.h"

DocumentMapper::DocumentMapper(QQuickItem *parent) : QQuickItem(parent),
    m_orientation(Qt::Vertical),
    m_spacing(SPACING_DEFAULT_VALUE),
    m_contentHeight(0.0),
    m_contentWidth(0.0),
    m_lastPageActualSize(0.0),
    m_pageAspectRatio(1.0),
    m_reverse(false)
{
    connect(this, &DocumentMapper::widthChanged, this, &DocumentMapper::_mapPages);
    connect(this, &DocumentMapper::heightChanged, this, &DocumentMapper::_mapPages);
    connect(parentItem(), &QQuickItem::widthChanged, this, &DocumentMapper::_updateSize);
    connect(parentItem(), &QQuickItem::heightChanged, this, &DocumentMapper::_updateSize);
}

QSGNode *DocumentMapper::updatePaintNode(QSGNode *, QQuickItem::UpdatePaintNodeData *)
{
    return nullptr;
}

QHash<int, PagePosition> DocumentMapper::actualMap() const
{
    return m_actualPagesCoordinates;
}

QPointF DocumentMapper::pagePosition(int pageIndex) const
{
    if (pageIndex < 0 || pageIndex > m_originalPagesMap.pages.size())
        return {  };

    auto pagePosition = m_originalPagesMap.pages.at(pageIndex);
    return { static_cast<qreal>(pagePosition.startX), static_cast<qreal>(pagePosition.startY)};
}

qreal DocumentMapper::contentHeight() const
{
    return m_contentHeight;
}

qreal DocumentMapper::contentWidth() const
{
    return m_contentWidth;
}

qreal DocumentMapper::spacing() const
{
    return m_spacing;
}

qreal DocumentMapper::lastPageActualSize() const
{
    return m_lastPageActualSize;
}

void DocumentMapper::forceUpdate()
{
    _updateSize();
    _mapPages();
}

void DocumentMapper::setDocumentProvider(BaseDocument *documentProvider)
{
    if (documentProvider == nullptr)
        return;

    if (m_documentProvider == documentProvider)
        return;

    m_documentProvider = documentProvider;

    auto pageCount = m_documentProvider->count();
    m_originalPagesMap.pages.clear();
    m_originalPagesMap.pages.reserve(pageCount);

    if (pageCount <= 1)
        m_spacing = 0;

    qreal originalPageStartX = 0.0;
    qreal originalPageStartY = 0.0;
    bool isNeedeRecalcWidth = qFuzzyCompare(static_cast<double>(m_pageAspectRatio),
                                            static_cast<double>(1.0));
    for (int i = 0; i < pageCount; ++i) {
        auto pageSize = m_documentProvider->pageSize(i);

        auto pageWidth = isNeedeRecalcWidth ? pageSize.width()
                                            : pageSize.height() * m_pageAspectRatio;
        m_originalPagesMap.pages.append({originalPageStartX,
                                         originalPageStartY,
                                         originalPageStartX + pageWidth,
                                         originalPageStartY + pageSize.height()});
        originalPageStartX += pageWidth;
        originalPageStartY += pageSize.height();
    }

    _mapPages();
}

void DocumentMapper::setOrientation(Qt::Orientation orientation)
{
    if (m_orientation == orientation)
        return;

    m_orientation = orientation;
    _mapPages();
}

void DocumentMapper::setSpecialPageIndexes(const std::map<int, int> &specialRangeIndexes,
                                           const std::set<int> &specialIndexes)
{
    m_spacialRangeIndexes = specialRangeIndexes;
    m_specialIndexes = specialIndexes;

    _mapPages();
}

void DocumentMapper::setPageAspectRatio(qreal pageAspectRatio)
{
    if (qFuzzyCompare(static_cast<double>(m_pageAspectRatio), static_cast<double>(pageAspectRatio)))
        return;

    m_pageAspectRatio = pageAspectRatio;
    if (!m_originalPagesMap.pages.empty())
        _mapPages();
}

void DocumentMapper::setReverse(bool reverse)
{
    if (m_reverse == reverse)
        return;

    m_reverse = reverse;
    _mapPages();
}

void DocumentMapper::setSpacing(qreal spacing)
{
    if (qFuzzyCompare(m_spacing, spacing))
        return;

    m_spacing = spacing;
    if (!m_originalPagesMap.pages.empty())
        _mapPages();
}

void DocumentMapper::_mapPages()
{
    if (width() <= 0 || height() <= 0)
        return;

    if (m_documentProvider == nullptr)
        return;

    int pageIndex = 0;
    auto contentHeight = -1.0f;
    auto contentWidth = -1.0f;
    PagePosition position;

    auto maxPageIndex = m_originalPagesMap.pages.size() - 1;
    if (!m_originalPagesMap.pages.empty() && !m_specialIndexes.empty()
        && m_specialIndexes.find(maxPageIndex) == m_specialIndexes.end()
        && m_originalPagesMap.pages.size() < *m_specialIndexes.rbegin()) {
        m_specialIndexes.insert(maxPageIndex);
    }

    switch (m_orientation) {
    case Qt::Vertical: {
        for (auto &page : m_originalPagesMap.pages) {
            auto inverseIndex = m_reverse ? m_originalPagesMap.pages.size() - 1 - pageIndex : pageIndex;
            if (isHasSpecial() && !isIndexSpecal(inverseIndex)) {
                ++pageIndex;
                continue;
            }

            position.end += width() * page.heightToWidthRatio() + m_spacing;
            m_actualPagesCoordinates.insert(inverseIndex, position);
            m_lastPageActualSize = position.end - position.start - m_spacing;
            position.start = position.end;
            ++pageIndex;
        }

        contentHeight = position.end;
        contentWidth = -1;
        break;
    }
    case Qt::Horizontal : {
        for (auto &page : m_originalPagesMap.pages) {
            auto inverseIndex = m_reverse ? m_originalPagesMap.pages.size() - 1 - pageIndex : pageIndex;
            if (isHasSpecial() && !isIndexSpecal(inverseIndex)) {
                ++pageIndex;
                continue;
            }

            auto fitHeight = qMin(width() * page.heightToWidthRatio(), height());
            position.end += fitHeight / page.heightToWidthRatio() + m_spacing;
            m_actualPagesCoordinates.insert(inverseIndex, position);
            m_lastPageActualSize = position.end - position.start - m_spacing;
            position.start = position.end;
            ++pageIndex;
        }

        contentHeight = - 1;
        contentWidth = position.end;
        break;
    }
    }

    if (!qFuzzyCompare(double(contentHeight), double(m_contentHeight))) {
        m_contentHeight = contentHeight;
        emit contentHeightChanged(m_contentHeight);
    }

    if (!qFuzzyCompare(double(contentWidth), double(m_contentWidth))) {
        m_contentWidth = contentWidth;
        emit contentWidthChanged(m_contentWidth);
    }

    emit mapEnd();
}

void DocumentMapper::_updateSize()
{
    if (parentItem() == nullptr)
        return;

    setWidth(parentItem()->width());
    setHeight(parentItem()->height());
}

bool DocumentMapper::isIndexSpecal(int pageIndex) const
{
    auto it = std::find_if(m_spacialRangeIndexes.begin(),
                           m_spacialRangeIndexes.end(),
                           [pageIndex](const std::pair<int, int> &t) -> bool {
                               return pageIndex >= t.first && pageIndex <= t.second;
                           });

    return m_specialIndexes.find(pageIndex) != m_specialIndexes.end() ||
                it != m_spacialRangeIndexes.end();
}

bool DocumentMapper::isHasSpecial() const
{
    return !(m_spacialRangeIndexes.empty() && m_specialIndexes.empty());
}

int DocumentMapper::minSpecialIndex() const
{
    int firstMin = m_specialIndexes.empty() ? 0 : *m_specialIndexes.begin();

    if (m_spacialRangeIndexes.empty())
        return firstMin;

    return std::min(firstMin,
                    std::min(m_spacialRangeIndexes.begin()->first, m_originalPagesMap.pages.size() - 1));
}

int DocumentMapper::maxSpecialIndex() const
{
    int firstMax = m_specialIndexes.empty()
                       ? 0
                       : std::min(*m_specialIndexes.rbegin(), m_originalPagesMap.pages.size() - 1);

    if (m_spacialRangeIndexes.empty())
        return firstMax;

    if (m_spacialRangeIndexes.size() == 1)
        return std::max(firstMax,
                        std::min(m_spacialRangeIndexes.begin()->second,
                                 m_originalPagesMap.pages.size() - 1));

    auto it = std::max_element(m_spacialRangeIndexes.begin(),
                               m_spacialRangeIndexes.end(),
                               [](const std::pair<int, int> &a, const std::pair<int, int> &b) {
                                   return a.second < b.second;
                               });

    return std::min(std::max(firstMax, it->second), m_originalPagesMap.pages.size() - 1);
}

qreal DocumentMapper::pageAspectRatio() const
{
    return m_pageAspectRatio;
}

PagePosition DocumentMapper::actualPagePosition(int pageIndex) const
{
    if (!m_actualPagesCoordinates.contains(pageIndex))
        return {  };

    return m_actualPagesCoordinates.value(pageIndex);
}

PageGeometry DocumentMapper::originalPageGeometry(int pageIndex) const
{
    if (pageIndex < 0 || pageIndex > m_originalPagesMap.pages.size())
        return {  };

    return m_originalPagesMap.pages.at(pageIndex);
}
