/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QQuickWindow>
#include <QSGSimpleTextureNode>

#include "baseannotation.h"

#include "pdfsimplenote.h"

PdfSimpleNote::PdfSimpleNote(QQuickItem *parent, BaseAnnotation *source) : QQuickItem(parent),
    m_highlighted(false),
    m_needUpdateImage(false),
    m_noteSource(source)
{
    setFlag(QQuickItem::ItemHasContents, true);
    setAcceptedMouseButtons(Qt::AllButtons);

    connect(this, &PdfSimpleNote::xChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::yChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::widthChanged, this, &PdfSimpleNote::update);
    connect(this, &PdfSimpleNote::heightChanged, this, &PdfSimpleNote::update);

    connect(parent, &QQuickItem::yChanged, this, &PdfSimpleNote::clearHighlight);
    connect(parent, &QQuickItem::xChanged, this, &PdfSimpleNote::clearHighlight);

    m_paper = QImage(1, 1, QImage::Format_RGBA8888);
    auto fillColor = m_noteSource->color.isValid()
            ? m_noteSource->color
            : qRgba(21, 222, 225, HIGHLIGHT_OPACITY_NOTE_OFF);
    fillColor.setAlpha(HIGHLIGHT_OPACITY_NOTE_OFF);
    m_paper.fill(fillColor);
}

QSGNode *PdfSimpleNote::updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *)
{
    if (height() <= 0 || width() <= 0)
        return nullptr;

    auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

    if (node == nullptr) {
        node = new QSGSimpleTextureNode();
        node->setOwnsTexture(true);
    }

    if (m_needUpdateImage || node->texture() == nullptr) {
        if (node->texture() != nullptr)
            node->texture()->deleteLater();

        node->setTexture(window()->createTextureFromImage(m_paper));

        m_needUpdateImage = false;
    }

    node->setRect(boundingRect());

    return node;
}

BaseAnnotation *PdfSimpleNote::source() const
{
    return m_noteSource;
}

bool PdfSimpleNote::event(QEvent *event)
{
    if (opacity() < 1.0f)
        return QQuickItem::event(event);

    switch (event->type()) {
    case QEvent::MouseButtonPress:
        _setHighlight(true);
        return true;

    case QEvent::MouseButtonRelease:
        if (m_noteSource != nullptr)
            emit triggered(m_noteSource->content, m_noteSource->author);
        _setHighlight(false);
        break;

    case QEvent::MouseMove:
        _setHighlight(false);
        break;

    default:
        break;
    }

    return QQuickItem::event(event);
}

void PdfSimpleNote::_setHighlight(bool highlight)
{
    if (!m_noteSource)
        return;

    if (m_highlighted != highlight) {
        m_highlighted = highlight;
        m_needUpdateImage = true;
        auto fillColor = m_noteSource->color.isValid()
                ? m_noteSource->color
                : qRgba(21, 222, 225, HIGHLIGHT_OPACITY_NOTE_OFF);
        fillColor.setAlpha(highlight ? HIGHLIGHT_OPACITY_NOTE_ON : HIGHLIGHT_OPACITY_NOTE_OFF);
        m_paper.fill(fillColor);
        update();
    }
}

void PdfSimpleNote::clearHighlight()
{
   _setHighlight(false);
}
