/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/
#include <QQuickWindow>
#include <QSGSimpleTextureNode>

#include "baseannotation.h"

#include "pdfsimpleannotation.h"

PdfSimpleAnnotation::PdfSimpleAnnotation(QQuickItem *parent, BaseAnnotation *source) : QQuickItem(parent),
    m_annotationSource(source),
    m_highlighted(false),
    m_needUpdateImage(false)
{
    setFlag(QQuickItem::ItemHasContents, true);
    setAcceptedMouseButtons(Qt::AllButtons);

    connect(this, &PdfSimpleAnnotation::xChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::yChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::widthChanged, this, &PdfSimpleAnnotation::update);
    connect(this, &PdfSimpleAnnotation::heightChanged, this, &PdfSimpleAnnotation::update);

    connect(parent, &QQuickItem::yChanged, this, &PdfSimpleAnnotation::clearHighlight);
    connect(parent, &QQuickItem::xChanged, this, &PdfSimpleAnnotation::clearHighlight);

    m_paper = QImage(1, 1, QImage::Format_RGBA8888);
    m_paper.fill(qRgba(1, 222, 121, HIGHLIGHT_OPACITY_ANNOT_OFF));
}

QSGNode *PdfSimpleAnnotation::updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *)
{
    if (height() <= 0 || width() <= 0)
        return nullptr;

    auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

    if (node == nullptr) {
        node = new QSGSimpleTextureNode();
        node->setOwnsTexture(true);
    }

    if (m_needUpdateImage || node->texture() == nullptr) {
        if (node->texture() != nullptr)
            node->texture()->deleteLater();

        node->setTexture(window()->createTextureFromImage(m_paper));

        m_needUpdateImage = false;
    }

    node->setRect(boundingRect());

    return node;
}

bool PdfSimpleAnnotation::event(QEvent *event)
{
    if (opacity() < 1.0f)
        return QQuickItem::event(event);

    switch (event->type()) {
    case QEvent::MouseButtonPress:
        _setHighlight(true);
        return true;

    case QEvent::MouseButtonRelease:
        if (m_annotationSource != nullptr)
            emit triggered(m_annotationSource);
        _setHighlight(false);
        break;

    case QEvent::MouseMove:
        _setHighlight(false);
        break;

    default:
        break;
    }

    return QQuickItem::event(event);
}

void PdfSimpleAnnotation::_setHighlight(bool highlight)
{
    if (m_highlighted != highlight) {
        m_highlighted = highlight;
        m_needUpdateImage = true;
        m_paper.fill(qRgba(1, 222, 121, highlight ? HIGHLIGHT_OPACITY_ANNOT_ON : HIGHLIGHT_OPACITY_ANNOT_OFF));
        update();
    }
}

void PdfSimpleAnnotation::clearHighlight()
{
    _setHighlight(false);
}

BaseAnnotation *PdfSimpleAnnotation::source() const
{
    return m_annotationSource;
}
