/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QTimer>
#include <QSGTexture>
#include <QSGSimpleTextureNode>
#include <QPropertyAnimation>
#include <QQuickWindow>
#include <QThreadPool>

#include "basepage.h"
#include "bitmaploadworker.h"

#include "pdfpagetile.h"

PdfPageTile::PdfPageTile(QQuickItem *parent) : QQuickItem(parent),
    m_needUpdateImage(false),
    m_renderInProcess(false),
    m_renderable(false),
    m_imageScale(1.0)
{
    setFlag(QQuickItem::ItemHasContents, true);

    m_timer = new QTimer(this);
    connect(m_timer, &QTimer::timeout, this, &PdfPageTile::_loadBitmap);

    m_timer->setInterval(150);
    m_timer->setSingleShot(true);

    connect(this, &PdfPageTile::renderableChanged, m_timer, static_cast<void(QTimer::*)(void)>(&QTimer::start));
    connect(this, &PdfPageTile::bitmapError, m_timer, static_cast<void(QTimer::*)(void)>(&QTimer::start));
    connect(this, &PdfPageTile::partReady, this, &PdfPageTile::update);

    m_animation = new QPropertyAnimation(this, "opacity");
    m_animation->setDuration(180);
    m_animation->setStartValue(0.42);
    m_animation->setEndValue(1.0);

    connect(m_animation, &QPropertyAnimation::finished, this, &PdfPageTile::animationEnded);
}

PdfPageTile::~PdfPageTile()
{
    m_timer->stop();
}

QSGNode *PdfPageTile::updatePaintNode(QSGNode *oldNode, QQuickItem::UpdatePaintNodeData *)
{
    if (height() <= 0 || width() <= 0)
        return nullptr;

    auto node = static_cast<QSGSimpleTextureNode *>(oldNode);

    if (node == nullptr) {
        node = new QSGSimpleTextureNode();
        node->setOwnsTexture(true);
    }

    if (node == nullptr)
        qFatal("Error: create node error.");

    if (node->texture() == nullptr) {
        auto paper = QImage(1, 1, QImage::Format_RGBA8888);
        paper.fill(Qt::white);

        node->setTexture(window()->createTextureFromImage(paper, QQuickWindow::TextureHasAlphaChannel));

        if (!m_pagePart.isNull())
            m_needUpdateImage = true;
    }

    if (m_renderable && m_needUpdateImage) {
        if (m_pagePart.isNull()) {
            emit bitmapError();
        } else {
            node->texture()->deleteLater();
            node->setTexture(window()->createTextureFromImage(m_pagePart));
        }

        m_needUpdateImage = false;
    }

    if (!m_renderInProcess && m_pagePart.isNull())
        emit bitmapError();

    node->setRect(boundingRect());

    return node;
}

void PdfPageTile::geometryChanged(const QRectF &newGeometry, const QRectF &oldGeometry)
{
    QQuickItem::geometryChanged(newGeometry, oldGeometry);
    m_timer->start();
}

void PdfPageTile::setPageSource(QSharedPointer<BasePage> pageSource)
{
    if (!pageSource)
        return;

    m_pageSource = pageSource;
    m_timer->start();
}

bool PdfPageTile::renderable() const
{
    return m_renderable;
}

bool PdfPageTile::isBitmap() const
{
    return !m_pagePart.isNull();
}

void PdfPageTile::setImageScale(qreal imageScale)
{
    if (qFuzzyCompare(imageScale, m_imageScale))
        return;

    m_imageScale = imageScale;
    m_timer->start();
}

void PdfPageTile::render()
{
    m_timer->start();
}

void PdfPageTile::setRenderable(bool renderable)
{
    if (m_renderable == renderable)
        return;

    m_renderable = renderable;
    emit renderableChanged(m_renderable);
}

void PdfPageTile::_loadBitmap()
{
    if (!m_renderable)
        return;

    if (!m_pageSource)
        return;

    if (m_renderInProcess)
        return;

    if (width() <= 0 || height() <= 0)
        return;

    auto pageSize = m_pageSource->originalSize();
    if (!pageSize.isFinished() || pageSize.isCanceled()) {
        m_timer->start();
        return;
    }

    auto pageScaleX = width() / pageSize.result().width();
    auto pageScaleY = height() / pageSize.result().height();

    auto *bitmapLoader = new BitmapLoaderWorker(m_pageSource, pageScaleX, pageScaleY, 0, m_imageScale, QPointF(-x(), -y()));
    connect(this, &PdfPageTile::destroyed, bitmapLoader, &BitmapLoaderWorker::cancel);
    connect(bitmapLoader, &BitmapLoaderWorker::done, this, [this](QImage result) {
        m_renderInProcess = false;
        if (result.isNull()) {
            emit bitmapError();
        } else {
            auto needAnimation = m_pagePart.isNull();
            m_pagePart = result;
            m_needUpdateImage = true;
            emit partReady();

            if (needAnimation) {
                m_animation->setStartValue(m_pagePart.isNull() ? 0.42 : 0.95);
                m_animation->start();
            }
            update();
        }
    });

    QThreadPool::globalInstance()->start(bitmapLoader);
    m_renderInProcess = true;
}
