/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include <QFile>
#include <QFutureWatcher>
#include <QSizeF>
#include <QThreadPool>

#include <amberpdf/pdfdocument.h>
#include <amberpdf/pdfbookmark.h>

#include "pagessizesloader.h"
#include "pagepreloader.h"
#include "basepage.h"
#include "basebookmark.h"

#include "pdfdocumentitem.h"

PdfDocumentItem::PdfDocumentItem(QObject *parent) : BaseDocument(parent)
{
    qRegisterMetaType<QHash<int, QSizeF>>();

    m_status = DocumentStatus::Null;
}

PdfDocumentItem::~PdfDocumentItem() = default;

QString PdfDocumentItem::path() const
{
    return m_pdfiumDocument ? m_pdfiumDocument->path() : QStringLiteral("");
}

QSizeF PdfDocumentItem::pageSize(int pageNumber) const
{
    if (m_status != DocumentStatus::Ready)
        return {  };

    return m_pageSizes.value(pageNumber);
}

int PdfDocumentItem::count() const
{
    return m_pdfiumDocument ? m_pdfiumDocument->pageCount() : -1;
}

QSharedPointer<BasePage> PdfDocumentItem::loadPage(int pageIndex)
{
    if (pageIndex < 0 || pageIndex >= count())
        return {  };

    if (m_loadedPages.contains(pageIndex))
        return m_loadedPages.value(pageIndex);

    if (m_pagesInProcess.contains(pageIndex))
        return {  };

    auto pageLoader = new PagePreloader(m_pdfiumDocument, pageIndex);
    connect(pageLoader, &PagePreloader::done, this, [&] (int loadedPageIndex, PageLoadStatus loadStatus) {
        if (loadStatus == PageLoadStatus::Success)
            m_loadedPages.insert(loadedPageIndex, QSharedPointer<BasePage>(
                                     new PdfPageItem(m_pdfiumDocument->page(loadedPageIndex))));

        emit pageLoaded(loadedPageIndex, loadStatus);
    });
    QThreadPool::globalInstance()->start(pageLoader);
    m_pagesInProcess.insert(pageIndex);

    return {  };
}

void PdfDocumentItem::startLoadBookmarks() const
{
    auto *bookmarksWatcher = new QFutureWatcher<QVector<PdfBookmark>>();
    connect(bookmarksWatcher,  &QFutureWatcher<QVector<PdfBookmark>>::finished,
            this,  [bookmarksWatcher, ctx = const_cast<PdfDocumentItem *>(this)]() {
                if (bookmarksWatcher == nullptr)
                    return;

                if (bookmarksWatcher->isFinished() && !bookmarksWatcher->isCanceled()) {
                    auto bookmarks = bookmarksWatcher->result();
                    if (bookmarks.isEmpty()) {
                        bookmarksWatcher->deleteLater();
                        return;
                    }

                    qDeleteAll(ctx->m_baseBookmarks);
                    ctx->m_baseBookmarks.clear();

                    for (const auto &bookmark : bookmarks) {
                        ctx->m_baseBookmarks.push_back(new BaseBookmark{ bookmark.title, bookmark.page,
                                                                         bookmark.level, bookmark.locationInPage
                                                       });
                    }

                    emit ctx->bookmarksLoaded();
                }

                bookmarksWatcher->deleteLater();
    });
    bookmarksWatcher->setFuture(m_pdfiumDocument->bookmarks());
}

QVector<BaseBookmark *> PdfDocumentItem::bookmarks() const
{
    return m_baseBookmarks;
}

int PdfDocumentItem::fileVersion() const
{
    return (m_pdfiumDocument.isNull() ? -1 : m_pdfiumDocument->fileVersion());
}

bool PdfDocumentItem::saveDocumentAs(const QString &path) const
{
    if (!m_pdfiumDocument)
        return false;

    if (!QFile::exists(path))
        return m_pdfiumDocument->saveDocumentAs(path);

    static QString firstSuffix = QStringLiteral(".amber_bak1");
    auto backupPathFirst = path + firstSuffix;
    auto saveResult =  m_pdfiumDocument->saveDocumentAs(backupPathFirst);
    if (!saveResult)
        return false;

    static QString seconsSuffix = QStringLiteral(".amber_bak2");
    auto backupPathSecond = path + seconsSuffix;
    saveResult = QFile::rename(path, backupPathSecond);
    if (!saveResult)
        return false;

    saveResult = QFile::rename(backupPathFirst, path);
    if (!saveResult)
        return false;

    QFile::remove(backupPathFirst);
    QFile::remove(backupPathSecond);

    return true;
}

void PdfDocumentItem::setPath(const QString &path)
{
    if (m_pdfiumDocument && (m_pdfiumDocument->path() == path))
        return;

    if (m_status != DocumentStatus::Loading) {
        m_status = DocumentStatus::Loading;
        emit statusChanged(m_status);
    }

    m_pdfiumDocument.reset(new PdfDocument());

    if (m_pdfiumDocument == nullptr) {
        m_status = DocumentStatus::Error;
        emit statusChanged(m_status);
        return;
    }

    connect(m_pdfiumDocument.data(), &PdfDocument::statusChanged, this, [&](PdfDocument::DocumentStatus status) {
        if (status == PdfDocument::Success || status == PdfDocument::Loading) {
            m_pdfiumDocument->pageCount();
            return;
        }

        m_status = DocumentStatus::Error;
        emit statusChanged(m_status);
    });

    connect(m_pdfiumDocument.data(), &PdfDocument::pageCountChanged, this, [&](int pagesCount) {
        if (m_pdfiumDocument->status() != PdfDocument::Success)
            return;

        if (pagesCount < 0) {
            m_status = DocumentStatus::Error;
            emit statusChanged(m_status);
            return;
        }

        auto pageLoader = new PagesSizesLoader(m_pdfiumDocument);
        connect(pageLoader, &PagesSizesLoader::done, this, [this](QHash<int, QSizeF> sizes) {
            m_pageSizes.swap(sizes);
            m_status = DocumentStatus::Ready;
            emit statusChanged(m_status);
        });
        QThreadPool::globalInstance()->start(pageLoader);
    });

    m_pdfiumDocument->loadDocument(path);

    emit pathChanged(path);
}
