/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "interface/basedocument.h"

#include "notesmodel.h"
#include "pageswithnotesmodel.h"

PagesWithNotesModel::PagesWithNotesModel(QObject *parent) : QAbstractListModel(parent) { }

int PagesWithNotesModel::rowCount(const QModelIndex &parent) const
{
    if (m_documentProvider == nullptr)
        return 0;

    return parent.isValid() ? 0 : m_documentProvider->count();
}

QVariant PagesWithNotesModel::data(const QModelIndex &index, int role) const
{
    if (m_documentProvider == nullptr)
        return {  };

    if (!index.isValid() || index.row() > m_documentProvider->count())
        return {  };

    NotesModel *notesModel = nullptr;
    if (m_notesModels.contains(index.row()) && m_notesModels.value(index.row()) != nullptr)
        notesModel = m_notesModels.value(index.row());

    if (notesModel == nullptr)
        m_documentProvider->loadPage(index.row());

    auto modelLoading = notesModel == nullptr || notesModel->isLoading();

    switch (role) {
    case LoadingRole:
        return QVariant::fromValue(modelLoading ? true : false);
    case NotesModelRole:
        return (modelLoading ? QVariant() : QVariant::fromValue(notesModel));
    case CountRole:
        return QVariant::fromValue(modelLoading ? 0 : notesModel->rowCount(QModelIndex()));
    case PageIndexRole:
        return QVariant::fromValue(index.row());
    }

    return {  };
}

QHash<int, QByteArray> PagesWithNotesModel::roleNames() const
{
    return {
        { LoadingRole, "loading" },
        { NotesModelRole, "notesModel" },
        { CountRole, "notesCount" },
        { PageIndexRole, "pageIndex" }
    };
}

void PagesWithNotesModel::setNewData(BaseDocument *provider)
{
    if (provider == nullptr || provider == m_documentProvider)
        return;

    m_documentProvider = provider;
    connect(m_documentProvider, &BaseDocument::pageLoaded, this, &PagesWithNotesModel::_addPage);
}

void PagesWithNotesModel::_addPage(int pageIndex)
{
    if (m_documentProvider == nullptr)
        return;

    auto page = m_documentProvider->loadPage(pageIndex);
    if (!page)
        return;

    if (m_notesModels.contains(pageIndex) && m_notesModels.value(pageIndex) != nullptr) {
        auto notesModel = m_notesModels.value(pageIndex);
        notesModel->setPageSource(page);
    } else {
        auto notesModel = new NotesModel(this);
        if (notesModel == nullptr)
            return;

        connect(notesModel, &NotesModel::loadingChanged, this, [this, pageIndex](bool loading) {
            Q_UNUSED(loading)
            emit dataChanged(index(pageIndex), index(pageIndex), { LoadingRole });
        });

        notesModel->setPageSource(page);
        m_notesModels.insert(pageIndex, notesModel);
    }

    emit dataChanged(index(pageIndex), index(pageIndex), { LoadingRole });
}
