/****************************************************************************
**
** Copyright (C) 2022 Open Mobile Platform LLC.
** Contact: https://community.omprussia.ru/open-source
**
** This file is part of the AmberPDF-QML-Plugin project.
**
** $QT_BEGIN_LICENSE:BSD$
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Open Mobile Platform LLC copyright holder nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
**
** $QT_END_LICENSE$
**
****************************************************************************/

#include "interface/baseannotation.h"
#include "interface/basepage.h"

#include "notesmodel.h"

NotesModel::NotesModel(QObject *parent) : QAbstractListModel(parent),
    m_loading(true)
{ }

int NotesModel::rowCount(const QModelIndex &parent) const
{
    if (!m_pageSource)
        return 0;

    return parent.isValid() ? 0 : m_notes.size();
}

QVariant NotesModel::data(const QModelIndex &index, int role) const
{
    if (!m_pageSource)
        return {  };

    if (!index.isValid() || index.row() > m_notes.size())
        return {  };

    const auto *note = m_notes.at(index.row());
    if (note == nullptr)
        return {  };

    switch (role) {
    case ContentRole: return QVariant::fromValue(note->content);
    case AuthorRole: return QVariant::fromValue(note->author);
    case ColorRole: return QVariant::fromValue(note->color);
    case IdRole: return QVariant::fromValue(note->annotationId);
    }

    return {  };
}

QHash<int, QByteArray> NotesModel::roleNames() const
{
    return {
        { ContentRole, "content" },
        { AuthorRole, "author" },
        { ColorRole, "color" },
        { IdRole, "id" }
    };
}

void NotesModel::setPageSource(QSharedPointer<BasePage> pageSource)
{
    if (!pageSource || pageSource == m_pageSource)
        return;

    m_pageSource = pageSource;
    connect(m_pageSource.data(), &BasePage::annotationsLoaded, this, &NotesModel::_collectNotes);
    m_pageSource->loadAnnotations();
    m_loading = true;
    emit loadingChanged(m_loading);

    _collectNotes();
}

bool NotesModel::isLoading() const
{
    return m_loading;
}

void NotesModel::_collectNotes()
{
    beginResetModel();

    m_notes.clear();
    const auto annotations = m_pageSource->annotations();
    for (const auto &annotation : annotations) {
        if (annotation == nullptr)
            continue;

        if (annotation->type == BaseAnnotation::AnnotationType::HighLight
                || annotation->type == BaseAnnotation::AnnotationType::Text) {
            m_notes.append(annotation);

            continue;
        }
    }

    endResetModel();
    m_loading = false;
    emit loadingChanged(m_loading);
}

