/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class TypeConversion : BaseCompilationTest() {

    @Test
    fun `check that Kotlin class is converted to Qt class`() {
        val kotlinSource = SourceFile.new(
            "Example.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Example
            
            @QtExport
            fun getExample() = Example()
            """.trimIndent()
        )

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(Path("test/Example.hpp"), "Example getExample();")
        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
            Example getExample()
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto kotlinResult = ns.getExample();
                auto qtResult = Example(kotlinResult);
                return qtResult;
            }
            """.trimIndent()
        )
    }

    @Test
    fun `check that Qt class is converted to Kotlin class`() {
        val compilation = run(
            SourceFile.new(
                "Example.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Example
            
            @QtExport
            fun doSomething(example : Example) {}
            """.trimIndent()
            )
        )

        compilation.assertThatFileHasContent(Path("test/Example.hpp"), "void doSomething(const Example &example);")
        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
            void doSomething(const Example &example)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto kotlinExample = example.unsafeKotlinPointer();
                ns.doSomething(kotlinExample);
            }
            """.trimIndent()
        )
    }
}
