/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class SuspendTests : BaseCompilationTest() {

    @Test
    fun `check that suspend top-level function that returns void is working`() {
        val kotlinSource = SourceFile.new(
            "SuspendTopLevel.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            suspend fun printHelloWorld() {
                println("Hello, World!")
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendTopLevel.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun printHelloWorldAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { printHelloWorld() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendTopLevel.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    auto future = printHelloWorld();
                    auto watcher = new QFutureWatcher<void>();
                    watcher->setFuture(future);
                    
                    QObject::connect(watcher, &QFutureWatcher<void>::finished, [watcher, &application]() {
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        compilation.assertThatFileHasContent(Path("test/SuspendTopLevel.hpp"), "QFuture<void> printHelloWorld();")
        compilation.assertThatFileHasContent(
            Path("test/SuspendTopLevel.cpp"), """
            QFuture<void> printHelloWorld()
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto coroutineLauncher = ns.printHelloWorldAsync();
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<void>((KotlinCoroutineLauncher *) coroutineLauncher);
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStdoutHas("Hello, World!".trimIndent())
    }

    @Test
    fun `check that suspend top-level function that returns non void is correct`() {
        val kotlinSource = SourceFile.new(
            "SuspendTopLevel.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            suspend fun sumAsync(x: Int, y:Int) : Int = x + y
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendTopLevel.kt"), """
            package test
            
            import kotlin.Int
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun sumAsyncAsync(x: Int, y: Int): CPointer<KotlinCoroutineLauncher> = cSuspend { sumAsync(x, y) }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendTopLevel.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    auto future = sumAsync(5, 5);
                    auto watcher = new QFutureWatcher<int>();
                    watcher->setFuture(future);
                    
                    QObject::connect(watcher, &QFutureWatcher<int>::finished, [watcher, &application]() {
                        
                        if (watcher->result() != 10) {
                            application.exit(1);
                        }
                        
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        compilation.assertThatFileHasContent(Path("test/SuspendTopLevel.hpp"), "QFuture<int> sumAsync(int x, int y);")
        compilation.assertThatFileHasContent(
            Path("test/SuspendTopLevel.cpp"), """
            QFuture<int> sumAsync(int x, int y)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto coroutineLauncher = ns.sumAsyncAsync(x, y);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<int>((KotlinCoroutineLauncher *) coroutineLauncher, [=](void *coroutineResult) {
                    return (int) ((long long) coroutineResult);
                });
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that suspend method that returns void is correct`() {
        val kotlinSource = SourceFile.new(
            "SuspendMethod.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Example {
                 suspend fun doSomeWork() : Unit { println("Do work...") }
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendMethod.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun Example.doSomeWorkAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { doSomeWork() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendMethod.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                Example example = Example();
                
                QTimer::singleShot(1000, [&]() {
                    auto future = example.doSomeWork();
                    auto watcher = new QFutureWatcher<void>();
                    watcher->setFuture(future);

                    QObject::connect(watcher, &QFutureWatcher<void>::finished, [watcher, &application]() {
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/SuspendMethod.hpp"), """
            |    QFuture<void> doSomeWork() const;
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/SuspendMethod.cpp"), """
            QFuture<void> Example::doSomeWork() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto coroutineLauncher = ns.doSomeWorkAsync(d_ptr);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<void>((KotlinCoroutineLauncher *) coroutineLauncher);
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStdoutHas("Do work...")
    }

    @Test
    fun `check that suspend method that returns non void is correct`() {
        val kotlinSource = SourceFile.new(
            "SuspendMethod.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            data class User(val id: Int, val name: String)
            
            @QtExport
            class UserRepository {
                 suspend fun getAllUsers() : List<User> = listOf(User(1, "User1"), User(2, "User2"))
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendMethod.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            public fun User.UserCopyNoArgs(): User = copy()
            
            @OptIn(ExperimentalForeignApi::class)
            public fun UserRepository.getAllUsersAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { getAllUsers() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendMethod.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                UserRepository repository = UserRepository();
                
                QTimer::singleShot(1000, [&]() {
                    auto future = repository.getAllUsers();
                    auto watcher = new QFutureWatcher<QList<User>>();
                    watcher->setFuture(future);

                    QObject::connect(watcher, &QFutureWatcher<QList<User>>::finished, [watcher, &application]() {
                        for (auto &user : watcher->result()) {
                            qInfo() << "User" << user.getId() << user.getName();
                        }
                        
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/SuspendMethod.hpp"), """
            |    QFuture<QList<User>> getAllUsers() const;
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/SuspendMethod.cpp"), """
            QFuture<QList<User>> UserRepository::getAllUsers() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto coroutineLauncher = ns.getAllUsersAsync(d_ptr);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<QList<User>>((KotlinCoroutineLauncher *) coroutineLauncher, [=](void *coroutineResult) {
                    return toQListTestUser(libshared_kref_kotlin_collections_List{ coroutineResult });
                });
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStderrHas(
            """
            User 1 "User1"
            User 2 "User2"
        """.trimIndent()
        )
    }

    @Test
    fun `check that suspend extension function is generated`() {
        val kotlinSource1 = SourceFile.new(
            "ExtensibleClass.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class ExtensibleClass
            """.trimIndent()
        )

        val kotlinSource2 = SourceFile.new(
            "SuspendExtensionFunction.kt", """
            package another
    
            import test.ExtensibleClass
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            suspend fun ExtensibleClass.dbConnectionString() = "DB connection string"
            """.trimIndent()
        )

        val generatedKotlinSource =  GeneratedFile(
            Path("another/SuspendExtensionFunction.kt"), """
            package another
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            import test.ExtensibleClass
            
            @OptIn(ExperimentalForeignApi::class)
            public fun ExtensibleClass.dbConnectionStringAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { dbConnectionString() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendExtensionFunction.hpp"
            
            using namespace test;
            using namespace another;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    auto future = dbConnectionString(ExtensibleClass());
                    auto watcher = new QFutureWatcher<QString>();
                    watcher->setFuture(future);

                    QObject::connect(watcher, &QFutureWatcher<QString>::finished, [watcher, &application]() {
                        qInfo() << watcher->result();
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(listOf(kotlinSource1, kotlinSource2), listOf(generatedKotlinSource))

        compilation.assertThatFileHasContent(
            Path("another/SuspendExtensionFunction.hpp"), "QFuture<QString> dbConnectionString(const test::ExtensibleClass &thiz);"
        )

        compilation.assertThatFileHasContent(
            Path("another/SuspendExtensionFunction.cpp"), """
            QFuture<QString> dbConnectionString(const test::ExtensibleClass &thiz)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.another;
                auto kotlinThiz = thiz.unsafeKotlinPointer();
                auto coroutineLauncher = ns.dbConnectionStringAsync(kotlinThiz);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<QString>((KotlinCoroutineLauncher *) coroutineLauncher, [=](void *coroutineResult) {
                    return coroutineResult != nullptr ? QString::fromUtf8((char *) coroutineResult) : QString();
                });
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStderrHas("\"DB connection string\"")
    }

    @Test
    fun `check that suspend method inside the class is generated`() {
        val kotlinSource1 = SourceFile.new(
            "ExtensibleClass.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class ExtensibleClass
            """.trimIndent()
        )

        val kotlinSource2 = SourceFile.new(
            "AnotherClass.kt", """
            package another
    
            import test.ExtensibleClass
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class AnotherClass {
                suspend fun ExtensibleClass.dbConnectionString() = "DB connection string"
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("another/AnotherClass.kt"), """
            package another
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            import test.ExtensibleClass
            
            @OptIn(ExperimentalForeignApi::class)
            public fun AnotherClass.dbConnectionStringAsync(`receiver`: ExtensibleClass): CPointer<KotlinCoroutineLauncher> = cSuspend { `receiver`.dbConnectionString() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "AnotherClass.hpp"
            
            using namespace test;
            using namespace another;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                AnotherClass another = AnotherClass();                
                
                QTimer::singleShot(1000, [&]() {
                    auto future = another.dbConnectionString(ExtensibleClass());
                    auto watcher = new QFutureWatcher<QString>();
                    watcher->setFuture(future);

                    QObject::connect(watcher, &QFutureWatcher<QString>::finished, [watcher, &application]() {
                        qInfo() << watcher->result();
                        application.exit();
                        watcher->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(listOf(kotlinSource1, kotlinSource2), listOf(generatedKotlinSource))

        compilation.assertThatFileHasContent(
            Path("another/AnotherClass.hpp"), """
            class AnotherClass
            {
            public:
                AnotherClass();
                AnotherClass(AnotherClass &&other);
                AnotherClass(libshared_kref_another_AnotherClass ptr);
                ~AnotherClass();
            
                AnotherClass &operator=(AnotherClass &&other);
            
                QFuture<QString> dbConnectionString(const test::ExtensibleClass &thiz) const;
            
                libshared_kref_another_AnotherClass unsafeKotlinPointer() const;
            private:
                libshared_kref_another_AnotherClass d_ptr;
            };
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("another/AnotherClass.cpp"), """
            QFuture<QString> AnotherClass::dbConnectionString(const test::ExtensibleClass &thiz) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.another;
                auto kotlinThiz = thiz.unsafeKotlinPointer();
                auto coroutineLauncher = ns.dbConnectionStringAsync(d_ptr, kotlinThiz);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<QString>((KotlinCoroutineLauncher *) coroutineLauncher, [=](void *coroutineResult) {
                    return coroutineResult != nullptr ? QString::fromUtf8((char *) coroutineResult) : QString();
                });
                return qtFutureResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStderrHas("\"DB connection string\"")
    }

    @Test
    fun `check that suspend function can be blocked`() {
        val kotlinSource = SourceFile.new(
            "SuspendTopLevel.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport

            @QtExport
            suspend fun returnHelloWorld() = "Hello, World!"
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendTopLevel.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun returnHelloWorldAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { returnHelloWorld() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendTopLevel.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    qInfo() << returnHelloWorld().result();
                    application.exit();
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStderrHas("\"Hello, World!\"")
    }

    @Test
    fun `check that suspend function can be cancelled`() {
        val kotlinSource = SourceFile.new(
            "SuspendTopLevel.kt", """
            package test

            import kotlinx.coroutines.delay
            import ru.aurora.kmp.qtbindings.QtExport

            @QtExport
            suspend fun functionWithDelay() {
                delay(1000)
                println("Coroutine is not cancelled")
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendTopLevel.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun functionWithDelayAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { functionWithDelay() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendTopLevel.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    functionWithDelay().cancel();
                });
                
                QTimer::singleShot(4000, [&]() {
                    qInfo() << "Exit";
                    application.exit();
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        assert(qtTestCaseResult.stderr == ("Exit\n"))
    }

    @Test
    fun `check that kotlin exception is reported`() {
        val kotlinSource = SourceFile.new(
            "SuspendTopLevel.kt", """
            package test

            import kotlinx.coroutines.delay
            import ru.aurora.kmp.qtbindings.QtExport

            @QtExport
            suspend fun functionWithException() {
                delay(10)
                throw RuntimeException("Runtime exception!")
            }
            """.trimIndent()
        )

        val generatedKotlinSource = GeneratedFile(
            Path("test", "SuspendTopLevel.kt"), """
            package test
            
            import kotlin.OptIn
            import kotlinx.cinterop.CPointer
            import kotlinx.cinterop.ExperimentalForeignApi
            import ru.aurora.kmp.qtbindings.cSuspend
            import ru.aurora.kmp.qtbindings.cruntime.KotlinCoroutineLauncher
            
            @OptIn(ExperimentalForeignApi::class)
            public fun functionWithExceptionAsync(): CPointer<KotlinCoroutineLauncher> = cSuspend { functionWithException() }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QTimer>
            
            #include "SuspendTopLevel.hpp"
            #include "CoroutineException.hpp"
            
            using namespace test;
            using namespace Aurora::Kmp::QtBindings;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                
                QTimer::singleShot(1000, [&]() {
                    auto future = functionWithException();
                    auto watcher = new QFutureWatcher<void>();
                    watcher->setFuture(future);

                    QObject::connect(watcher, &QFutureWatcher<void>::finished, [watcher, &application]() {
                        QFuture<void> future = watcher->future();

                        try {
                            future.waitForFinished();
                        } catch (const CoroutineException& e) {
                            qInfo() << e.message();
                        }

                        watcher->deleteLater();
                        application.exit();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource, generatedKotlinSource)

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStderrHas("\"kotlin.RuntimeException: Runtime exception!\"")
    }
}
