/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class OverloadingTests : BaseCompilationTest() {

    @Test
    fun `check that class method overloading is supported`() {
        val kotlinSource = SourceFile.new(
            "Test.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Test {
                fun overloaded(x: Boolean, y: String) { println("overloaded1") }
                fun overloaded(x: String) { println("overloaded2") }
                fun overloaded(x: Int) { println("overloaded3") }
                fun overloaded() { println("overloaded4") }
                fun overloaded(x: Short) { println("overloaded5") }
                fun overloaded(x: String, y: String) { println("overloaded6") }
                fun overloaded(x: Int, y: Int, z: Int) { println("overloaded7") }
            }
            """.trimIndent()
        )

        val qtTestCase = """
            #include "Test.hpp"
            
            using namespace test;
            
            int main() {
                Test test = Test();
                
                test.overloaded(true, "");
                test.overloaded("");
                test.overloaded(1);
                test.overloaded();
                test.overloaded((short) 1);
                test.overloaded(QString(""), ""); // Add QString to avoid casting (char *) to (bool)
                test.overloaded(1, 2, 3);                

                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Test.hpp"), """
            |    void overloaded(bool x, const QString &y) const;
            |    void overloaded(const QString &x) const;
            |    void overloaded(int x) const;
            |    void overloaded() const;
            |    void overloaded(short x) const;
            |    void overloaded(const QString &x, const QString &y) const;
            |    void overloaded(int x, int y, int z) const;
            """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Test.cpp"), """
            void Test::overloaded(bool x, const QString &y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                auto byteArrayY = y.toUtf8();
                auto kotlinY = byteArrayY.data();
                ns.overloaded(d_ptr, x, kotlinY);
            }
            """.trimIndent(), """
            void Test::overloaded(const QString &x) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                auto byteArrayX = x.toUtf8();
                auto kotlinX = byteArrayX.data();
                ns.overloaded_(d_ptr, kotlinX);
            }
            """.trimIndent(), """
            void Test::overloaded(int x) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                ns.overloaded__(d_ptr, x);
            }
            """.trimIndent(), """
            void Test::overloaded() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                ns.overloaded___(d_ptr);
            }
            """.trimIndent(), """
            void Test::overloaded(short x) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                ns.overloaded____(d_ptr, x);
            }
            """.trimIndent(), """
            void Test::overloaded(const QString &x, const QString &y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                auto byteArrayX = x.toUtf8();
                auto kotlinX = byteArrayX.data();
                auto byteArrayY = y.toUtf8();
                auto kotlinY = byteArrayY.data();
                ns.overloaded_____(d_ptr, kotlinX, kotlinY);
            }
            """.trimIndent(), """
            void Test::overloaded(int x, int y, int z) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                ns.overloaded______(d_ptr, x, y, z);
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStdoutHas(
            """
            overloaded1
            overloaded2
            overloaded3
            overloaded4
            overloaded5
            overloaded6
            overloaded7
        """.trimIndent()
        )
    }

    @Test
    fun `check that top level function overloading is supported`() {
        val kotlinSource1 = SourceFile.new(
            "example.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun overloaded() { println("overloaded1") }
            """.trimIndent()
        )

        val kotlinSource2 = SourceFile.new(
            "Example.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun overloaded(x: Int) { println("overloaded2") }
            """.trimIndent()
        )

        val kotlinSource3 = SourceFile.new(
            "Example2.kt", """
            package test
    
            fun overloaded(x: String) { println("overloaded3") }
            """.trimIndent()
        )

        val kotlinSource4 = SourceFile.new(
            "Example3.kt", """
            package test
            import ru.aurora.kmp.qtbindings.QtExport

            @QtExport
            fun overloaded(x: Boolean, y: String) { println("overloaded4") }
            """.trimIndent()
        )

        val qtTestCase = """
            #include "example.hpp"
            #include "Example.hpp"
            #include "Example3.hpp"
            
            using namespace test;
            
            int main() {
                overloaded();
                overloaded(1);
                overloaded(true, "");
                return 0;
            }
        """.trimIndent()

        val compilation = run(listOf(kotlinSource1, kotlinSource2, kotlinSource3, kotlinSource4))

        compilation.assertThatFileHasContent(
            Path("test/example.hpp"), """
            void overloaded();
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/example.cpp"), """
            void overloaded()
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                ns.overloaded___();
            }
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example.hpp"), """
            void overloaded(int x);
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
            void overloaded(int x)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                ns.overloaded__(x);
            }
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example3.hpp"), """
            void overloaded(bool x, const QString &y);
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example3.cpp"), """
            void overloaded(bool x, const QString &y)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto byteArrayY = y.toUtf8();
                auto kotlinY = byteArrayY.data();
                ns.overloaded(x, kotlinY);
            }
            """.trimIndent()
        )

        // TODO: find a way for testing compilation (result depends on file order now)
        // val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        // qtTestCaseResult.assertIsOk()
        // qtTestCaseResult.assertStdoutHas("""
        //    overloaded1
        //    overloaded2
        //    overloaded4
        // """.trimIndent())
    }

    @Test
    fun `check that suspend method overloading is supported`() {
        val kotlinSource = SourceFile.new(
            "Example.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport

            @QtExport
            class Test {
                suspend fun asyncData(x : String) { println("overloaded1") }
                suspend fun asyncData(x : Int) { println("overloaded2") }
            }
            """.trimIndent()
        )

        val qtTestCase = """
            #include <QFutureWatcher>
            #include <QCoreApplication>
            #include <QThread>
            #include <QTimer>
            
            #include "Example.hpp"
            
            using namespace test;
            
            int main(int argc, char *argv[]) {
                auto application = QCoreApplication(argc, argv);
                Test test = Test();
                
                QTimer::singleShot(1000, [&]() {
                    auto future1 = test.asyncData("");
                    auto watcher1 = new QFutureWatcher<void>();
                    watcher1->setFuture(future1);
                    
                    QObject::connect(watcher1, &QFutureWatcher<void>::finished, [watcher1]() {
                        watcher1->deleteLater();
                    });
                    
                    QThread::sleep(1); // Wait for the first async call completion
                    
                    auto future2 = test.asyncData(1);
                    auto watcher2 = new QFutureWatcher<void>();
                    watcher2->setFuture(future2);
                    
                    QObject::connect(watcher2, &QFutureWatcher<void>::finished, [watcher2, &application]() {
                        application.exit();
                        watcher2->deleteLater();
                    });
                });
                
                return application.exec();
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Example.hpp"), """
            |    QFuture<void> asyncData(const QString &x) const;
            |    QFuture<void> asyncData(int x) const;
            """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
            QFuture<void> Test::asyncData(const QString &x) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto byteArrayX = x.toUtf8();
                auto kotlinX = byteArrayX.data();
                auto coroutineLauncher = ns.asyncDataAsync(d_ptr, kotlinX);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<void>((KotlinCoroutineLauncher *) coroutineLauncher);
                return qtFutureResult;
            }
            """.trimIndent(), """
            QFuture<void> Test::asyncData(int x) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                auto coroutineLauncher = ns.asyncDataAsync_(d_ptr, x);
                auto qtFutureResult = Aurora::Kmp::QtBindings::coroutine<void>((KotlinCoroutineLauncher *) coroutineLauncher);
                return qtFutureResult;
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStdoutHas("""
            overloaded1
            overloaded2
        """.trimIndent())
    }
}
