/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import com.google.devtools.ksp.processing.Resolver
import ru.aurora.kmp.qtbindings.ksp.export.*
import ru.aurora.kmp.qtbindings.ksp.gen.CLibraryAPI
import ru.aurora.kmp.qtbindings.ksp.gen.TypeConverter
import ru.aurora.kmp.qtbindings.ksp.model.*

private data class ResolutionContext(val spec: ExportSpec, val currentFile: ExportedFile)

private data class ResolveData(
    val headerIncludes: Set<QtInclude> = setOf(),
    val sourceIncludes: Set<QtInclude> = setOf(),
    val typeConverters: LinkedHashSet<TypeConverterDescriptor> = linkedSetOf(),
) {
    operator fun plus(other: ResolveData): ResolveData {
        val typeConverters = LinkedHashSet<TypeConverterDescriptor>(this.typeConverters)
        typeConverters.addAll(other.typeConverters)

        return ResolveData(
            this.headerIncludes + other.headerIncludes, this.sourceIncludes + other.sourceIncludes, typeConverters
        )
    }

    operator fun plus(typeConverters: LinkedHashSet<TypeConverterDescriptor>): ResolveData {
        val newTypeConverters = LinkedHashSet<TypeConverterDescriptor>(this.typeConverters)
        newTypeConverters.addAll(typeConverters)
        return this.copy(typeConverters = newTypeConverters)
    }
}

internal class QtResolver(private val cApi: CLibraryAPI, private val kspResolver: Resolver) {

    private lateinit var scopeManager: KotlinScopeManager

    fun resolve(exportSpec: ExportSpec): List<QtFile> {
        scopeManager = KotlinScopeManager(kspResolver, exportSpec)
        return exportSpec.userFiles.map {
            val context = ResolutionContext(exportSpec, it)
            it.resolve(context)
        }
    }

    private fun ExportedFile.resolve(context: ResolutionContext): QtFile {
        val (qtClasses, data1) = classes.map { it.resolve(context) }.unzip()
        val (qtFunctions, data2) = functions.map { it.resolve(context) }.unzip()

        val data = data1.sum() + data2.sum()
        val ns = QtNamespace(filePackage.components)

        val qtFile = QtFile(
            fileName,
            dependencyFilePath,
            ns,
            data.headerIncludes + QtInclude.cKotlinApi(cApi.libName),
            data.sourceIncludes + QtInclude.cppLocalHeader(fileName),
            data.typeConverters,
            qtClasses,
            qtFunctions,
        )

        return qtFile
    }

    private fun ExportedFunction.resolve(context: ResolutionContext): Pair<QtFunction, ResolveData> {
        val (returnType, data1) = returnType.resolve(context)
        val (parameters, data2) = parameters.map { it.resolve(context) }.unzip()

        val asyncInclude = if (isAsync) ResolveData(
            sourceIncludes = setOf(QtInclude.CoroutineOperationHeader), headerIncludes = setOf(QtInclude.QFuture)
        ) else ResolveData()
        val data = data1 + data2.sum() + returnType.typeConverter(ConversionType.KotlinToQt) + asyncInclude

        val scope = scopeManager.getScope(this)
        val cName = scope.cName(this)
        val ns = QtNamespace(context.currentFile.filePackage.components)
        val qtFunction = QtFunction(originalKotlinName, cName, isAsync, false, ns, returnType, parameters)

        return Pair(qtFunction, data)
    }

    private fun ExportedClass.resolve(context: ResolutionContext): Pair<QtClass, ResolveData> {
        val (props, data1) = properties.map { it.resolver(context, name) }.unzip()
        val (methods, data2) = methods.map { it.resolve(context, name) }.unzip()

        val ns = QtNamespace(containingFile.filePackage.components)
        val qtClass = QtClass(name, ns, isDataClass, props + methods)

        val data = data1.sum() + data2.sum()

        return Pair(qtClass, data)
    }

    private fun ExportedMethod.resolve(context: ResolutionContext, parentClassName: String) =
        resolve(context).map { func, data ->
            val method = QtMethod(
                // <init> for constructor in Kotlin, need to update for Qt
                if (isConstructor) parentClassName else func.name,
                func.cName,
                func.isAsync,
                if (isConstructor) false else true, // constructors no no
                func.ns,
                parentClassName,
                isConstructor,
                func.returnType,
                func.parameters
            )
            Pair(method, data)
        }

    private fun ExportedProperty.resolver(
        context: ResolutionContext, parentClassName: String
    ): Pair<QtMethod, ResolveData> {
        // TODO: unite logic for finding all type converters, it done in Method and Property now
        val (type, data) = type.resolve(context)
        val scope = scopeManager.getScope(this)
        val cName = scope.cName(this)
        val capitalizedName = name.capitalizeFirstChar()
        val ns = QtNamespace(context.currentFile.filePackage.components)
        val (method, converters) = when (accessor) {
            AccessorType.Getter -> Pair(
                QtMethod(
                    "get$capitalizedName",
                    cName,
                    false,
                    isConst = true,
                    ns = ns,
                    parentClassName = parentClassName,
                    isConstructor = false,
                    returnType = type,
                    parameters = emptyList(),
                ), type.typeConverter(ConversionType.KotlinToQt)
            )

            AccessorType.Setter -> Pair(
                QtMethod(
                    "set$capitalizedName",
                    cName,
                    false,
                    isConst = false,
                    ns = ns,
                    parentClassName = parentClassName,
                    isConstructor = false,
                    returnType = QtType.Primitive.Void,
                    parameters = listOf(
                        QtParameter("set", type)
                    ),
                ), type.typeConverter(ConversionType.KotlinToQt)
            )
        }

        return Pair(method, data + converters)
    }

    private fun ExportedParameter.resolve(context: ResolutionContext): Pair<QtParameter, ResolveData> {
        val (type, data) = type.resolve(context)
        return Pair(QtParameter(name, type), data + type.typeConverter(ConversionType.QtToKotlin))
    }

    private fun ExportedType.resolve(context: ResolutionContext): Pair<QtType, ResolveData> {
        return when (this) {
            is ExportedType.Unit -> Pair(QtType.Primitive.Void, ResolveData())
            is ExportedType.Bool -> Pair(QtType.Primitive.Bool, ResolveData())
            is ExportedType.Char -> Pair(QtType.Primitive.Char, ResolveData())
            is ExportedType.Byte -> Pair(QtType.Primitive.SignedChar, ResolveData())
            is ExportedType.Short -> Pair(QtType.Primitive.Short, ResolveData())
            is ExportedType.Int -> Pair(QtType.Primitive.Int, ResolveData())
            is ExportedType.Long -> Pair(QtType.Primitive.Long, ResolveData())
            is ExportedType.UByte -> Pair(QtType.Primitive.UnsignedChar, ResolveData())
            is ExportedType.UShort -> Pair(QtType.Primitive.UnsignedShort, ResolveData())
            is ExportedType.UInt -> Pair(QtType.Primitive.UnsignedInt, ResolveData())
            is ExportedType.ULong -> Pair(QtType.Primitive.UnsignedLong, ResolveData())
            is ExportedType.Float -> Pair(QtType.Primitive.Float, ResolveData())
            is ExportedType.Double -> Pair(QtType.Primitive.Double, ResolveData())
            is ExportedType.String -> Pair(QtType.Primitive.QString, ResolveData(setOf(QtInclude.QString)))

            is ExportedType.List -> {
                val (type, data) = elementType.resolve(context)
                val listType = QtType.Collection.QList(type, isMutable)
                val finalData = data + ResolveData(setOf(QtInclude.QList))
                Pair(listType, finalData)
            }

            is ExportedType.Class -> {
                val currentTypeFile = classDeclaration.containingFile
                val isLocalType = currentTypeFile == context.currentFile
                val inTheSameNamespace = currentTypeFile.filePackage == context.currentFile.filePackage

                val name = classDeclaration.name
                val ns = QtNamespace(classDeclaration.containingFile.filePackage.components)
                val type = QtType.Class(name, ns, !inTheSameNamespace)

                val includes = if (!isLocalType) setOf(QtInclude.cppLocalHeader(currentTypeFile.fileName)) else setOf()
                val resolveData = ResolveData(headerIncludes = includes)
                Pair(type, resolveData)
            }
        }
    }

    private fun QtType.typeConverter(conversionType: ConversionType): LinkedHashSet<TypeConverterDescriptor> {
        return when (this) {
            is QtType.Collection.QList -> {
                val fqElementType = elementType.withFq()
                val set = elementType.typeConverter(conversionType)
                set.add(TypeConverterDescriptor.List(fqElementType, isMutable, conversionType))
                return set
            }

            else -> LinkedHashSet()
        }
    }
}

private fun <F, S, R> Pair<F, S>.map(transform: (F, S) -> R) = transform(first, second)

private fun QtType.Collection.QList.finalElementType(): QtType {
    return when (elementType) {
        is QtType.Collection.QList -> elementType.finalElementType()
        else -> elementType
    }
}

private fun List<ResolveData>.sum(): ResolveData = fold(ResolveData()) { acc, el -> acc + el }
