/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp.internal

import java.io.File

class QtCompilationResult(
    exitCode: Int,
    stdout: String,
    stderr: String,
    executable: File,
) : ProcessResult(executable, exitCode, stdout, stderr) {
    override fun assertIsOk() = assert(isOk) { "Failed to compile ${executable.absolutePath}: $stdout" }
}

class QtCompiler(val workingDir: File) {

    companion object {
        private const val TEST_EXECUTABLE_NAME = "qtbindings-test"
        private const val KOTLIN_INCLUDE_DIRS_TEMPLATE = "\$KOTLIN_INCLUDE_DIRS"
        private const val KOTLIN_LINK_ARGS_TEMPLATE = "\$KOTLIN_LINK_ARGS"
        private const val QT_BINDINGS_INCLUDE_DIRS_TEMPLATE = "\$QT_BINDINGS_INCLUDE_DIRS"
        private const val QT_BINDINGS_HEADERS_TEMPLATE = "\$QT_BINDINGS_HEADERS"
        private const val QT_BINDINGS_SOURCES_TEMPLATE = "\$QT_BINDINGS_SOURCES"
    }

    private val mesonTemplate = """
        project('$TEST_EXECUTABLE_NAME', ['cpp'], version: '0.0.1')
        
        qt5 = import('qt5')
        
        kotlin_library_dep = declare_dependency(
            include_directories: [$KOTLIN_INCLUDE_DIRS_TEMPLATE],
            link_args: [$KOTLIN_LINK_ARGS_TEMPLATE],
        )
        
        deps = [
            dependency('qt5', modules: ['Core', 'Concurrent']),
            kotlin_library_dep,
        ]
        
        inc = include_directories(${QT_BINDINGS_INCLUDE_DIRS_TEMPLATE})
        
        headers = [$QT_BINDINGS_HEADERS_TEMPLATE]
        sources = [$QT_BINDINGS_SOURCES_TEMPLATE]

        mocs = qt5.compile_moc(sources: sources, headers: headers)        
        
        executable(
            '$TEST_EXECUTABLE_NAME',
            [sources, mocs],
            dependencies: deps,
            include_directories: inc,
        )
    """.trimIndent()


    fun compileExecutable(headers: List<File>, sources: List<File>, cLibrary: File, cHeader: File): QtCompilationResult {
        val mesonFile = createMesonFile(headers, sources, cLibrary, cHeader)
        val executableFile = File(workingDir, "build/$TEST_EXECUTABLE_NAME")

        val setupMesonCommand = "meson setup build".split(' ')
        val compileMesonCommand = "meson compile -C build".split(' ')

        val setupMesonProcess = ProcessBuilder(setupMesonCommand).directory(mesonFile.parentFile).start()
        val setupExitCode = setupMesonProcess.waitFor()
        val setupStdout = setupMesonProcess.inputStream.bufferedReader().readText()
        val setupStderr = setupMesonProcess.errorStream.bufferedReader().readText()

        if (setupExitCode != 0) return QtCompilationResult(setupExitCode, setupStdout, setupStderr, executableFile)

        val compileMesonProcess = ProcessBuilder(compileMesonCommand).directory(mesonFile.parentFile).start()
        val compileExitCode = compileMesonProcess.waitFor()
        val compileStdout = compileMesonProcess.inputStream.bufferedReader().readText()
        val compileStderr = compileMesonProcess.errorStream.bufferedReader().readText()

        return QtCompilationResult(compileExitCode, compileStdout, compileStderr, executableFile)
    }

    private fun getRelativePaths(files: List<File>): List<String> =
        files.map { it.relativeTo(workingDir).path }.toSet().toList().distinct()

    private fun getRelativeIncludeDirectories(headers: List<File>): List<String> =
        getRelativePaths(headers.map { it.parentFile })

    private fun getRelativeIncludeDirectories(vararg headers: File) = getRelativeIncludeDirectories(headers.toList())

    private fun createMesonFile(headers: List<File>, sources: List<File>, cLibrary: File, cHeader: File): File {
        val kotlinIncDirs = getRelativeIncludeDirectories(cHeader)
        val kotlinLinkArgs = listOf("-L${cLibrary.parent}", "-l${cLibrary.name.removePrefix("lib").removeSuffix(".a")}")
        val qtBindingsIncludeDirs = getRelativeIncludeDirectories(headers)
        val qtBindingsHeaders = getRelativePaths(headers)
        val qtBindingsSources = getRelativePaths(sources)

        val mesonContent = mesonTemplate.replace(KOTLIN_INCLUDE_DIRS_TEMPLATE, kotlinIncDirs.joinToString { "'$it'" })
            .replace(KOTLIN_LINK_ARGS_TEMPLATE, kotlinLinkArgs.joinToString { "'$it'" })
            .replace(QT_BINDINGS_INCLUDE_DIRS_TEMPLATE, qtBindingsIncludeDirs.joinToString { "'$it'" })
            .replace(QT_BINDINGS_HEADERS_TEMPLATE, qtBindingsHeaders.joinToString { "'$it'" })
            .replace(QT_BINDINGS_SOURCES_TEMPLATE, qtBindingsSources.joinToString { "'$it'" })
        val mesonFile = File(workingDir, "meson.build")
        mesonFile.writeText(mesonContent)
        return mesonFile
    }
}
