/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class ClassTests : BaseCompilationTest() {

    @Test
    fun `check that all base constructors, operators and destructor are generated`() {
        val kotlinSource = SourceFile.new(
            "Empty.kt", """
            package test
    
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Empty
            """.trimIndent()
        )

        val qtHeader = GeneratedFile(
            Path("test/Empty.hpp"), """
            // Generated code, DO NOT EDIT IT MANUALLY!
            
            #ifndef TEST_EMPTY_HPP
            #define TEST_EMPTY_HPP
            
            #include "libshared_api.h"
            
            namespace test {
            
            class Empty;
            
            class Empty
            {
            public:
                Empty();
                Empty(Empty &&other);
                Empty(libshared_kref_test_Empty ptr);
                ~Empty();
            
                Empty &operator=(Empty &&other);
            
                libshared_kref_test_Empty unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Empty d_ptr;
            };
            
            } /* namespace test */
            
            #endif /* TEST_EMPTY_HPP */
            """.trimIndent()
        )

        val qtSource = GeneratedFile(
            Path("test/Empty.cpp"), """
            // Generated code, DO NOT EDIT IT MANUALLY!
            
            #include "Empty.hpp"
            
            namespace test {
            
            Empty::Empty()
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Empty;
                d_ptr = ns.Empty();
            }
            
            Empty::Empty(Empty &&other)
            {
                d_ptr.pinned = other.d_ptr.pinned;
                other.d_ptr.pinned = nullptr;
            }
            
            Empty::Empty(libshared_kref_test_Empty ptr)
                : d_ptr(ptr)
            {}
            
            Empty::~Empty()
            {
                auto s = libshared_symbols();
                s->DisposeStablePointer(d_ptr.pinned);
            }
            
            Empty &Empty::operator=(Empty &&other)
            {
                if (this != &other) {
                    this->d_ptr = other.d_ptr;
                    other.d_ptr.pinned = nullptr;
                    return *this;
                }
                return *this;
            }
            
            libshared_kref_test_Empty Empty::unsafeKotlinPointer() const
            {
                return d_ptr;
            }
            
            } /* namespace test */
            """.trimIndent()
        )

        val qtTestCase = """
            #include "Empty.hpp"
            #include <utility>
            
            using namespace test;
            
            int main() {
                {
                    Empty empty = Empty(); // Check constructor
                    Empty empty2 = std::move(empty); // Check move operator
                    
                    if (empty.unsafeKotlinPointer().pinned != nullptr) {
                        return 1;
                    }
                    
                    Empty empty3 = Empty(std::move(empty2)); // Check move constructor
                    
                    if (empty.unsafeKotlinPointer().pinned != nullptr) {
                        return 1;
                    }
                    
                    // Check destructor
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource, qtHeader, qtSource)
        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that Qt method arguments passed to the C method and result is returned`() {
        val kotlinSource = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Test() {
                fun sum(x : Int, y: Int, z: Int) : Int { return x + y + z }
            }
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Test.hpp"
            
            using namespace test;
            
            int main() {
                Test test = Test();
                int sumResult = test.sum(1, 2, 3);
               
                if (sumResult != 6) {
                    return 1;
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Test.cpp"), """
            int Test::sum(int x, int y, int z) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                auto kotlinResult = ns.sum(d_ptr, x, y, z);
                return kotlinResult;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that Unit method returns void`() {
        val kotlinSource = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Test() {
                fun printHelloWorld() : Unit { println("Hello, world!") }
            }
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Test.hpp"
            
            using namespace test;
            
            int main() {
                Test test = Test();
                test.printHelloWorld();
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Test.cpp"), """
            void Test::printHelloWorld() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Test;
                ns.printHelloWorld(d_ptr);
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
        qtTestCaseResult.assertStdoutHas("Hello, world!")
    }

    @Test
    fun `check that data class has copy function, copy constructor, copy operator and property method`() {
        val kotlinSource = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            data class Test(val someStr : String)
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Test.hpp"
            
            using namespace test;
            
            int main() {
                {
                    Test test1 = Test("First"); // Check constructor
                    Test test2 = test1.copy("Second"); // Check copy function
                    
                    if (test1.unsafeKotlinPointer().pinned == test2.unsafeKotlinPointer().pinned) {
                        return 1;
                    }
                    
                    Test test3 = test2; // Check copy operator
                    
                    if (test2.unsafeKotlinPointer().pinned == test3.unsafeKotlinPointer().pinned) {
                        return 2;
                    }

                    Test test4 = Test(test3); // Check copy constructor
                    
                    if (test3.unsafeKotlinPointer().pinned == test4.unsafeKotlinPointer().pinned) {
                        return 3;
                    }
                    
                    if (test1.getSomeStr() != "First") {
                        return 4;
                    }
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Test.kt"), """
            public fun Test.TestCopyNoArgs(): Test = copy()
            """.trimIndent()
        )

        compilation.assertThatFileHasContent(
            Path("test/Test.hpp"), """
        |    Test copy(const QString &someStr) const;
        """.trimMargin(), """
        |    Test(const Test &other);
        """.trimMargin(), """
        |    Test &operator=(const Test &other);
        """.trimMargin(), """
        |    QString getSomeStr() const;
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Test.cpp"), """
            Test::Test(const Test &other)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test;
                d_ptr = ns.TestCopyNoArgs(other.d_ptr);
            }
        """.trimIndent(), """
            Test &Test::operator=(const Test &other)
            {
                if (this != &other) {
                    auto s = libshared_symbols();
                    auto ns = s->kotlin.root.test;
                    this->d_ptr = ns.TestCopyNoArgs(other.d_ptr);
                    return *this;
                }
                return *this;
            }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that public methods are exported`() {
        val kotlinSource = SourceFile.new(
            "Calculator.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Calculator {
                fun sum(x: Int, y: Int) : Int = x + y
                fun subtract(x: Int, y: Int) : Int = x - y
                fun multiply(x: Int, y: Int) : Int = x * y
                fun divide(x: Int, y: Int) : Int = x / y
            }
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Calculator.hpp"
            
            using namespace test;
            
            int main() {
                Calculator calculator = Calculator();
                
                if (calculator.sum(2, 2) != 4) {
                    return 1;
                }
                
                if (calculator.subtract(2, 2) != 0) {
                    return 2;
                }

                if (calculator.multiply(2, 3) != 6) {
                    return 3;
                }

                if (calculator.divide(2, 2) != 1) {
                    return 4;
                }

                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Calculator.hpp"), """
        |    int sum(int x, int y) const;
        """.trimMargin(), """
        |    int subtract(int x, int y) const;
        """.trimMargin(), """
        |    int multiply(int x, int y) const;
        """.trimMargin(), """
        |    int divide(int x, int y) const;
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Calculator.cpp"), """
            int Calculator::sum(int x, int y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Calculator;
                auto kotlinResult = ns.sum(d_ptr, x, y);
                return kotlinResult;
            }
            """.trimIndent(), """
            int Calculator::subtract(int x, int y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Calculator;
                auto kotlinResult = ns.subtract(d_ptr, x, y);
                return kotlinResult;
            }
            """.trimIndent(), """
            int Calculator::multiply(int x, int y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Calculator;
                auto kotlinResult = ns.multiply(d_ptr, x, y);
                return kotlinResult;
            }
            """.trimIndent(), """
            int Calculator::divide(int x, int y) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Calculator;
                auto kotlinResult = ns.divide(d_ptr, x, y);
                return kotlinResult;
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that non public methods are not exported`() {
        val source = SourceFile.new(
            "Calculator.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Calculator {
                internal fun sum(x: Int, y: Int) : Int = x + y
                protected fun subtract(x: Int, y: Int) : Int = x - y
                private fun multiply(x: Int, y: Int) : Int = x * y
                private fun divide(x: Int, y: Int) : Int = x / y
            }
        """.trimIndent()
        )

        val compilation = run(source)

        compilation.assertThatFileHasContent(
            Path("test/Calculator.hpp"), """
            class Calculator
            {
            public:
                Calculator();
                Calculator(Calculator &&other);
                Calculator(libshared_kref_test_Calculator ptr);
                ~Calculator();

                Calculator &operator=(Calculator &&other);

                libshared_kref_test_Calculator unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Calculator d_ptr;
            };
        """.trimIndent()
        )
    }

    @Test
    fun `check that public read-only properties are exported`() {
        val kotlinSource = SourceFile.new(
            "Example.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class Example(val str: String, val integer: Int, val boolean: Boolean)
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Example.hpp"
            
            using namespace test;
            
            int main() {
                Example example = Example("Hello!", 111, true);
               
                if (example.getStr() != "Hello!") {
                    return 1;
                }

                if (example.getInteger() != 111) {
                    return 2;
                }

                if (example.getBoolean() != true) {
                    return 3;
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Example.hpp"), """
            |    QString getStr() const;
            |    int getInteger() const;
            |    bool getBoolean() const;
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
            QString Example::getStr() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Example;
                auto kotlinResult = ns.get_str(d_ptr);
                auto qtResult = kotlinResult != nullptr ? QString::fromUtf8((char *) kotlinResult) : QString();
                if (kotlinResult != nullptr) {
                    s->DisposeString(kotlinResult);
                }
                return qtResult;
            }
            """.trimIndent(), """
            int Example::getInteger() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Example;
                auto kotlinResult = ns.get_integer(d_ptr);
                return kotlinResult;
            }
            """.trimIndent(), """
            bool Example::getBoolean() const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.test.Example;
                auto kotlinResult = ns.get_boolean(d_ptr);
                return kotlinResult;
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that public read-write properties are exported`() {
        val kotlinSource = SourceFile.new(
            "Example.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            data class Example(var str: String)
        """.trimIndent()
        )

        val qtTestCase = """
            #include "Example.hpp"
            
            using namespace test;
            
            int main() {
                Example example = Example("First");
                
                if (example.getStr() != "First") {
                    return 1;
                }
                
                example.setStr("Second");
                
                if (example.getStr() != "Second") {
                    return 2;
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(kotlinSource)

        compilation.assertThatFileHasContent(
            Path("test/Example.hpp"), """
            |    QString getStr() const;
            |    void setStr(const QString &set);
        """.trimMargin()
        )

        compilation.assertThatFileHasContent(
            Path("test/Example.cpp"), """
        QString Example::getStr() const
        {
            auto s = libshared_symbols();
            auto ns = s->kotlin.root.test.Example;
            auto kotlinResult = ns.get_str(d_ptr);
            auto qtResult = kotlinResult != nullptr ? QString::fromUtf8((char *) kotlinResult) : QString();
            if (kotlinResult != nullptr) {
                s->DisposeString(kotlinResult);
            }
            return qtResult;
        }
        """.trimIndent(), """
        void Example::setStr(const QString &set)
        {
            auto s = libshared_symbols();
            auto ns = s->kotlin.root.test.Example;
            auto byteArraySet = set.toUtf8();
            auto kotlinSet = byteArraySet.data();
            ns.set_str(d_ptr, kotlinSet);
        }
        """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that non-public properties are not exported`() {
        val source = SourceFile.new(
            "Example.kt", """
            package test

            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            data class Example(
                private var str1: String,
                private val str2: String,
                internal var int1: Int,
                internal val int2: Int,
            )
        """.trimIndent()
        )

        val compilation = run(source)

        compilation.assertThatFileHasContent(
            Path("test/Example.hpp"), """
            class Example
            {
            public:
                Example(const QString &str1, const QString &str2, int int1, int int2);
                Example(const Example &other);
                Example(Example &&other);
                Example(libshared_kref_test_Example ptr);
                ~Example();
            
                Example &operator=(const Example &other);
                Example &operator=(Example &&other);
            
                QString component1() const;
                QString component2() const;
                int component3() const;
                int component4() const;
                Example copy(const QString &str1, const QString &str2, int int1, int int2) const;
            
                libshared_kref_test_Example unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Example d_ptr;
            };
        """.trimIndent()
        )
    }
}
