/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class GeneralTests : BaseCompilationTest() {

    @Test
    fun `check that header guard is generated correctly`() {
        val source = SourceFile.new(
            "Test.kt", """
            package check.that.headerguard.correct
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun test() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("check/that/headerguard/correct/Test.hpp"), """
                #ifndef CHECK_THAT_HEADERGUARD_CORRECT_TEST_HPP
                #define CHECK_THAT_HEADERGUARD_CORRECT_TEST_HPP
            """.trimIndent(), "#endif /* CHECK_THAT_HEADERGUARD_CORRECT_TEST_HPP */"
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that prohibited symbols for header guard is replaced`() {
        val strangeFileNameWithoutExtension = "Test!@#$%^&*()_..._t"

        val source = SourceFile.new(
            "$strangeFileNameWithoutExtension.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun test() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/$strangeFileNameWithoutExtension.hpp"),
            """
                #ifndef TEST_TEST_______________T_HPP
                #define TEST_TEST_______________T_HPP
            """.trimIndent(),
            "#endif /* TEST_TEST_______________T_HPP */",
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that namespace is generated`() {
        val source = SourceFile.new(
            "Test.kt", """
            package ru.auroraos.example
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun test() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("ru/auroraos/example/Test.hpp"), """
            namespace ru {
            namespace auroraos {
            namespace example {
            """.trimIndent(), """
            } /* namespace example */
            } /* namespace auroraos */
            } /* namespace ru */
            """.trimIndent()
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that header and source files have generated code warning`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun test() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/Test.hpp"), "// Generated code, DO NOT EDIT IT MANUALLY!"
        )
        compilation.assertThatFileHasContent(
            Path("test/Test.cpp"), "// Generated code, DO NOT EDIT IT MANUALLY!"
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that a forward declaration has been generated for a function that uses a class defined in the same file`() {
        val source = SourceFile.new(
            "ForwardDeclaration.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun getExample() = Example(1)
            
            @QtExport
            data class Example(val x : Int)
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/ForwardDeclaration.hpp"), """
            class Example;

            Example getExample();
            
            class Example
            {
            public:
                Example(int x);
                Example(const Example &other);
                Example(Example &&other);
                Example(libshared_kref_test_Example ptr);
                ~Example();
            
                Example &operator=(const Example &other);
                Example &operator=(Example &&other);
            
                int getX() const;
                int component1() const;
                Example copy(int x) const;
            
                libshared_kref_test_Example unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Example d_ptr;
            };
            """.trimIndent()
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that a forward declaration has been generated for a class that uses a class defined in the same file`() {
        val source = SourceFile.new(
            "ForwardDeclaration.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            class ExampleManager {
                fun getExamples() = emptyList<Example>()
            }
            
            @QtExport
            data class Example(val x: Int)
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/ForwardDeclaration.hpp"),
            "class Example;",
            "class ExampleManager;",
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that KMP source sets can have file with the same name`() {
        val source1 = SourceFile.new(
            "commonMain/Utils.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun projectName() = "Example Project"
        """.trimIndent()
        )

        val source2 = SourceFile.new(
            "linuxMain/Utils.kt", """
            package test
            
            import ru.aurora.kmp.qtbindings.QtExport
            
            @QtExport
            fun osName() = "Linux"
        """.trimIndent()
        )

        val compilation = run(listOf(source1, source2))

        compilation.assertThatFileHasContent(
            Path("test/Utils.hpp"), """
            namespace test {

            QString projectName();
            QString osName();

            } /* namespace test */
            """.trimIndent()
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }
}
