/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import com.squareup.kotlinpoet.ClassName
import com.squareup.kotlinpoet.FunSpec
import com.squareup.kotlinpoet.ParameterSpec
import com.squareup.kotlinpoet.ParameterizedTypeName.Companion.parameterizedBy
import com.squareup.kotlinpoet.STAR

internal fun generateListFunctionSpecs(): List<FunSpec> {
    return listOf(
        generateListGetSizeFunSpec(),
        generateListGetElementByIndexFunSpec(),
        generateMutableListCreateEmptyFunSpec(),
        generateMutableListAddClassFunSpec(),
        *generateMutableListAddSimpleElementFunSpecs().toTypedArray(),
        generateMutableListToListFunSpec(),
    )
}

private fun generateListGetSizeFunSpec(): FunSpec {
    val funName = listGetSizePredefinedMemberName.simpleName.removePredefinedSuffix()
    val builder = FunSpec.builder(funName)
    val listPtrParameter = ParameterSpec.builder("ptr", cOpaquePointerName.copy(nullable = true)).build()
    builder.addAnnotation(createOptInAnnotationSpec(experimentalForeignApiClassName))
    builder.addParameter(listPtrParameter)
    builder.returns(intClassName)
    builder.addCode("return %M(%N)", listGetSizePredefinedMemberName, listPtrParameter)
    return builder.build()
}

private fun generateListGetElementByIndexFunSpec(): FunSpec {
    val funName = listGetElementByIndexPredefinedMemberName.simpleName.removePredefinedSuffix()
    val builder = FunSpec.builder(funName)
    val nullableCOpaquePointer = cOpaquePointerName.copy(nullable = true)
    val listPtrParameter = ParameterSpec.builder("ptr", nullableCOpaquePointer).build()
    val elementIndexParameter = ParameterSpec.builder("index", intClassName).build()
    builder.addAnnotation(createOptInAnnotationSpec(experimentalForeignApiClassName))
    builder.addParameters(listOf(listPtrParameter, elementIndexParameter))
    builder.returns(nullableCOpaquePointer)
    builder.addCode(
        "return %M(%N, %N)", listGetElementByIndexPredefinedMemberName, listPtrParameter, elementIndexParameter
    )
    return builder.build()
}

private fun generateMutableListCreateEmptyFunSpec(): FunSpec {
    val funName = mutableListCreateEmptyPredefinedMemberName.simpleName.removePredefinedSuffix()
    val builder = FunSpec.builder(funName)
    builder.returns(mutableListClassName.parameterizedBy(anyClassName))
    builder.addCode("return %M()", mutableListCreateEmptyPredefinedMemberName)
    return builder.build()
}

private fun generateMutableListAddClassFunSpec(): FunSpec {
    val funName = mutableListAddClassPredefinedMemberName.simpleName.removePredefinedSuffix()
    val builder = FunSpec.builder(funName)
    val listParameter = ParameterSpec.builder("list", mutableListClassName.parameterizedBy(anyClassName)).build()
    val elementPtrParameter = ParameterSpec.builder("ptr", cOpaquePointerName).build()
    builder.addAnnotation(createOptInAnnotationSpec(experimentalForeignApiClassName))
    builder.addParameter(listParameter)
    builder.addParameter(elementPtrParameter)
    builder.returns(unitClassName)
    builder.addCode("return %M(%N, %N)", mutableListAddClassPredefinedMemberName, listParameter, elementPtrParameter)
    return builder.build()
}

private fun generateMutableListAddSimpleElementFunSpecs(): List<FunSpec> {
    return listOf(
        "Unit",
        "Boolean",
        "Char",
        "Byte",
        "Short",
        "Int",
        "Long",
        "UByte",
        "UShort",
        "UInt",
        "ULong",
        "Float",
        "Double",
        "String"
    ).map {
        val typeClassName = ClassName("kotlin", it)
        val funName = mutableListAddSimpleElementPredefinedMemberName.simpleName.replaceSimpleElementPredefinedSuffix(
            typeClassName.simpleName
        )
        val builder = FunSpec.builder(funName)
        val listParameter = ParameterSpec.builder("l", mutableListClassName.parameterizedBy(typeClassName)).build()
        val elementParameter = ParameterSpec.builder("el", typeClassName).build()
        builder.addParameter(listParameter)
        builder.addParameter(elementParameter)
        builder.returns(unitClassName)
        builder.addCode(
            "return %M(%N, %N)", mutableListAddSimpleElementPredefinedMemberName, listParameter, elementParameter
        )
        builder.build()
    }
}

private fun generateMutableListToListFunSpec(): FunSpec {
    val funName = mutableListToListPredefinedMemberName.simpleName.removePredefinedSuffix()
    val builder = FunSpec.builder(funName)
    val listParameter = ParameterSpec.builder("mutableList", mutableListClassName.parameterizedBy(STAR)).build()
    builder.addParameter(listParameter)
    builder.returns(listClassName.parameterizedBy(STAR))
    builder.addCode("return %M(%N)", mutableListToListPredefinedMemberName, listParameter)
    return builder.build()
}
private fun String.replaceSimpleElementPredefinedSuffix(replaceBy: String): String {
    return this.replace("SimpleElementPredefined", replaceBy)
}
