/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.*
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import org.jetbrains.kotlin.konan.target.KonanTarget
import org.junit.Before
import org.junit.Rule
import org.junit.rules.TemporaryFolder
import java.io.ByteArrayOutputStream
import java.io.File
import java.nio.file.Path

data class GeneratedFile(val path: Path, val content: String)

@OptIn(ExperimentalCompilerApi::class)
open class BaseCompilationTest {
    @Rule
    @JvmField
    val temporaryFolder: TemporaryFolder = TemporaryFolder().apply { }

    lateinit var konan: KotlinNativeCompiler
    lateinit var qtCompiler: QtCompiler

    companion object {
        const val EMPTY_TEST_FILE = "int main() { return 0; }" // Check that compilation works
    }

    @Before
    fun setUp() {
        konan = KotlinNativeCompiler(temporaryFolder.root)
        qtCompiler = QtCompiler(temporaryFolder.root)
    }

    fun run(
        kotlinSources: List<SourceFile>, generatedFiles: List<GeneratedFile> = listOf()
    ): KotlinCompilation {
        return KotlinCompilation().apply {
            languageVersion = "2.1"
            workingDir = temporaryFolder.root
            messageOutputStream = ByteArrayOutputStream()
            configureKsp(useKsp2 = true) {
                symbolProcessorProviders += QtBindingsSymbolProcessingProvider()
            }
            sources = kotlinSources
            inheritClassPath = true
            kspWithCompilation = true
            kspIncremental = true
            kspProcessorOptions = mutableMapOf("qtBindings.libName" to "shared")

            // Could not find a way for testing native compilation,
            // so let's just compare expected and actual generated code
            compile() //.assertSuccessfulCompilation()

            for (file in generatedFiles) {
                assertThatFileHasBeenGenerated(file.path, file.content)
            }
        }
    }

    fun run(file: SourceFile, vararg generatedFiles: GeneratedFile) =
        run(listOf(file), generatedFiles.asList())

    fun run(file: SourceFile, generatedFiles: List<GeneratedFile> = listOf()) =
        run(listOf(file), generatedFiles)

    fun creteFileAtWorkingDirectory(filename: String, content: String): File {
        val file = temporaryFolder.root.resolve(filename)
        file.bufferedWriter().use { it.write(content) }
        return file
    }

    fun compileAndRunQtTestCase(compilation: KotlinCompilation, qtTestCase: String = EMPTY_TEST_FILE): ProcessResult {
        val kotlinFiles = compilation.getKotlinSources()
        val (headers, sources) = compilation.getCppFiles()

        val konanResult =
            konan.compileStaticLibrary(KonanTarget.LINUX_X64, KotlinNativeCompiler.DEFAULT_LIB_NAME, kotlinFiles)
        konanResult.assertIsOk()

        val testFile = creteFileAtWorkingDirectory("test.cpp", qtTestCase)
        val qtCompilationResult =
            qtCompiler.compileExecutable(headers, sources + testFile, konanResult.cLibrary, konanResult.cHeader)
        qtCompilationResult.assertIsOk()

        return NativeProcessRunner(qtCompilationResult.executable).run()
    }
}
