/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp.model

internal sealed class QtType(protected open val shouldBeFq: Boolean) {

    abstract fun asString(): String
    open val ns: QtNamespace = QtNamespace.empty

    val qtFqName
        get() : String {
            val ns = ns.components.joinToString("::")
            return if (ns.isNotBlank()) "$ns::${asString()}" else asString()
        }

    val name get() = if (shouldBeFq) qtFqName else asString()

    open fun withFq() = this
    open fun withNotFq() = this

    val isVoid = this is Primitive.Void

    sealed class Primitive : QtType(false) {

        object Void : Primitive() {
            override fun asString() = "void"
        }

        object Bool : Primitive() {
            override fun asString() = "bool"
        }

        object Char : Primitive() {
            override fun asString() = "char"
        }

        object SignedChar : Primitive() {
            override fun asString() = "signed char"
        }

        object Short : Primitive() {
            override fun asString() = "short"
        }

        object Int : Primitive() {
            override fun asString() = "int"
        }

        object Long : Primitive() {
            override fun asString() = "long long"
        }

        object UnsignedChar : Primitive() {
            override fun asString() = "unsigned char"
        }

        object UnsignedShort : Primitive() {
            override fun asString() = "unsigned short"
        }

        object UnsignedInt : Primitive() {
            override fun asString() = "unsigned int"
        }

        object UnsignedLong : Primitive() {
            override fun asString() = "unsigned long long"
        }

        object Float : Primitive() {
            override fun asString() = "float"
        }

        object Double : Primitive() {
            override fun asString() = "double"
        }

        object QString : Primitive() {
            override fun asString() = "QString"
        }
    }

    sealed class Collection : QtType(false) {
        data class QList(
            val elementType: QtType,
            val isMutable: Boolean,
        ) : Collection() {
            override fun asString() = "QList<${elementType.name}>"
        }

        // TODO: implement Map
    }

    data class Class(
        val className: String, override val ns: QtNamespace, override val shouldBeFq: Boolean
    ) : QtType(shouldBeFq) {
        override fun asString() = className
        override fun withFq() = copy(shouldBeFq = true)
        override fun withNotFq() = copy(shouldBeFq = false)
    }
}
