/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp.gen

import ru.aurora.kmp.qtbindings.ksp.model.IncludeType
import ru.aurora.kmp.qtbindings.ksp.model.QtFile
import ru.aurora.kmp.qtbindings.ksp.model.QtInclude
import ru.aurora.kmp.qtbindings.ksp.model.QtNamespace

internal interface QtGenerator {
    fun generate(file: QtFile): String
}

internal abstract class QtBaseGenerator : QtGenerator {
    private val indent = " ".repeat(4)

    protected fun StringBuilder.withIndent(block: StringBuilder.() -> Unit) {
        val builder = StringBuilder().apply(block)
        val prefixLength = indent.length
        val newLineIndices = mutableListOf<Int>()

        for (i in 0 until builder.length) {
            // Do not add indent on last line and empty lines
            if (builder[i] == '\n' && (i + 1) != builder.length && builder[i + 1] != '\n') {
                newLineIndices.add(i)
            }
        }

        if (builder.isNotEmpty()) builder.insert(0, indent)

        newLineIndices.reversed().forEach { originalIndex ->
            val adjustedIndex = originalIndex + prefixLength
            builder.insert(adjustedIndex + 1, indent)
        }

        append(builder)
    }

    protected fun <T> StringBuilder.withNewLine(block: () -> T): T {
        return try {
            block()
        } finally {
            appendLine()
        }
    }

    protected fun <T> StringBuilder.appendWithSpace(data: T): StringBuilder = append("$data ")

    protected fun StringBuilder.generatedCodeWarning() {
        appendLine("// Generated code, DO NOT EDIT IT MANUALLY!")
    }

    protected fun StringBuilder.generateIncludes(includes: Set<QtInclude>) {
        if (includes.isEmpty()) return
        val includeList = includes.sortedBy { x -> x.name }
        val (localIncludes, left) = includeList.partition { it.type == IncludeType.Local }
        val (qIncludes, others) = left.partition { it.name.startsWith('Q') }

        val includeGroups = listOf(qIncludes, others, localIncludes)

        includeGroups.forEachIndexed { index, includeGroup ->
            includeGroup.forEach { appendLine("#include ${it.asString()}") }
            if (index != includeGroups.lastIndex && includeGroup.any()) appendLine()
        }
    }

    protected fun StringBuilder.namespace(qtNamespace: QtNamespace? = null, block: StringBuilder.() -> Unit) {
        if (qtNamespace != null) appendLine(qtNamespace.components.joinToString("\n") { "namespace $it {" })
        else appendLine("namespace {")

        appendLine()
        block()
        appendLine()

        if (qtNamespace != null) appendLine(
            qtNamespace.components.reversed().joinToString("\n") { "} /* namespace $it */" })
        else appendLine("} /* anonymous */")
    }
}
