/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.aurora.kmp.qtbindings.ksp

import ru.aurora.kmp.qtbindings.ksp.model.QtType

internal enum class ConversionType {
    QtToKotlin, KotlinToQt
}

internal sealed class TypeConverterDescriptor {

    abstract val name: String
    abstract val conversionType: ConversionType

    data class List(val elementType: QtType, val isMutable: Boolean, override val conversionType: ConversionType) :
        TypeConverterDescriptor() {

        override val name = converterName(elementType, isMutable, conversionType)
        val type = QtType.Collection.QList(elementType, isMutable)

        companion object {
            fun converterName(elementType: QtType, isMutable: Boolean, conversionType: ConversionType): String {
                return (listOf(listPrefix(isMutable, conversionType)) + elementType.words()).toCamelCase()
            }
        }
    }

    companion object {
        private fun listPrefix(isMutable: Boolean, conversionType: ConversionType): String {
            val prefix = if (conversionType == ConversionType.KotlinToQt) "to" else "from"
            return prefix + if (isMutable) "MutableQList" else "QList"
        }
    }
}

private fun QtType.words(): List<String> {
    return when (this) {
        is QtType.Collection.QList -> this.qtFqName.splitToTheWords()
        else -> this.qtFqName.splitToTheWords()
    }
}

private fun String.splitToTheWords(): List<String> {
    return this.split(" ", "<", ">", ",", "::").filter { it.isNotBlank() }
}
