/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp.internal

import org.gradle.api.Project
import org.gradle.api.attributes.Usage
import org.gradle.testfixtures.ProjectBuilder
import org.jetbrains.kotlin.gradle.plugin.KotlinPlatformType
import org.jetbrains.kotlin.gradle.plugin.mpp.KotlinNativeTarget
import org.jetbrains.kotlin.gradle.plugin.mpp.KotlinUsages
import org.jetbrains.kotlin.gradle.utils.NativeCompilerDownloader
import org.jetbrains.kotlin.konan.target.KonanTarget
import java.io.File

class NativeCompilationResult(
    val exitCode: Int,
    val stdout: String,
    val stderr: String,
    val cLibrary: File,
    val cHeader: File,
) {
    val isOk = exitCode == 0
    fun assertIsOk() = assert(isOk) { "Failed to compile Kotlin C Library ${cLibrary.absolutePath}: $stderr" }
}

class KotlinNativeCompiler(workingDir: File) {
    val project: Project = ProjectBuilder.builder().withProjectDir(workingDir).build()
    val compilerExecutable: String

    companion object {
        const val DEFAULT_LIB_NAME = "shared"
    }

    init {
        project.repositories.apply {
            mavenCentral() // For Kotlin/Native compiler
            mavenLocal() // For qtbindings-annotations, qtbindings-core
        }

        // Download Kotlin/Native compiler
        val downloader = NativeCompilerDownloader(project)
        downloader.downloadIfNeeded()

        compilerExecutable = downloader.compilerDirectory.resolve("bin/konanc").normalize().absolutePath
    }

    fun compileStaticLibrary(target: KonanTarget, libName: String, files: List<File>): NativeCompilationResult {
        check(target is KonanTarget.LINUX_X64 || target is KonanTarget.LINUX_ARM64)

        val klibs = mutableSetOf<String>()
        val projectDir = project.projectDir
        val sources = files.map { it.path }
        val outputDir = projectDir.resolve("kotlin-native-build").also { it.mkdirs() }
        val outputName = "lib$libName.a"
        val library = outputDir.resolve(outputName)
        val header = outputDir.resolve("lib${libName}_api.h")

        project.findKLibArtifactsForDependency("ru.auroraos.kmp", "qtbindings-core", "0.1.0", target).forEach {
            klibs.add(it.absolutePath)
        }

        project.findKLibArtifactsForDependency("ru.auroraos.kmp", "qtbindings-annotations", "0.1.0", target).forEach {
            klibs.add(it.absolutePath)
        }

        val command = mutableListOf(
            compilerExecutable,
            "-g",
            *sources.toTypedArray(),
            *klibs.flatMap { listOf("-l", it) }.toTypedArray(),
            "-output",
            library.path,
            "-produce",
            "static",
            "-target",
            target.name,
        )

        val process = ProcessBuilder(command).directory(projectDir).redirectOutput(ProcessBuilder.Redirect.PIPE)
            .redirectError(ProcessBuilder.Redirect.PIPE).start()

        val exitCode = process.waitFor()
        val output = process.inputStream.bufferedReader().readText()
        val errorOutput = process.errorStream.bufferedReader().readText()

        return NativeCompilationResult(exitCode, output, errorOutput, library, header)
    }
}

fun Project.findKLibArtifactsForDependency(
    group: String, name: String, version: String, target: KonanTarget
): Set<File> {
    val dependency = project.dependencies.create(
        mapOf(
            "group" to group, "name" to name, "version" to version
        )
    )

    val config = project.configurations.detachedConfiguration(dependency).apply {
        attributes.attribute(Usage.USAGE_ATTRIBUTE, project.objects.named(Usage::class.java, KotlinUsages.KOTLIN_API))
        attributes.attribute(KotlinPlatformType.attribute, KotlinPlatformType.native)
        attributes.attribute(KotlinNativeTarget.konanTargetAttribute, target.name)
    }

    return config.resolve()
}
