/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
@file:OptIn(ExperimentalCompilerApi::class)

package ru.auroraos.kmp.qtbindings.ksp.internal

import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi

import com.tschuchort.compiletesting.KotlinCompilation
import com.tschuchort.compiletesting.kspSourcesDir
import java.io.File
import java.nio.file.Path
import kotlin.io.path.Path
import kotlin.io.path.extension
import kotlin.math.max

val KotlinCompilation.sourceDir: File get() = workingDir.resolve("sources")
val KotlinCompilation.kspKotlinSourcesDir: File get() = kspSourcesDir.resolve("kotlin")
val KotlinCompilation.kspResourcesDir: File get() = kspSourcesDir.resolve("resources")

fun KotlinCompilation.getKotlinSources(): List<File> {
    val generatedKotlinSources = kspKotlinSourcesDir.walkTopDown().filter { it.name.endsWith(".kt") }
    val kotlinSources = sourceDir.walkTopDown().filter { it.name.endsWith(".kt") }
    return (kotlinSources + generatedKotlinSources).toList()
}

/**
 * Returns <Headers, Sources>.
 */
fun KotlinCompilation.getCppFiles(): Pair<List<File>, List<File>> {
    val generatedSources = kspResourcesDir.walkTopDown()
    val headers = generatedSources.filter { it.name.endsWith(".hpp") || it.name.endsWith(".h") }
    val sources = generatedSources.filter { it.name.endsWith(".cpp") || it.name.endsWith(".с") }
    return Pair(headers.toList(), sources.toList())
}

fun Path.toKspLocation(): Path {
    return when (this.extension) {
        "kt" -> {
            Path("kotlin", toString())
        }

        "java" -> {
            Path("java", toString())
        }

        else -> { // For .cpp, .hpp files
            Path("resources", toString())
        }
    }
}

fun KotlinCompilation.assertThatFileHasBeenGenerated(path: Path, content: String) {
    val path = path.toKspLocation()
    val file = kspSourcesDir.resolve(path.toString())
    file.assertThatExists()

    val actualLines = file.readLines()
    val expectedLines = content.lines()
    val lines = max(actualLines.size, expectedLines.size)

    for (i in 0..lines - 1) {
        val actualLine =
            actualLines.elementAtOrNull(i) ?: assert(false) { "Generated file has more lines than expected" }
        val expectedLine =
            expectedLines.elementAtOrNull(i) ?: assert(false) { "Generated file has less lines than expected" }
        assert(actualLine == expectedLine) {
            "The contents of the file differ on line $i: $path"
        }
    }
}

fun KotlinCompilation.assertThatFileHasContent(path: Path, content: List<String>) {
    val file = kspSourcesDir.resolve(path.toKspLocation().toString())
    file.assertThatExists()

    val fileLines = file.readLines()

    content.forEach { content ->
        val contentLines = content.lines()
        val sublistsOfSpecifiedSize = fileLines.windowed(contentLines.size)
        assert(sublistsOfSpecifiedSize.any { it == contentLines }) {
            "File $path does not contain:\n$content"
        }
    }
}

fun KotlinCompilation.assertThatFileHasContent(path: Path, vararg content: String) =
    assertThatFileHasContent(path, content.asList())

fun KotlinCompilation.assertThatKSPOutputHasContent(vararg lines: String) {
    val outputLines = messageOutputStream.toString().lines()
    lines.forEach { line ->
        assert(outputLines.any { it.contains(line) }) {
            "Compiler output does not contain: $line"
        }
    }
}