/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp

import ru.auroraos.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class NotSupported : BaseCompilationTest() {

    @Test
    fun `check that vararg is not supported for top-level function`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun varArgFunction(vararg str : String) = str.joinToString()
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent("Test.kt:6: Failed to resolve function parameter")
    }

    @Test
    fun `check that vararg is not supported for class method`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class Test {
                fun varArgFunction(vararg str : String) = str.joinToString()
            }
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/Test.hpp"), """
            class Test
            {
            public:
                Test();
                Test(Test &&other);
                Test(libshared_kref_test_Test ptr);
                ~Test();

                Test &operator=(Test &&other);

                libshared_kref_test_Test unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Test d_ptr;
            };
        """.trimIndent()
        )

        compilation.assertThatKSPOutputHasContent("Test.kt:7: Failed to resolve function parameter")
    }

    @Test
    fun `check that methods with unsupported type is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class Test {
                fun getSet() = setOf<Int>()
            }
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatFileHasContent(
            Path("test/Test.hpp"), """
            class Test
            {
            public:
                Test();
                Test(Test &&other);
                Test(libshared_kref_test_Test ptr);
                ~Test();
            
                Test &operator=(Test &&other);
            
                libshared_kref_test_Test unsafeKotlinPointer() const;
            private:
                libshared_kref_test_Test d_ptr;
            };
        """.trimIndent()
        )

        compilation.assertThatKSPOutputHasContent("Test.kt:7: Failed to resolve function return type")
    }

    @Test
    fun `check that top level functions with unsupported type is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun getSet() = setOf<Int>()
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent("Test.kt:6: Failed to resolve function return type")
    }

    @Test
    fun `check that expected function is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            expect fun test()
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent("Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can not be applied to the expected classes and functions")
    }

    @Test
    fun `check that expected class is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            expect class Test
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent("Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can not be applied to the expected classes and functions")
    }

    @Test
    fun `check non-public classes are not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            internal class Test1
            
            @QtExport
            private class Test2
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to public classes and functions",
            "Test.kt:9: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to public classes and functions",
        )
    }

    @Test
    fun `check non-public top-level functions are not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            internal fun test() = ""
            
            @QtExport
            private fun test2() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to public classes and functions",
            "Test.kt:9: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to public classes and functions",
        )
    }

    @Test
    fun `check generic classes are not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class Test1<T>
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can not be applied to the generic classes and functions",
        )
    }

    @Test
    fun `check generic functions are not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun <T> test2() = ""
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport can not be applied to the generic classes and functions",
        )
    }

    @Test
    fun `check that nested class is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            class OuterClass {
                @QtExport
                class Nested
            }
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:7: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to the top-level classes and functions",
        )
    }

    @Test
    fun `check that object is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            object OuterClass {
                val test: String = ""
            }
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent("Test.kt:6: Annotation ru.auroraos.kmp.qtbindings.QtExport is not supported for objects")
    }

    @Test
    fun `check that non top-level function is not exported`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test

            import ru.auroraos.kmp.qtbindings.QtExport
            
            class OuterClass {
                @QtExport
                fun nested() {}
            }
        """.trimIndent()
        )

        val compilation = run(source)
        compilation.assertThatKSPOutputHasContent(
            "Test.kt:7: Annotation ru.auroraos.kmp.qtbindings.QtExport can only be applied to the top-level classes and functions",
        )
    }
}
