/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp

import ru.auroraos.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class IncludeTests : BaseCompilationTest() {

    @Test
    fun `check that C library header is included in the header file`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun test() = ""
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.hpp"), "#include \"libshared_api.h\"")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that header file in included in source file`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class Test
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.cpp"), "#include \"Test.hpp\"")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }


    @Test
    fun `check that QString is included in the header file when it's used`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun str() = ""
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.hpp"), "#include <QString>")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that QList is included in the header file when it's used (List)`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun list() : List<Int> = emptyList()
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.hpp"), "#include <QList>")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that QList is included in the header file when it's used (MutableList)`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun mutableList() : MutableList<Int> = mutableListOf()
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.hpp"), "#include <QList>")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that CoroutineOperation is included in the source file when suspend function is used`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            suspend fun str() = ""
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.cpp"), "#include \"CoroutineOperation.hpp\"")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that QFuture is included in the header file when suspend function is used`() {
        val source = SourceFile.new(
            "Test.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            suspend fun str() = ""
        """.trimIndent()
        )
        val compilation = run(source)
        compilation.assertThatFileHasContent(Path("test/Test.hpp"), "#include <QFuture>")
        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that header for another generated class is included is included when the class is used`() {
        val source1 = SourceFile.new(
            "Test1.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            data class Data(val data: String)
        """.trimIndent()
        )

        val source2 = SourceFile.new(
            "Test2.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun getData() : Data = Data("TestString")
        """.trimIndent()
        )

        val compilation = run(listOf(source1, source2))
        compilation.assertThatFileHasContent(Path("test/Test2.hpp"), "#include \"Test1.hpp\"", "Data getData();")
        compilation.assertThatFileHasContent(
            Path("test/Test2.cpp"), """
        Data getData()
        {
            auto s = libshared_symbols();
            auto ns = s->kotlin.root.test;
            auto kotlinResult = ns.getData();
            auto qtResult = Data(kotlinResult);
            return qtResult;
        }
        """.trimIndent()
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }

    @Test
    fun `check that class from another package is imported with full qualification`() {
        val source1 = SourceFile.new(
            "PackageTest.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            data class Data(val data: String)
        """.trimIndent()
        )

        val source2 = SourceFile.new(
            "PackageExample.kt", """
            package example
            
            import test.Data    
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            fun returnData() : Data = Data("TestString")
        """.trimIndent()
        )

        val compilation = run(listOf(source1, source2))
        compilation.assertThatFileHasContent(Path("example/PackageExample.hpp"), "test::Data returnData();")
        compilation.assertThatFileHasContent(
            Path("example/PackageExample.cpp"), """
            test::Data returnData()
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.example;
                auto kotlinResult = ns.returnData();
                auto qtResult = test::Data(kotlinResult);
                return qtResult;
            }
            """.trimIndent()
        )

        compileAndRunQtTestCase(compilation).assertIsOk() // Check that code can be compiled
    }
}
