/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp

import ru.auroraos.kmp.qtbindings.ksp.internal.*
import com.tschuchort.compiletesting.SourceFile
import org.jetbrains.kotlin.compiler.plugin.ExperimentalCompilerApi
import kotlin.io.path.Path
import kotlin.test.Test

@OptIn(ExperimentalCompilerApi::class)
class ExtensionFunctionTests : BaseCompilationTest() {

    @Test
    fun `check that extension function for user class is generated`() {
        val kotlinSource1 = SourceFile.new(
            "ExtensibleClass.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class ExtensibleClass
        """.trimIndent()
        )

        val kotlinSource2 = SourceFile.new(
            "ExtensionFunction.kt", """
            package another
            
            import ru.auroraos.kmp.qtbindings.QtExport
            import test.ExtensibleClass
            
            @QtExport
            fun ExtensibleClass.name() = "Extensible class"
        """.trimIndent()
        )

        val qtTestCase = """
            #include "ExtensionFunction.hpp"
            #include "ExtensibleClass.hpp"
            
            using namespace test;
            using namespace another;
            
            int main() {
                if (name(ExtensibleClass()) != "Extensible class") {
                    return 1;
                }
               
                return 0;
            }
        """.trimIndent()

        val compilation = run(listOf(kotlinSource1, kotlinSource2))

        compilation.assertThatFileHasContent(
            Path("another/ExtensionFunction.hpp"), "QString name(const test::ExtensibleClass &thiz);"
        )
        compilation.assertThatFileHasContent(
            Path("another/ExtensionFunction.cpp"), """
            QString name(const test::ExtensibleClass &thiz)
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.another;
                auto kotlinThiz = thiz.unsafeKotlinPointer();
                auto kotlinResult = ns.name(kotlinThiz);
                auto qtResult = kotlinResult != nullptr ? QString::fromUtf8((char *) kotlinResult) : QString();
                if (kotlinResult != nullptr) {
                    s->DisposeString(kotlinResult);
                }
                return qtResult;
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }

    @Test
    fun `check that extension method for user class is generated`() {
        val kotlinSource1 = SourceFile.new(
            "ExtensibleClass.kt", """
            package test
            
            import ru.auroraos.kmp.qtbindings.QtExport
            
            @QtExport
            class ExtensibleClass
        """.trimIndent()
        )

        val kotlinSource2 = SourceFile.new(
            "ExtensionMethod.kt", """
            package another
            
            import ru.auroraos.kmp.qtbindings.QtExport
            import test.ExtensibleClass
            
            @QtExport
            class ClassWithExtensionFunction {
                fun ExtensibleClass.name() = "Extensible class"            
            }
        """.trimIndent()
        )

        val qtTestCase = """
            #include "ExtensionMethod.hpp"
            #include "ExtensibleClass.hpp"
            
            using namespace test;
            using namespace another;
            
            int main() {
                ClassWithExtensionFunction a = ClassWithExtensionFunction();

                if (a.name(ExtensibleClass()) != "Extensible class") {
                    return 1;
                }

                return 0;
            }
        """.trimIndent()

        val compilation = run(listOf(kotlinSource1, kotlinSource2))

        compilation.assertThatFileHasContent(
            Path("another/ExtensionMethod.hpp"), """
            class ClassWithExtensionFunction
            {
            public:
                ClassWithExtensionFunction();
                ClassWithExtensionFunction(ClassWithExtensionFunction &&other);
                ClassWithExtensionFunction(libshared_kref_another_ClassWithExtensionFunction ptr);
                ~ClassWithExtensionFunction();

                ClassWithExtensionFunction &operator=(ClassWithExtensionFunction &&other);

                QString name(const test::ExtensibleClass &thiz) const;

                libshared_kref_another_ClassWithExtensionFunction unsafeKotlinPointer() const;
            private:
                libshared_kref_another_ClassWithExtensionFunction d_ptr;
            };
            """.trimIndent()
        )
        compilation.assertThatFileHasContent(
            Path("another/ExtensionMethod.cpp"), """
            QString ClassWithExtensionFunction::name(const test::ExtensibleClass &thiz) const
            {
                auto s = libshared_symbols();
                auto ns = s->kotlin.root.another.ClassWithExtensionFunction;
                auto kotlinThiz = thiz.unsafeKotlinPointer();
                auto kotlinResult = ns.name(d_ptr, kotlinThiz);
                auto qtResult = kotlinResult != nullptr ? QString::fromUtf8((char *) kotlinResult) : QString();
                if (kotlinResult != nullptr) {
                    s->DisposeString(kotlinResult);
                }
                return qtResult;
            }
            """.trimIndent()
        )

        val qtTestCaseResult = compileAndRunQtTestCase(compilation, qtTestCase)
        qtTestCaseResult.assertIsOk()
    }
}
