/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp.gen

import ru.auroraos.kmp.qtbindings.ksp.ConversionType
import ru.auroraos.kmp.qtbindings.ksp.TypeConverterDescriptor
import ru.auroraos.kmp.qtbindings.ksp.capitalizeFirstChar
import ru.auroraos.kmp.qtbindings.ksp.model.QtType

internal class TypeConverter(private val cApi: CLibraryAPI) {

    fun kotlinToQt(varName: String, type: QtType): String {
        return when (type) {
            is QtType.Primitive.QString -> "$varName != nullptr ? QString::fromUtf8((char *) $varName) : QString()"
            is QtType.Class -> "${type.name}($varName)"
            is QtType.Collection.QList -> "${
                TypeConverterDescriptor.List.converterName(
                    type.elementType, type.isMutable, ConversionType.KotlinToQt
                )
            }($varName)"

            else -> varName
        }
    }

    fun opaquePointerToQt(varName: String, type: QtType): String {
        return when (type) {
            is QtType.Primitive.Bool, is QtType.Primitive.Long, is QtType.Primitive.UnsignedLong -> "(${type.name}) $varName"

            is QtType.Primitive.Char, is QtType.Primitive.SignedChar, is QtType.Primitive.Short, is QtType.Primitive.Int, is QtType.Primitive.UnsignedChar, is QtType.Primitive.UnsignedShort, is QtType.Primitive.UnsignedInt, is QtType.Primitive.Float, is QtType.Primitive.Double -> "(${type.name}) ((${QtType.Primitive.Long.name}) $varName)"

            is QtType.Primitive.QString -> "$varName != nullptr ? QString::fromUtf8((char *) $varName) : QString()"
            is QtType.Class -> "${type.name}(${cApi.kotlin.className(type.name)}{ $varName })"
            is QtType.Collection.QList -> {
                val listConverterName = TypeConverterDescriptor.List.converterName(
                    type.elementType, type.isMutable, ConversionType.KotlinToQt
                )
                val kotlinList = cApi.getKotlinListType(type.isMutable)
                "$listConverterName($kotlinList{ $varName })"
            }

            is QtType.Primitive.Void -> throw IllegalStateException("The type should be processed earlier: ${type.name}")
        }
    }

    fun qtToKotlin(varName: String, type: QtType): Pair<List<String>, String> {
        return when (type) {
            is QtType.Primitive.QString ->  {
                val cVarName = varName.capitalizeFirstChar()
                Pair(
                    listOf("auto byteArray$cVarName = $varName.toUtf8()"), "byteArray$cVarName.data()"
                )
            }

            is QtType.Class -> Pair(emptyList(), "$varName.unsafeKotlinPointer()")
            is QtType.Collection.QList -> {
                val convertLine = "${
                    TypeConverterDescriptor.List.converterName(
                        type.elementType, type.isMutable, ConversionType.QtToKotlin
                    )
                }($varName)"
                Pair(emptyList(), convertLine)
            }


            else -> Pair(emptyList(), varName)
        }
    }
}
