/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp.gen

import ru.auroraos.kmp.qtbindings.ksp.ConversionType
import ru.auroraos.kmp.qtbindings.ksp.TypeConverterDescriptor
import ru.auroraos.kmp.qtbindings.ksp.model.*

internal class QtSourceGenerator(val cApi: CLibraryAPI) : QtBaseGenerator() {

    private val typeConverter = TypeConverter(cApi)
    private val varScope = VariableScope()

    override fun generate(file: QtFile): String {
        return buildString {
            val generators = mutableListOf<StringBuilder.() -> Unit>()

            if (file.functions.any()) generators.add {
                file.functions.forEachIndexed { index, function ->
                    generateTopLevelFunction(function)
                    if (index != file.functions.lastIndex) appendLine()
                }
            }

            if (file.classes.any()) generators.add {
                file.classes.forEachIndexed { index, cppClass ->
                    generateClass(cppClass)
                    if (index != file.classes.lastIndex) appendLine()
                }
            }

            generatedCodeWarning()
            appendLine()

            if (file.sourceInclude.any()) {
                generateIncludes(file.sourceInclude)
                appendLine()
            }

            if (file.typeConverters.any()) {
                generateTypeConverters(file.typeConverters)
                appendLine()
            }

            namespace(file.namespace) {
                // Generate realization of top-level functions and classes
                generators.forEachIndexed { index, generator ->
                    generator()
                    if (index != generators.lastIndex) appendLine()
                }
            }
        }
    }

    private fun StringBuilder.generateTypeConverters(converters: Set<TypeConverterDescriptor>) {
        namespace {
            converters.forEachIndexed { index, converter ->
                when (converter) {
                    is TypeConverterDescriptor.List -> {
                        when (converter.conversionType) {
                            ConversionType.KotlinToQt -> generateKListConverter(converter)
                            ConversionType.QtToKotlin -> generateQListConverter(converter)
                        }
                    }
                }
                if (converters.size - 1 != index) appendLine()
            }
        }
    }

    private fun StringBuilder.generateTopLevelFunction(f: QtFunction) = generateFunction(f, f.name)

    private fun StringBuilder.generateFunction(
        func: QtFunction, fqFuncName: String, extraArgs: List<String> = emptyList()
    ) {
        val parameters = func.parameters.joinToString { it.toFunctionParameter() }
        val cFunctionName = func.cName
        val returnType = func.returnType.toReturnType(func.isAsync)

        function("$returnType ${fqFuncName}($parameters)${if (func.isConst) " const" else ""}") {
            func.parameters.forEach { varScope.forceAdd(it.name) }

            val symbolsVar = generateSymbolsVariable()
            val nsVar = generateFunctionNSVariable(symbolsVar, func)

            // Convert Qt arguments to Kotlin arguments
            val kotlinParameters = func.parameters.map { convertQtVariableToKotlinVariable(it.name, it.type) }
            val kotlinFinalParameters = (extraArgs + kotlinParameters.map { it.first }).joinToString()
            val functionCall = "${nsVar}.${cFunctionName}($kotlinFinalParameters)"

            // FunctionCall cases (Do not forget to generate free before return)
            // 0: Void, Sync -> Just 'return'
            // 1: Void, Async -> Return 'coroutine<QFuture<Void>>(functionCall)'
            // 2. Non-void, Sync -> Just 'return functionCall'
            // 3. Non-void, Async -> Return 'coroutine<QFuture<Void>>(functionCall, transformer)'

            val freeParameters = { kotlinParameters.forEach { generateFree(symbolsVar, it.first, it.second, false) } }

            if (!func.isAsync) {
                if (func.returnType.isVoid) {
                    appendLine("$functionCall;")
                    freeParameters()
                } else {
                    val functionCallResult = saveKotlinExpressionAtVariable("result", functionCall)
                    val freeKotlinResult = { generateFree(symbolsVar, functionCallResult, func.returnType, true) }
                    val (qtResult, _) = convertKotlinVariableToQtVariable(functionCallResult, func.returnType)

                    freeParameters()
                    freeKotlinResult()
                    returnWith(qtResult)
                }
            } else {
                val coroutineLauncher = saveExpressionAtVariable("coroutineLauncher", functionCall)
                val qFuture = launchCoroutine(func.returnType, coroutineLauncher)

                freeParameters()
                returnWith(qFuture)
            }
        }
    }

    private fun StringBuilder.launchCoroutine(resultType: QtType, coroutineLauncherVar: String): String {
        val qtBindingsNS = "Aurora::Kmp::QtBindings"
        val kotlinCoroutineLauncher = "KotlinCoroutineLauncher"
        val fqCoroutine = "$qtBindingsNS::coroutine"
        val typedCoroutineType = "$fqCoroutine<${resultType.name}>"
        val convertedCoroutineLauncherVar = "($kotlinCoroutineLauncher *) $coroutineLauncherVar"

        val coroutineCall = when (resultType) {
            is QtType.Primitive.Void -> "$typedCoroutineType($convertedCoroutineLauncherVar)"
            else -> buildString {
                varScope.scope {
                    val coroutineResultVar = "coroutineResult"
                    overrideAdd(coroutineResultVar)

                    appendLine("$typedCoroutineType($convertedCoroutineLauncherVar, [=](void *$coroutineResultVar) {")
                    withIndent {
                        returnWith(typeConverter.opaquePointerToQt(coroutineResultVar, resultType))
                    }
                    append("})")
                }
            }
        }

        return saveQtExpressionAtVariable("futureResult", coroutineCall)
    }

    private fun StringBuilder.convertQtVariableToKotlinVariable(name: String, type: QtType): Pair<String, QtType> {
        val (lines, kotlinValue) = typeConverter.qtToKotlin(name, type)
        lines.forEach { appendLine("$it;") }

        if (kotlinValue == name) return Pair(kotlinValue, type)
        return Pair(saveKotlinExpressionAtVariable(name, kotlinValue), type)
    }

    private fun StringBuilder.convertKotlinVariableToQtVariable(name: String, type: QtType): Pair<String, QtType> {
        val kotlinValue = typeConverter.kotlinToQt(name, type)
        if (kotlinValue == name) return Pair(kotlinValue, type)
        return Pair(saveQtExpressionAtVariable(name, kotlinValue), type)
    }

    private fun StringBuilder.generateClass(qtClass: QtClass) {
        val generators = mutableListOf<() -> Unit>()
        val (constructors, methods) = qtClass.methods.partition { it.isConstructor }

        generators.add {
            constructors.forEach { constructor ->
                generateConstructor(constructor)
                appendLine()
            }
            if (qtClass.hasCopyMethod) {
                generateCopyConstructor(qtClass)
                appendLine()
            }
            generateMoveConstructor(qtClass)
            appendLine()
            generateKotlinWrapperConstructor(qtClass)
            appendLine()
            generateDestructor(qtClass)
        }

        generators.add {
            if (qtClass.hasCopyMethod) {
                generateCopyOperator(qtClass)
                appendLine()
            }
            generateMoveOperator(qtClass)
        }

        if (methods.any()) generators.add {
            methods.forEachIndexed { index, method ->
                generateMethod(method)
                if (index != methods.lastIndex) appendLine()
            }
        }

        generators.add { generateUnsafeKotlinPointerMethod(qtClass) }

        // Generate constructors and methods
        generators.forEachIndexed { index, generator ->
            generator()
            if (index < generators.lastIndex) {
                appendLine()
            }
        }
    }

    private fun StringBuilder.generateConstructor(constructor: QtMethod) {
        val parameters = constructor.parameters.joinToString { it.toFunctionParameter() }
        val name = constructor.name

        function("${name}::${name}($parameters)") {
            constructor.parameters.forEach { varScope.forceAdd(it.name) }
            val symbolsVar = generateSymbolsVariable()
            val nsVar = generateFunctionNSVariable(symbolsVar, constructor)

            val convertedArgs = constructor.parameters.map { convertQtVariableToKotlinVariable(it.name, it.type) }
            val free = { convertedArgs.forEach { generateFree(symbolsVar, it.first, it.second, false) } }

            appendLine("d_ptr = $nsVar.$name(${convertedArgs.joinToString { it.first }});")
            free()
        }
    }

    private fun StringBuilder.generateCopyConstructor(qtClass: QtClass) {
        val name = qtClass.name
        function("$name::$name(const $name &other)") {
            val symbolsVar = generateSymbolsVariable()
            val nsVar = generateClassNSVariable(symbolsVar, qtClass)
            appendLine("d_ptr = $nsVar.${name}CopyNoArgs(other.d_ptr);") // TODO: Remove hardcoded CopyNoArgs method name
        }
    }

    private fun StringBuilder.generateMoveConstructor(qtClass: QtClass) {
        val name = qtClass.name
        function("$name::$name($name &&other)") {
            appendLine("d_ptr.pinned = other.d_ptr.pinned;")
            appendLine("other.d_ptr.pinned = nullptr;")
        }
    }

    private fun StringBuilder.generateKotlinWrapperConstructor(qtClass: QtClass) {
        val name = qtClass.name
        appendLine("$name::$name(${cApi.kotlin.className(qtClass.fqName)} ptr)")
        withIndent {
            appendLine(": d_ptr(ptr)")
        }
        appendLine("{}")
    }

    private fun StringBuilder.generateCopyOperator(qtClass: QtClass) {
        val name = qtClass.name
        function("$name &$name::operator=(const $name &other)") {
            `if`("this != &other") {
                val symbolsVar = generateSymbolsVariable()
                val ns = generateClassNSVariable(symbolsVar, qtClass)
                appendLine("this->d_ptr = $ns.${name}CopyNoArgs(other.d_ptr);") // TODO: Remove hardcoded CopyNoArgs method name
                returnWith("*this")
            }

            returnWith("*this")
        }
    }

    private fun StringBuilder.generateMoveOperator(qtClass: QtClass) {
        val name = qtClass.name
        function("$name &$name::operator=($name &&other)") {
            `if`("this != &other") {
                appendLine("this->d_ptr = other.d_ptr;")
                appendLine("other.d_ptr.pinned = nullptr;")
                returnWith("*this")
            }
            returnWith("*this")
        }
    }

    private fun StringBuilder.generateMethod(m: QtMethod) =
        generateFunction(m, "${m.parentClassName}::${m.name}", listOf("d_ptr"))

    private fun StringBuilder.generateUnsafeKotlinPointerMethod(qtClass: QtClass) {
        val kotlinPtr = cApi.kotlin.className(qtClass.fqName)
        function("$kotlinPtr ${qtClass.name}::unsafeKotlinPointer() const") {
            returnWith("d_ptr")
        }
    }

    private fun StringBuilder.generateDestructor(qtClass: QtClass) {
        val name = qtClass.name
        function("$name::~$name()") {
            val s = generateSymbolsVariable()
            appendLine("$s->DisposeStablePointer(d_ptr.pinned);")
        }
    }

    private fun StringBuilder.saveQtExpressionAtVariable(varName: String, expr: String): String {
        val generatedVar = varScope.generateQt(varName)
        return saveExpressionAtVariableUnsafe(generatedVar, expr)
    }

    private fun StringBuilder.saveKotlinExpressionAtVariable(varName: String, expr: String): String {
        val generatedVar = varScope.generateKotlin(varName)
        return saveExpressionAtVariableUnsafe(generatedVar, expr)
    }

    private fun StringBuilder.saveExpressionAtVariable(varName: String, expr: String): String {
        val generatedVar = varScope.generate(varName)
        return saveExpressionAtVariableUnsafe(generatedVar, expr)
    }

    private fun StringBuilder.saveExpressionAtVariableUnsafe(varName: String, expr: String): String {
        appendLine("auto $varName = $expr;")
        return varName
    }

    private fun StringBuilder.generateSymbolsVariable(): String {
        return saveExpressionAtVariable("s", "${cApi.kotlin.symbolsMethodName}()")
    }

    private fun StringBuilder.generateClassNSVariable(symbolVar: String, c: QtClass): String {
        return saveExpressionAtVariable("ns", cApi.namespace(symbolVar, c.ns.components))
    }

    private fun StringBuilder.generateFunctionNSVariable(symbolVar: String, func: QtFunction): String {
        val ns = when (func) {
            is QtMethod -> if (func.isAsync) func.ns.components else func.ns.components + func.parentClassName
            else -> func.ns.components
        }

        return saveExpressionAtVariable("ns", cApi.namespace(symbolVar, ns))
    }

    private fun StringBuilder.generateKListConverter(listConverter: TypeConverterDescriptor.List) {
        val elementType = listConverter.elementType
        val paramType = cApi.getKotlinListType(listConverter)
        val listType = "QList<${elementType.name}>"
        val name = listConverter.name

        function("$listType $name($paramType kotlinList)") {
            val kListVar = varScope.generate("kotlinList")
            val qtResultVar = varScope.generate("qtResult")
            appendLine("$listType $qtResultVar;")
            val symbolVar = generateSymbolsVariable()
            val nsVar = saveExpressionAtVariable("ns", cApi.generated.namespace(symbolVar))

            `for`("int i = 0; i < $nsVar.${cApi.generated.list.getSizeFun}($kListVar.pinned); i++") {
                appendLine("auto ptr = $nsVar.${cApi.generated.list.getElementByIndexFun}($kListVar.pinned, i);")
                val qtElement = typeConverter.opaquePointerToQt("ptr", elementType)
                appendLine("qtResult.append($qtElement);")
                generateFreeVoidPointer(symbolVar, "ptr", elementType, true)
            }

            returnWith(qtResultVar)
        }
    }

    private fun StringBuilder.generateQListConverter(listConverter: TypeConverterDescriptor.List) {
        val name = listConverter.name
        val paramType = "QList<${listConverter.elementType.name}>"
        val returnType = cApi.getKotlinListType(listConverter)
        val elementType = listConverter.elementType
        val addFunc = cApi.getAddFunction(listConverter)

        function("$returnType $name(const $paramType &qtList)") {
            val qtListVar = varScope.generate("qtList")
            val s = generateSymbolsVariable()
            val ns = saveExpressionAtVariable("ns", cApi.generated.namespace(s))

            val kotlinResult =
                saveKotlinExpressionAtVariable("result", "$ns.${cApi.generated.mutableList.createEmpty}()")
            val qtElementVar = "qtElement"
            foreach(true, qtElementVar, qtListVar) {

                val (kotlinElement) = convertQtVariableToKotlinVariable(qtElementVar, elementType)
                if (elementType is QtType.Class || elementType is QtType.Collection.QList) {
                    appendLine("$ns.$addFunc($kotlinResult, $kotlinElement.pinned);")
                } else {
                    appendLine("$ns.$addFunc($kotlinResult, $kotlinElement);")
                }

                generateFree(s, kotlinElement, elementType, false)
            }

            if (listConverter.isMutable) {
                returnWith(kotlinResult)
            } else {
                val finalResult = saveKotlinExpressionAtVariable(
                    "result", "$ns.${cApi.generated.mutableList.toList}($kotlinResult)"
                )
                generateFree(s, kotlinResult, listConverter.type, false)
                returnWith(finalResult)
            }
        }
    }

    private fun StringBuilder.generateFree(symbolVar: String, varName: String, type: QtType, freeStr: Boolean) {
        when (type) {
            is QtType.Primitive.QString -> {
                if (freeStr) {
                    `if`("$varName != nullptr") {
                        appendLine("${symbolVar}->DisposeString($varName);")
                    }
                }
            }

            is QtType.Collection.QList -> {
                `if`("$varName.pinned != nullptr") {
                    appendLine("${symbolVar}->DisposeStablePointer($varName.pinned);")
                }
            }

            else -> Unit
        }
    }

    private fun StringBuilder.generateFreeVoidPointer(
        symbolVar: String, varName: String, type: QtType, freeStr: Boolean
    ) {
        when (type) {
            is QtType.Primitive.QString -> {
                if (freeStr) {
                    `if`("$varName != nullptr") {
                        appendLine("${symbolVar}->DisposeString((char *) $varName);")
                    }
                }
            }

            is QtType.Collection.QList -> {
                `if`("$varName != nullptr") {
                    appendLine("${symbolVar}->DisposeStablePointer($varName);")
                }
            }

            else -> Unit
        }
    }

    private fun StringBuilder.returnWith(str: String) {
        appendLine("return $str;")
    }

    private fun StringBuilder.function(definition: String, body: StringBuilder.() -> Unit) {
        varScope.scope {
            appendLine(definition)
            appendLine("{")
            withIndent {
                body()
            }
            appendLine("}")
        }
    }

    private fun StringBuilder.foreach(
        isConst: Boolean, elName: String, iterable: String, body: StringBuilder.() -> Unit
    ) {
        val const = if (isConst) "const " else ""
        appendLine("for (${const}auto &$elName : $iterable) {")
        withIndent {
            body()
        }
        appendLine("}")
    }

    private fun StringBuilder.`for`(forBlock: String, body: StringBuilder.() -> Unit) {
        appendLine("for ($forBlock) {")
        withIndent {
            body()
        }
        appendLine("}")
    }

    private fun StringBuilder.`if`(cond: String, body: StringBuilder.() -> Unit) {
        appendLine("if ($cond) {")
        withIndent {
            body()
        }
        appendLine("}")
    }
}

private fun CLibraryAPI.getAddFunction(listConverter: TypeConverterDescriptor): String {
    return when (listConverter) {
        is TypeConverterDescriptor.List -> when (listConverter.elementType) {
            is QtType.Primitive.Void -> generated.mutableList.addClass
            is QtType.Primitive.Bool -> generated.mutableList.addBool
            is QtType.Primitive.Char -> generated.mutableList.addChar
            is QtType.Primitive.SignedChar -> generated.mutableList.addByte
            is QtType.Primitive.Short -> generated.mutableList.addShort
            is QtType.Primitive.Int -> generated.mutableList.addInt
            is QtType.Primitive.Long -> generated.mutableList.addLong
            is QtType.Primitive.UnsignedChar -> generated.mutableList.addUByte
            is QtType.Primitive.UnsignedShort -> generated.mutableList.addUShort
            is QtType.Primitive.UnsignedInt -> generated.mutableList.addUInt
            is QtType.Primitive.UnsignedLong -> generated.mutableList.addULong
            is QtType.Primitive.Float -> generated.mutableList.addFloat
            is QtType.Primitive.Double -> generated.mutableList.addDouble
            is QtType.Primitive.QString -> generated.mutableList.addString
            is QtType.Collection.QList, is QtType.Class -> generated.mutableList.addClass
        }
    }
}
