/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp.export

import com.google.devtools.ksp.symbol.KSType
import com.google.devtools.ksp.symbol.KSTypeReference

/**
 * Describes kotlin types that supported to be exported to Qt.
 */
internal sealed class ExportedType {

    object Unit : ExportedType()
    object Bool : ExportedType()
    object Char : ExportedType()
    object Byte : ExportedType()
    object Short : ExportedType()
    object Int : ExportedType()
    object Long : ExportedType()
    object UByte : ExportedType()
    object UShort : ExportedType()
    object UInt : ExportedType()
    object ULong : ExportedType()
    object Float : ExportedType()
    object Double : ExportedType()
    object String : ExportedType()

    data class List(
        val elementType: ExportedType, val isMutable: Boolean
    ) : ExportedType()

    data class Class(
        val classDeclaration: ExportedClass,
    ) : ExportedType() {
        val fqName = classDeclaration.name
    }
}

internal fun KSTypeReference.toExportedType(userDefinedClasses: List<ExportedClass>): ExportedType? {

    val resolvedType = this.resolve()

    if (resolvedType.isError) {
        return null
    }

    val fqName = resolvedType.qualifiedName()

    return when (fqName) {
        "kotlin.Unit" -> ExportedType.Unit
        "kotlin.Boolean" -> ExportedType.Bool
        "kotlin.Char" -> ExportedType.Char
        "kotlin.Byte" -> ExportedType.Byte
        "kotlin.Short" -> ExportedType.Short
        "kotlin.Int" -> ExportedType.Int
        "kotlin.Long" -> ExportedType.Long
        "kotlin.UByte" -> ExportedType.UByte
        "kotlin.UShort" -> ExportedType.UShort
        "kotlin.UInt" -> ExportedType.UInt
        "kotlin.ULong" -> ExportedType.ULong
        "kotlin.Float" -> ExportedType.Float
        "kotlin.Double" -> ExportedType.Double
        "kotlin.String" -> ExportedType.String
        "kotlin.collections.List", "kotlin.collections.MutableList" -> {
            val elementType =
                resolvedType.arguments.firstOrNull()?.type?.toExportedType(userDefinedClasses) ?: return null
            val isMutable = fqName.endsWith("MutableList")
            ExportedType.List(elementType, isMutable)
        }

        else -> {
            val correspondingClass = userDefinedClasses.find { it.fqName == fqName }

            if (correspondingClass != null) {
                ExportedType.Class(correspondingClass)
            } else {
                null
            }
        }
    }
}

internal fun KSType.qualifiedName(): String {
    return declaration.qualifiedName?.asString() ?: declaration.simpleName.asString()
}
