/**
 * SPDX-FileCopyrightText: Copyright 2025 Open Mobile Platform LLC <community@omp.ru>
 * SPDX-License-Identifier: BSD-3-Clause
 */
package ru.auroraos.kmp.qtbindings.ksp.export

import com.google.devtools.ksp.symbol.KSFile
import ru.auroraos.kmp.qtbindings.ksp.removeKotlinExtension

/**
 * Describes data that needs to be exported to Qt.
 * Helps to managed user defined classes to resolve types.
 */
internal class ExportSpec {
    private val userClassesMap: MutableMap<String, ExportedClass> = mutableMapOf()
    private val userFilesMap: MutableMap<String, ExportedFile> = mutableMapOf()

    val userFiles: List<ExportedFile>
        get() = userFilesMap.values.toList()

    val userClasses: List<ExportedClass>
        get() = userClassesMap.values.toList()

    // KSP code generator creates files by the following rule: ksp/resources/package/name/filename.[hpp,cpp]
    // But KMP can have files with the same name and package in different sources sets:
    //      * commonMain/package/name/file.kt
    //      * linuxMain/package/name/file.kt
    //
    // For this reason we should use mapping from fully qualified fileName (package.name.filename)
    // to the exported file to avoid creating the same file twice with different content.
    fun getFileOrCreate(fqName: String, file: () -> ExportedFile) = userFilesMap.getOrPut(fqName, file)
    fun getFile(fqName: String) = userFilesMap[fqName]

    fun getClassOrCreate(fqName: String, userClass: () -> ExportedClass): ExportedClass {
        return userClassesMap.getOrPut(fqName, userClass)
    }

    fun isUserClass(fqName: String): Boolean {
        return userClassesMap.contains(fqName)
    }
}

/**
 * Fully qualified name of the file is equal to packageName + filename without extension
 */
internal val KSFile.fqName get() = "${packageName.asString()}.${fileName.removeKotlinExtension()}"
